/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#include <sys/param.h>
#include <sys/device.h>
#include <sys/systm.h>

#include <uvm/uvm.h>

#include <mips/locore.h>

#include <machine/autoconf.h>
#include <machine/vcache.h>

static int	cpu_match(device_t, cfdata_t, void *);
static void	cpu_attach(device_t, device_t, void *);

struct cpu_softc {
	device_t sc_dev;
	struct cpu_info *sc_info;
};

CFATTACH_DECL_NEW(cpu, sizeof(struct cpu_softc),
    cpu_match, cpu_attach, NULL, NULL);

static int
cpu_match(device_t parent, cfdata_t match, void *aux)
{
	return 1;
}

static void
cpu_attach(device_t parent, device_t self, void *aux)
{
	uint32_t versrev;
	struct cpu_info *ci = &cpu_info_store;
	int sets, ways, line, colors, tcolors;

	aprint_normal(": TSAR mips id 0x%x", cpu_id);
	versrev = tsar_vcache_read(VC_RELEASE);
	aprint_normal(", L1 cache version %d revision %d\n",
	    (versrev >> 16) & 0xffff, versrev & 0xffff);
	versrev = tsar_vcache_read(VC_PARAMS);
	line = ci->ci_l1_cls = 1 << VC_PARAMS_NBL(versrev);
	sets = ci->ci_l1_ics = 1 << VC_PARAMS_SCI(versrev);
	ways = ci->ci_l1_icw = 1 << VC_PARAMS_WCI(versrev);
	aprint_normal_dev(self, "I-cache %dK %d-way", line * sets * ways / 1024, ways);
	sets = ci->ci_l1_its = 1 << VC_PARAMS_STI(versrev);
	ways = ci->ci_l1_itw = 1 << VC_PARAMS_WTI(versrev);
	aprint_normal(", I-tlb %d entries %d-way\n", sets * ways, ways);
	sets = ci->ci_l1_dcs = 1 << VC_PARAMS_SCD(versrev);
	ways = ci->ci_l1_dcw = 1 << VC_PARAMS_WCD(versrev);
	aprint_normal_dev(self, "D-cache %dK %d-way", line * sets * ways / 1024, ways);
	sets = ci->ci_l1_dts = 1 << VC_PARAMS_STD(versrev);
	ways = ci->ci_l1_dtw = 1 << VC_PARAMS_WTD(versrev);
	aprint_normal(", D-tlb %d entries %d-way\n", sets * ways, ways);

	/* compute number of cache colors */
	colors = 2;
	tcolors = ci->ci_l1_dts * ci->ci_l1_cls / PAGE_SIZE;
	if (tcolors > colors)
		colors = tcolors;
	tcolors = ci->ci_l1_its * ci->ci_l1_cls / PAGE_SIZE;
	if (tcolors > colors)
		colors = tcolors;
	if (colors & (colors - 1)) {
		aprint_error_dev(self,
		    "number of colors %d not a power of 2\n", colors);
	} else {
		if (colors <= uvmexp.ncolors)
			return;
		aprint_debug_dev(self, "%d page colors\n", colors);
		uvm_page_recolor(colors);
	}
}
