/* $NetBSD: $ */
/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: cpu.c,v 1.24 2005/12/11 12:18:39 christos Exp $");

#define __INTR_PRIVATE
#include <sys/malloc.h>
#include <sys/types.h>
#include <machine/proc.h>
#include <sys/cpu.h>
#include <machine/intr.h>
#include <machine/xicu.h>

/*
 * map IPL to mask for cp0:status. bit 1 means interrupt is disabled
 * line 6 is unused and always masked.
 */

#define MIPS_SPL_NONE		(MIPS_INT_MASK_5)
#define MIPS_SPL_SOFTCLOCK \
    (MIPS_SPL_NONE | MIPS_SOFT_INT_MASK_0 | MIPS_INT_MASK_4)
#define MIPS_SPL_SOFTBIO (MIPS_SPL_SOFTCLOCK)
#define MIPS_SPL_SOFTNET \
    (MIPS_SPL_SOFTCLOCK | MIPS_SOFT_INT_MASK_1 | MIPS_INT_MASK_3)
#define MIPS_SPL_SOFTSERIAL (MIPS_SPL_SOFTNET)
#define MIPS_SPL_VM		(MIPS_SPL_SOFTSERIAL | MIPS_INT_MASK_2)
#define	MIPS_SPL_SCHED		(MIPS_SPL_VM | MIPS_INT_MASK_1)
#define MIPS_SPL_DDB		(MIPS_SPL_SCHED | MIPS_INT_MASK_0)
#define MIPS_SPL_HIGH		(MIPS_INT_MASK)

static const struct ipl_sr_map tsarmips_ipl_sr_map = {
    .sr_bits = {
	[IPL_NONE] = MIPS_SPL_NONE,
	[IPL_SOFTCLOCK] = MIPS_SPL_SOFTCLOCK,
	[IPL_SOFTBIO] = MIPS_SPL_SOFTBIO,
	[IPL_SOFTNET] = MIPS_SPL_SOFTNET,
	[IPL_SOFTSERIAL] = MIPS_SPL_SOFTSERIAL,
	[IPL_VM] = MIPS_SPL_VM,
	[IPL_SCHED] = MIPS_SPL_SCHED,
	[IPL_DDB] = MIPS_SPL_DDB,
	[IPL_HIGH] = MIPS_SPL_HIGH
    }
};

void
intr_init()
{
	ipl_sr_map = tsarmips_ipl_sr_map;
}

#ifdef MULTIPROCESSOR
int
cpu_send_ipi(struct cpu_info *ci, int tag)
{
	return xicu_send_ipi(ci, tag);
}
#endif

void
cpu_intr(int pri, vaddr_t pc, u_int32_t status)
{
	xicu_intr(pri, pc, status);
}

irq_t
intr_establish(device_t xicu, irq_type_t type, irq_line_t irq, ipl_t ipl,
    const char *name, char *intrstr, int (*handler)(void *), void *ih_arg,
    struct cpu_info *ci)
{
	irq_t intrhand;

	intrhand = malloc(sizeof(struct xicu_intrhand), M_TEMP, M_NOWAIT);
	if (intrhand == NULL)
		return NULL;
	intrhand->ih_func = handler;
	intrhand->ih_arg = ih_arg;
	if (xicu_establish(xicu, type, irq, ipl, name, intrstr, intrhand,
	    ci)) {
		free(intrhand, M_TEMP);
		return NULL;
	}
	evcnt_attach_dynamic(&intrhand->ih_ev, EVCNT_TYPE_INTR, NULL,
	    name, "intr");
	return intrhand;
}
