/* $NetBSD: $ */

/*-
  * Copyright (c) 2009 UPMC/LIP6
  * All rights reserved.
  * This software is distributed under the following condiions
  * compliant with the NetBSD foundation policy.
  *
  * Redistribution and use in source and binary forms, with or without
  * modification, are permitted provided that the following conditions
  * are met:
  * 1. Redistributions of source code must retain the above copyright
  *    notice, this list of conditions and the following disclaimer.
  * 2. Redistributions in binary form must reproduce the above copyright
  *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
  *
  * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  * POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: autoconf.c,v 1.71 2007/12/03 15:34:09 ad Exp $");

#include "opt_cputype.h"
#include "opt_soclib.h"
#include "opt_ddb.h"
#include "ksyms.h"
#include "opt_multiprocessor.h"
#include "simhelper.h"

#define __INTR_PRIVATE

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/atomic.h> 
#include <sys/conf.h>
#include <sys/reboot.h>
#include <sys/device.h>
#include <sys/mbuf.h>
#include <sys/msgbuf.h>
#include <sys/cpu.h>
#include <sys/kcore.h>
#include <sys/exec.h>
#include <sys/proc.h>
#include <sys/lwp.h>
#include <sys/savar.h>
#include <sys/ucontext.h>
#include <sys/user.h>

#include <sys/ras.h>
#include <sys/sysctl.h>
#include <sys/ksyms.h>
#include <sys/bitops.h>

#include <machine/autoconf.h>
#include <machine/bootinfo.h>
#include <machine/pte.h>
#include <machine/vcache.h>
#include <machine/locore.h>
#include <machine/intr.h>

#include <mips/cache.h>
#include <mips/psl.h>
#include <mips/userret.h>
#include <mips/frame.h>

#if NKSYMS || defined(DDB) || defined(LKM)
#include <machine/db_machdep.h>
#include <ddb/db_extern.h>
#endif

#include <uvm/uvm_extern.h>

void tsar_icache_sync_all(void);
void tsar_icache_sync_range(vaddr_t, vsize_t);
void tsar_icache_sync_range_index(vaddr_t, vsize_t);
void tsar_pdcache_wbinv_all(void);
void tsar_pdcache_wbinv_range(vaddr_t, vsize_t);
void tsar_pdcache_wbinv_range_index(vaddr_t, vsize_t);
void tsar_pdcache_inv_range(vaddr_t, vsize_t);
void tsar_pdcache_wb_range(vaddr_t, vsize_t);

struct mips_cache_ops mips_cache_ops = {
	.mco_icache_sync_all = tsar_icache_sync_all,
	.mco_icache_sync_range = tsar_icache_sync_range,
	.mco_icache_sync_range_index = tsar_icache_sync_range_index,
	.mco_pdcache_wbinv_all = tsar_pdcache_wbinv_all,
	.mco_pdcache_wbinv_range = tsar_pdcache_wbinv_range,
	.mco_pdcache_wbinv_range_index = tsar_pdcache_wbinv_range_index,
	.mco_pdcache_inv_range = tsar_pdcache_inv_range,
	.mco_pdcache_wb_range = tsar_pdcache_wb_range
};
	

void mach_init(void);

int physmem;
struct vm_map *mb_map = NULL;
struct vm_map *phys_map = NULL;

paddr_t mips_avail_end;	/* PA of last available physical page */
int mem_cluster_cnt;
phys_ram_seg_t mem_clusters[VM_PHYSSEG_MAX];

struct cpu_info cpu_info_store
#ifdef MULTIPROCESSOR
	__section(".data1")
	__aligned(1LU << ilog2((2*sizeof(struct cpu_info)-1)))
#endif
    = {
	.ci_curlwp = &lwp0,
#ifndef NOFPU
	.ci_fpcurlwp = &lwp0,
#endif
	.ci_cpl = IPL_HIGH,
#ifdef MULTIPROCESSOR
	.ci_flags = CPUF_PRIMARY|CPUF_PRESENT|CPUF_RUNNING,
#endif
}; /* boot CPU info */

volatile uint32_t tsar_cpus_running[BOOTINFO_NCPUS_MAX / 32];
volatile uint32_t tsar_cpus_hatched[BOOTINFO_NCPUS_MAX / 32];

struct btinfo_cpus *bi_cpus;

vaddr_t cp0_ebase;

void
mach_init()
{
	vaddr_t msgb;
	vaddr_t vaend;
	vaddr_t fdt_va;
	paddr_t paend, ram_start;
	struct btinfo_magic *bi_magic;
	struct btinfo_symtab *bi_sym;
	struct btinfo_fdt *bi_fdt;
	extern char kernel_text[], end[];
	int err;

	extern const struct splsw std_splsw;
	mips_splsw = std_splsw;

	intr_init();
	mttycn_attach();
	printf("mach_init\n");

	/* look up informations from boot loader */
	bi_magic = lookup_bootinfo(BTINFO_MAGIC, bootinfo);
	if (bi_magic && bi_magic->magic == BOOTINFO_MAGIC) {
		bi_sym = lookup_bootinfo(BTINFO_SYMTAB, bootinfo);
		bi_cpus = lookup_bootinfo(BTINFO_CPUS, bootinfo);
		bi_fdt = lookup_bootinfo(BTINFO_FDT, bootinfo);
	} else {
		panic("no bootinfo");
	}
	if (bi_fdt == NULL)
		panic("no FDT provided");
	printf("fdt @0x%lx, size 0x%lx\n", (u_long)bi_fdt->fdt_physaddr,
	    (u_long)bi_fdt->fdt_size);

	vaend = mips_round_page((vaddr_t)end);

#if NKSYM > 0 || defined(DDB) || defined(LKM)
	if (bi_sym)
		vaend = mips_round_page(bi_sym->esym + KERNBASE);
#endif
	paend = vaend - KERNBASE;
	printf("vaend 0x%lx paend 0x%lx\n", (u_long)vaend, (u_long)paend);
	pmap_bootstrap(&vaend, &msgb, &paend,
	    &fdt_va, bi_fdt->fdt_physaddr, bi_fdt->fdt_size);
	printf("vaend now 0x%lx paend 0x%lx\n", (u_long)vaend, (u_long)paend);
	printf("uvm_setpagesize");
	uvm_setpagesize();
#if NKSYMS || defined(DDB) || defined(LKM)
	if (bi_sym) {
		printf(" load symyab");
		ksyms_init(bi_sym->nsym, (char *)(bi_sym->ssym + KERNBASE),
		    (char *)(bi_sym->esym + KERNBASE));
	}
#endif
	printf(" done\n");
	/*
	 * Copy exception-dispatch code down to exception vector.
	 * Initialize locore-function vector.
	 */
	printf("mips_vector_init");
	mips_vector_init(NULL);
	printf(" done\n");

	boothowto = RB_SINGLE | AB_DEBUG;

#if 0 /* NKSYMS || defined(DDB) || defined(LKM) XXX */
	/* init symbols if present */
	if (esym)
		ksyms_init((char *)esym - (char *)ssym, ssym, esym);
#endif
	
	err = fdt_parse_header((void *)fdt_va, fdt);
	if (err != 0) {
		printf("fdt_parse_header failed: %d\n", err);
		panic("fdt_parse_header");
	}

	mem_cluster_cnt = 0;
	if (SOCLIB_RAM_START == 0)
		ram_start = PAGE_SIZE;
	else
		ram_start = SOCLIB_RAM_START;
	if (SOCLIB_RAM_START < (vaddr_t)kernel_text - KERNBASE) {
		mem_clusters[mem_cluster_cnt].start = SOCLIB_RAM_START;
		mem_clusters[mem_cluster_cnt].size =
		    ((vaddr_t)kernel_text - KERNBASE - SOCLIB_RAM_START);
		printf("mem_clusters[%d] 0x%" PRIx32 " 0x%" PRIx32 "\n",
		    mem_cluster_cnt, (u_int)mem_clusters[mem_cluster_cnt].start,
		    (u_int)mem_clusters[mem_cluster_cnt].size);
		uvm_page_physload(atop(SOCLIB_RAM_START),
		    atop((vaddr_t)kernel_text - KERNBASE),
		    atop(ram_start),
		    atop((vaddr_t)kernel_text - KERNBASE),
		    VM_FREELIST_DEFAULT);
		mem_cluster_cnt++;
	}
	if (SOCLIB_RAM_END > ((vaddr_t)kernel_text - KERNBASE)) {
		mem_clusters[mem_cluster_cnt].start = paend;
		mem_clusters[mem_cluster_cnt].size =
		    SOCLIB_RAM_END - paend;
		printf("mem_clusters[%d] 0x%" PRIx32 " 0x%" PRIx32 "\n",
		    mem_cluster_cnt, (u_int)mem_clusters[mem_cluster_cnt].start,
		    (u_int)mem_clusters[mem_cluster_cnt].size);
		uvm_page_physload(atop((vaddr_t)kernel_text - KERNBASE),
		    atop(SOCLIB_RAM_END),
		    atop(paend),
		    atop(SOCLIB_RAM_END),
		    VM_FREELIST_DEFAULT);
		mem_cluster_cnt++;
	}

	physmem = (SOCLIB_RAM_END - SOCLIB_RAM_START) / PAGE_SIZE;
	mips_avail_end = mips_trunc_page(SOCLIB_RAM_END - 1);

	/* initialise kernel message buffer */
	printf("initmsgbuf");
	initmsgbuf((void *)msgb, round_page(MSGBUFSIZE));
	printf(" done\n");
}

void *
lookup_bootinfo(int type, char *bootinfop)
{
	struct btinfo_common *bt;
	char *bi;

	bi = bootinfop;
	do {
		bt = (struct btinfo_common *)bi;
		//printf("bi %p %d %d\n", bi, bt->type, bt->next);
		if (bt->type == type)
			return (void *)bi;
		bi += bt->next;
	} while (bt->next != 0 &&
	    (vaddr_t)bi < (vaddr_t)bootinfop + BOOTINFO_SIZE);
	return NULL;
}

uint32_t loockup_bootinfo_cpus(char *);

uint32_t
loockup_bootinfo_cpus(char *bootinfop)
{
	struct btinfo_magic *bi_magic;
	struct btinfo_cpus *cpus;
	bi_magic = lookup_bootinfo(BTINFO_MAGIC, bootinfop);
	if (bi_magic && bi_magic->magic == BOOTINFO_MAGIC) {
		cpus = lookup_bootinfo(BTINFO_CPUS, bootinfop);
		if (cpus != NULL)
			return cpus->cpu_vectors;
	}
	return 0xffffffff;
}

void
consinit(void)
{
	mttycn_attach();
}

void
cpu_initclocks()
{
}

void setstatclockrate(int r)
{
}

void
cpu_startup()
{
	vaddr_t minaddr, maxaddr;
	char pbuf[9];
	/*
	 * Allocate a submap for physio
	 */
	phys_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    VM_PHYS_SIZE, 0, false, NULL);

	/*
	 * allocate mbuf cluster submap.
	 */
	mb_map = uvm_km_suballoc(kernel_map, &minaddr, &maxaddr,
	    nmbclusters * mclbytes, VM_MAP_INTRSAFE, false, NULL);

	/* say hello to the world */
	printf("%s%s", copyright, version);

	format_bytes(pbuf, sizeof(pbuf), physmem << PAGE_SHIFT);
	printf("total memory = %s\n", pbuf);
	format_bytes(pbuf, sizeof(pbuf), ptoa(uvmexp.free));
	printf("avail memory = %s\n", pbuf);
}

void
cpu_reboot(howto, bootstr)
        volatile int howto;     /* XXX volatile to keep gcc happy */
	char *bootstr;
{
	printf("%s\n\n", ((howto & RB_HALT) != 0) ? "halted." : "rebooting...");
#if NSIMHELPER > 0
	if (howto & RB_POWERDOWN)
		simhelper_end(0);
	else if (howto & RB_HALT)
		simhelper_end(1);
	else
		simhelper_end(2);
#endif
	while (1);
	/*NOTREACHED*/
}


void
tsar_icache_sync_all(void)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1);
#endif
}

void
tsar_icache_sync_range(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_icache_sync_range_index(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_ICACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_all(void)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_range(vaddr_t va, vsize_t size)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wbinv_range_index(vaddr_t va, vsize_t size)
{
	wbflush();
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_inv_range(vaddr_t va, vsize_t size)
{
#ifdef VCACHE
	tsar_vcache_write(VC_DCACHE_FLUSH, 1); /* XXX could do better */
#endif
}

void
tsar_pdcache_wb_range(vaddr_t va, vsize_t size)
{
	wbflush();
}

void
tsar_inval_icache_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t end = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < end) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_ICACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

void
tsar_inval_dcache_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t end = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < end) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_DCACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

void
tsar_inval_caches_parange(paddr_t pa, size_t sz)
{
#ifdef VCACHE
	paddr_t end = pa + sz;
	int lnsz = curcpu()->ci_l1_cls;

	pa = pa & ~((paddr_t)lnsz -1);
	while (pa < end) {
		tsar_vcache_write(VC_DATA_H, pa >> 32);
		tsar_vcache_write(VC_DATA_L, pa & 0xffffffff);
		tsar_vcache_write(VC_ICACHE_INVAL_PA, 0);
		tsar_vcache_write(VC_DCACHE_INVAL_PA, 0);
		pa += lnsz;
	}
#endif
}

extern struct locoresw mips32_locoresw;

struct locoresw mips_locoresw;

extern const struct splsw std_splsw;
struct splsw mips_splsw;

struct mips_options mips_options = {
        .mips_cpu_id = 0xffffffff,
	.mips_fpu_id = 0xffffffff,
};

/* the following is used externally (sysctl_hw) */
char	machine[] = MACHINE;		/* from <machine/param.h> */
char	machine_arch[] = MACHINE_ARCH;	/* from <machine/param.h> */
char	cpu_model[128];

static const struct pridtab tsar_cputab[] = {
	{ 0, 0x00163200, -1, -1, CPU_ARCH_MIPS32, -1,
	  CPU_MIPS_CAUSE_IV | CPU_MIPS_USE_WAIT, 0 , 0,
	  "TSAR mips32"},
};

/*
 * Initialize the hardware exception vectors, and the jump table used to
 * call locore cache and TLB management functions, based on the kind
 * of CPU the kernel is running on.
 */
void
mips_vector_init(const struct splsw *splsw)
{
	struct mips_options * const opts = &mips_options;
	/* r4000 exception handler address and end */
	extern char mips32_exception[];
	/* MIPS32/MIPS64 interrupt exception handler */
	extern char mips32_intr[], mips32_intr_end[];

	/*
	 * XXX Set-up curlwp/curcpu again.  They may have been clobbered
	 * beween verylocore and here.
	 */
	opts->mips_cpu = &tsar_cputab[0];
	opts->mips_cpu_arch = tsar_cputab[0].cpu_isa;
	opts->mips_num_tlb_entries = -1;

	opts->mips_cpu_flags = opts->mips_cpu->cpu_flags;
	opts->mips_has_llsc = (opts->mips_cpu_flags & CPU_MIPS_NO_LLSC) == 0;
	opts->mips3_pg_shift = PAGE_SHIFT;

	cpu_info_store.ci_cpuid = mips_cp0_ebase_read() & 0x1ff;
	lwp0.l_cpu = &cpu_info_store;
	curlwp = &lwp0;
	tsar_cpus_running[0] = 1; /* curcpu()->cpu_index is running */
	tsar_cpus_hatched[0] = 1; /* curcpu()->cpu_index is running */

	mips_splsw = std_splsw; /* XXX already done in mach_init */

	/*
	 * Now initialize our ISA-dependent function vector.
	 */

	printf(" ebase");
	/* write ebase register */
	mips_cp0_ebase_write(cp0_ebase);
	cp0_ebase = mips_cp0_ebase_read() & (~0xfff);
	printf("=0x%x", (uint32_t)cp0_ebase);

	/*
	 * Copy down exception vector code.
	 */

	printf(" memcpy1");
	if (mips32_intr - mips32_exception > 0x80)
		panic("startup: General exception vector code too large");
	memcpy((void *)(cp0_ebase + (MIPS3_GEN_EXC_VEC & PAGE_MASK)),
	    mips32_exception, mips32_intr - mips32_exception);
	printf(" memcpy2");
	if (mips32_intr_end - mips32_intr > 0x80)
		panic("startup: interrupt exception vector code too large");
#if 1	/* XXX - why doesn't mipsNN_intr() work? */
	memcpy((void *)(cp0_ebase + (MIPS3_INTR_EXC_VEC & PAGE_MASK)),
	    mips32_intr, mips32_intr_end - mips32_intr);
#else
	memcpy((void *)(cp0_ebase + (MIPS3_INTR_EXC_VEC & PAGE_MASK)),
	    mips32_exception, mips32_intr - mips32_exception);
#endif
	printf(" mips_icache_sync_all");
	mips_icache_sync_all();
	printf(" mips_dcache_wbinv_all");
	mips_dcache_wbinv_all();

	/* Clear BEV in SR so we start handling our own exceptions */
	printf(" mips_cp0_status_write");
	mips_cp0_status_write(mips_cp0_status_read() & ~MIPS_SR_BEV);
	printf(" done\n");
	mips_locoresw = mips32_locoresw;

	CPU_IDLE = mips_wait_idle;
}

/*
 * Set registers on exec.
 * Clear all registers except sp, pc, and t9.
 * $sp is set to the stack pointer passed in.  $pc is set to the entry
 * point given by the exec_package passed in, as is $t9 (used for PIC
 * code by the MIPS elf abi).
 */
void
setregs(l, pack, stack)
	struct lwp *l;
	struct exec_package *pack;
	u_long stack;
{
	struct trapframe *tf = l->l_md.md_utf;

	printf("setregs %p %p 0x%lx\n %p", l, pack, stack, l->l_proc->p_psstr);

	memset(tf, 0, sizeof(struct trapframe));
        tf->tf_regs[_R_SP] = (int)stack;
        tf->tf_regs[_R_PC] = (int)pack->ep_entry & ~3;
        tf->tf_regs[_R_T9] = (int)pack->ep_entry & ~3; /* abicall requirement */
        tf->tf_regs[_R_SR] = PSL_USERSET;

	/*
	 * Set up arguments for _start():
	 *	_start(stack, obj, cleanup, ps_strings);
	 *
	 * Notes:
	 *	- obj and cleanup are the auxiliary and termination
	 *	  vectors.  They are fixed up by ld.elf_so.
	 *	- ps_strings is a NetBSD extension.
	 */
        tf->tf_regs[_R_A0] = (intptr_t)stack;
        tf->tf_regs[_R_A1] = 0;
        tf->tf_regs[_R_A2] = 0;
        tf->tf_regs[_R_A3] = (intptr_t)l->l_proc->p_psstr;

	if (l->l_md.md_flags & MDP_FPUSED)
		fpudiscard_lwp(l);
	memset(&l->l_addr->u_pcb.pcb_fpregs, 0, sizeof(struct fpreg));
	l->l_md.md_flags &= ~MDP_FPUSED;
	l->l_md.md_ss_addr = 0;
}

SYSCTL_SETUP(sysctl_machdep_setup, "sysctl machdep subtree setup")
{

	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_NODE, "machdep", NULL,
		       NULL, 0, NULL, 0,
		       CTL_MACHDEP, CTL_EOL);

	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRUCT, "console_device", NULL,
		       sysctl_consdev, 0, NULL, sizeof(dev_t),
		       CTL_MACHDEP, CPU_CONSDEV, CTL_EOL);
#ifdef __HAVE_BOOTINFO_H
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRING, "booted_kernel", NULL,
		       sysctl_machdep_booted_kernel, 0, NULL, 0,
		       CTL_MACHDEP, CPU_BOOTED_KERNEL, CTL_EOL);
#endif
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT,
		       CTLTYPE_STRING, "root_device", NULL,
		       sysctl_root_device, 0, NULL, 0,
		       CTL_MACHDEP, CPU_ROOT_DEVICE, CTL_EOL);
	sysctl_createv(clog, 0, NULL, NULL,
		       CTLFLAG_PERMANENT|CTLFLAG_IMMEDIATE,
                       CTLTYPE_INT, "llsc", NULL,
                       NULL, MIPS_HAS_LLSC, NULL, 0,
                       CTL_MACHDEP, CPU_LLSC, CTL_EOL);
}

/*
 * These variables are needed by /sbin/savecore.
 */
u_int32_t dumpmag = 0x8fca0101;	/* magic number */
int	dumpsize = 0;		/* pages */
long	dumplo = 0;		/* blocks */

struct user dumppcb;		/* Actually, struct pcb would do. */

/*
 * This is called by main to set dumplo and dumpsize.
 * Dumps always skip the first CLBYTES of disk space
 * in case there might be a disk label stored there.
 * If there is extra space, put dump at the end to
 * reduce the chance that swapping trashes it.
 */
void
cpu_dumpconf(void)
{
	/* XXX notyer */
}

/*
 * Dump the kernel's image to the swap partition.
 */
#define	BYTES_PER_DUMP	PAGE_SIZE

void
dumpsys(void)
{
	/* XXX notyet */
}


/* 
 * Start a new LWP
 */
void
startlwp(arg)
	void *arg;
{
	ucontext_t *uc = arg;
	int err;

	err = cpu_setmcontext(curlwp, &uc->uc_mcontext, uc->uc_flags);
#if DIAGNOSTIC
	if (err) {
		printf("Error %d from cpu_setmcontext.", err);
	}
#endif
	pool_put(&lwp_uc_pool, uc);

	userret(curlwp);
}
