
// -*- mode: c++ -*-

#include <cstdlib>
#include <iostream>
#include <memory>
#include <list>
using std::cout;
using std::cerr;
using std::endl;
using std::auto_ptr;
using std::list;

#include "Lofig.h"
#include "Loins.h"
#include "Locon.h"
#include "Losig.h"

#include "hurricane/Warning.h"
#include "hurricane/UpdateSession.h"
#include "hurricane/Collection.h"
#include "hurricane/DataBase.h"
#include "hurricane/Technology.h"
#include "hurricane/Layer.h"
#include "hurricane/Library.h"
#include "hurricane/Cell.h"
#include "hurricane/Instance.h"
#include "hurricane/Net.h"
#include "hurricane/Plug.h"
#include "hurricane/Pin.h"
#include "hurricane/NetExternalComponents.h"
#include "crlcore/AllianceFramework.h"
#include "crlcore/Catalog.h"
#include "crlcore/ToolBox.h"
#include "RawViewer.h"
#include "MbkBridge.h"

using Hurricane::ForEachIterator;
using Hurricane::dbo_ptr;
using Hurricane::Error;
using Hurricane::DbU;
using Hurricane::Point;
using Hurricane::Box;
using Hurricane::Transformation;
using Hurricane::DataBase;
using Hurricane::Technology;
using Hurricane::Layer;
using Hurricane::Instance;
using Hurricane::Cell;
using Hurricane::Library;
using Hurricane::Net;
using Hurricane::Plug;
using Hurricane::Pin;
using Hurricane::NetExternalComponents;
using Hurricane::UpdateSession;
using CRL::AllianceFramework;
using CRL::Catalog;
using CRL::createPlugsAndPinsRing;
using CRL::placeNet;


static Library* __mbkLibrary = NULL;


Cell* toHurricane ( Lofig* lofig )
{
  Cell* cell = NULL;

  try {
    if ( !lofig ) {
      cerr << "[ERROR] mbkloParser(): empty lofig." << endl;
      exit(1);
    }

    if ( !DataBase::getDB() ) DataBase::create();

    if ( !__mbkLibrary ) {
      __mbkLibrary = Library::create ( DataBase::getDB(), "MBK" );
    }

  //cerr << "toHurricane() " << lofig->getName() << endl;

    if ( lofig->getName() == "and2" )
      return AllianceFramework::get()->getCell ( "a2_x2", Catalog::State::Views );
    if ( lofig->getName() == "or2" )
      return AllianceFramework::get()->getCell ( "o2_x2", Catalog::State::Views );
    if ( lofig->getName() == "xor2" )
      return AllianceFramework::get()->getCell ( "xr2_x1", Catalog::State::Views );

    cell = __mbkLibrary->getCell(lofig->getName());
    if ( cell ) return cell;

    cell = Cell::create ( __mbkLibrary, lofig->getName() );

    const list<Losig*>& sigs = lofig->getSignals();
    list<Losig*>::const_iterator isig = sigs.begin();
    for ( ; isig != sigs.end() ; ++isig ) {
      Net* net = Net::create ( cell, (*isig)->getName() );
      net->setExternal ( (*isig)->getType() == Locon::External );
    }

    const list<Locon*>& connectors = lofig->getConnectors();
    list<Locon*>::const_iterator icon = connectors.begin();
    for ( ; icon != connectors.end() ; ++icon ) {
    // Note: Signal name override connector name.
      string netName;
      Losig* sig = (*icon)->getSignal();

      if ( sig ) netName = sig->getName();
      else {
        netName = (*icon)->getName();
        cerr << "[ERROR] Locon <" << (*icon)->getName()
             << " of Lofig <" << getString(cell->getName())
             << "> is not connected (making dummy signal)." << endl;
      }
      
      Net* net = cell->getNet ( netName );
      if ( not net ) {
        Net* net = Net::create ( cell, netName );
        net->setExternal ( true );
      }

      switch ( (*icon)->getDirection() ) {
        case Locon::In:        net->setDirection(Net::Direction::IN);        break;
        case Locon::Out:       net->setDirection(Net::Direction::OUT);       break;
        case Locon::Inout:     net->setDirection(Net::Direction::INOUT);     break;
        case Locon::Tristate:  net->setDirection(Net::Direction::TRISTATE);  break;
        case Locon::Transcv:   net->setDirection(Net::Direction::TRISTATE);  break;
        default:
        case Locon::Unknown:   net->setDirection(Net::Direction::UNDEFINED); break;
      }
    }

    const list<Loins*>& instances = lofig->getInstances();
    list<Loins*>::const_iterator iins = instances.begin();
    for ( ; iins != instances.end() ; ++iins ) {
      Cell*     model      = toHurricane ( (*iins)->getModel() );
      Instance* huInstance = Instance::create ( cell, (*iins)->getName(), model );

      const list<Locon*>& connectors = (*iins)->getConnectors();
      list<Locon*>::const_iterator icon = connectors.begin();
      for ( ; icon != connectors.end() ; ++icon ) {
        Net* masterNet = huInstance->getMasterCell()->getNet((*icon)->getName());
        if ( not masterNet ) {
          cerr << "[ERROR] Model (Lofig) <" << getString(huInstance->getMasterCell()->getName())
               << "> of Instance (Loins) <" << getString(huInstance->getName())
               << "> do not have connector (Locon) named <" << (*icon)->getName()
               << "> (ignored)." << endl;
          continue;
        }

        Losig* signal = (*icon)->getSignal();
        if ( not signal ) {
          cerr << "[ERROR] Locon <" << (*icon)->getName()
               << " of instance <" << getString(huInstance->getName())
               << "> is not connected (ignored)." << endl;
          continue;
        }

        Net*  net  = cell->getNet(signal->getName());
        Plug* plug = huInstance->getPlug(masterNet);
        plug->setNet(net);
      }
    }
  }
  catch ( Error& e ) {
    cerr << e.what() << endl;
    exit ( 1 );
  }

  return cell;
}


Cell* placeAsNetlist ( Cell* cell, unsigned int depth )
{
  if (  cell->isTerminal() ) return cell;
  if ( !cell->getAbutmentBox().isEmpty() ) return cell;

  try {
    UpdateSession::open();
  
    Point bottomLeft (0,0);
  
    forEach ( Instance*, iinstance, cell->getInstances() ) {
      placeAsNetlist(iinstance->getMasterCell(),depth+1);
      Box modelBox = iinstance->getMasterCell()->getAbutmentBox();
  
      if (depth % 2)
        bottomLeft.translate(-modelBox.getXMin(),0);
      else
        bottomLeft.translate(0,-modelBox.getYMin());
  
      iinstance->setTransformation(Transformation(bottomLeft));
      iinstance->materialize();
  
      if (depth % 2)
        bottomLeft.translate(modelBox.getXMax(),0);
      else
        bottomLeft.translate(0,modelBox.getYMax());
    }
  
    Box ab = cell->getBoundingBox().inflate(DbU::lambda(10.0));
    cell->setAbutmentBox( ab );
  
    vector<Net*> inputs;
    vector<Net*> outputs;
    vector<Net*> undefineds;
  
    forEach ( Net*, inet, cell->getNets() ) {
      if ( not inet->isExternal() ) continue;

      switch ( inet->getDirection() ) {
        case Net::Direction::IN:
          inputs.push_back(*inet); break;
        case Net::Direction::OUT:
        case Net::Direction::INOUT:
        case Net::Direction::TRISTATE:
          outputs.push_back(*inet); break;
        case Net::Direction::UNDEFINED:
          undefineds.push_back(*inet); break;
      }
    }
  
    Layer* layer = DataBase::getDB()->getTechnology()->getLayer("METAL2");
  
    if (inputs.size()) {
      DbU::Unit step = ab.getHeight() / inputs.size();
      DbU::Unit y    = ab.getYMin() + step/2;
  
      for ( size_t i=0 ; i<inputs.size() ; ++i ) {
      //cerr << inputs[i]->getName() << ":" << DbU::getValueString(y) << endl;
        Pin* pin = Pin::create( inputs[i]
                              , inputs[i]->getName()
                              , Pin::AccessDirection::WEST
                              , Pin::PlacementStatus::PLACED
                              , layer
                              , ab.getXMin()
                              , y
                              , DbU::lambda(2.0)
                              , DbU::lambda(2.0)
                              );
        NetExternalComponents::setExternal(pin);
        y += step;
      }
    }
  
    if (outputs.size()) {
      DbU::Unit step = ab.getHeight() / outputs.size();
      DbU::Unit y    = ab.getYMin() + step/2;
  
      for ( size_t i=0 ; i<outputs.size() ; ++i ) {
      //cerr << outputs[i]->getName() << ":" << DbU::getValueString(y) << endl;
        Pin* pin = Pin::create( outputs[i]
                              , outputs[i]->getName()
                              , Pin::AccessDirection::EAST
                              , Pin::PlacementStatus::PLACED
                              , layer
                              , ab.getXMax()
                              , y
                              , DbU::lambda(2.0)
                              , DbU::lambda(2.0)
                              );
        NetExternalComponents::setExternal(pin);
        y += step;
      }
    }
  
    createPlugsAndPinsRing(cell);
  
    cerr << cell << endl;
    forEach ( Net*, inet, cell->getNets() ) {
      if ( not placeNet(*inet) ) {
        cerr << "  " << *inet << " is not placed" << endl;
        inet->setPosition(cell->getAbutmentBox().getCenter());
      }
    }
      
    UpdateSession::close();
  }
  catch ( Error& e ) {
    cerr << e.what() << endl;
    exit ( 1 );
  }

  return cell;
}
