/*
 * Gestion de la structure ABL (Arbres binaires Lisp-Like)
 * -----------------------------------------------------------------------------
 * Creation [10 mai 2006] [Alain Greiner, Franck Wajsb"urt]
 * -----------------------------------------------------------------------------
 */
#include <var.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

/*
 * Fonctions et Variables <static> (locales au fichier courant)
 * -----------------------------------------------------------------------------
 */

/* 
 * allocation d'une zone de memoire de <size> octets dans le tas 
 * avec verification du pointeur retourne par malloc. 
 * s'il n'y a plus de memoire, on affiche le message <mess> et on sort du
 * programme.
 * Cette fonction est <static>, c-a-d utilisable uniquement dans ce fichier
 */
static void *safe_malloc (size_t size, const char *mess)
{
    void *zone = malloc (size);
    if (zone == NULL)
    {
        perror (mess);
        exit (1);
    }
    return zone;
}

/*
 * verifie que <cond> est  1
 * si oui: on sort sans rien faire
 * si non: on affiche le message <mess> et on sort
 * Cette fonction est <static>, c-a-d utilisable uniquement dans ce fichier
 * exemple: assert ((1==0),"ligne %d: toujours faux %s", 46, "evidemment")
 */
static void assert (unsigned cond, const char *mess, ...)
{
    va_list arg;
    va_start(arg, mess);
    if (cond == 0)
    {
        fprintf (stderr, "ERREUR var : ");
        vfprintf (stderr, mess , arg);
        fprintf (stderr, "\n");
        exit (1); 
    }
}

/* 
 * Dictionnaire de variable indexe par le nom
 * -----------------------------------------------------------------------------
 * La cle de recherche est le nom de la variable.
 * La valeur associee a la cle est un pointeur vers la structure definissant 
 * la variable.
 * Le dictionnaire est cree statiquement dans une variable globale
 */
typedef struct dico_name_t
{
    char *NAME;
    var_t *VAR;
    struct dico_name_t *NEXT;
}
dico_name_t;
static dico_name_t *dico_name[NB_VAR];

/*
 * Fonction de recherche/creation d'une entree dans le dictionnaire de 
 * variable indexe par les noms.
 * La variable de nom <name> est recherche dans le dictionnaire.
 * En sortie, 
 * - la variable pointe par *hit est initialise avec l'entree du
 *   dictionnaire contenant cette variable.
 * - et la fonction rend
 *   1 si l'entree existait deja 
 *   0 si l'entree n'existait pas et qu'elle a du etre cree
 */
static int lookup_dico_name (char *name, dico_name_t ** hit)
{
    assert (name && strlen (name), "nom de variable incorrect : %s", name);

    /* calcul de la valeur de hachage : code */
    unsigned code = 0;
    unsigned len_name = strlen (name);
    unsigned l;
    for (l = 0; l < len_name; l += 4)
    {
        unsigned seg =
            (name[(l + 3) % len_name] << 24) | (name[(l + 2) % len_name] << 16)
            | (name[(l + 1) % len_name] << 8) | (name[l]);
        code = (code >> 1) ^ seg;
    }
    code = code % NB_VAR;

    /* cherche l'entree name et sort si trouve */
    dico_name_t *item;
    for (item = dico_name[code]; item; item = item->NEXT)
    {
        if (strcmp (item->NAME, name) == 0)
        {
            *hit = item;
            return 1;
        }
    }

    /* creation si absent */
    item = safe_malloc (sizeof (dico_name_t), "lookup_dico_name");
    item->NAME = safe_malloc (strlen (name), "lookup_dico_name");
    strcpy (item->NAME, name);
    item->NEXT = dico_name[code];
    dico_name[code] = item;
    *hit = item;
    return 0;
}

/* 
 * Dictionnaire de variable indexe par l'index
 * -----------------------------------------------------------------------------
 * La cle de recherche est l'index de la variable.
 * La valeur associee a la cle est un pointeur vers la structure definissant la
 * variable.
 * Le dictionnaire est cree statiquement dans une variable globale
 */
typedef struct dico_index_t
{
    unsigned INDEX;
    var_t *VAR;
    struct dico_index_t *NEXT;
}
dico_index_t;
static dico_index_t *dico_index[NB_VAR];

/*
 * Fonction de recherche/creation d'une entree dans le dictionnaire de 
 * variable indexe par les index.
 * La variable d'index <index> est recherche dans le dictionnaire.
 * En sortie, 
 * - la variable pointe par *hit est initialise avec l'entree du
 *   dictionnaire contenant cette variable.
 * - et la fonction rend
 *   1 si l'entree existait deja 
 *   0 si l'entree n'existait pas et qu'elle a du etre cree
 */
static int lookup_dico_index (unsigned index, dico_index_t ** hit)
{
    /* calcul de la valeur de hachage : code */
    unsigned code = index % NB_VAR;

    /* cherche l'entree name et sort si trouve */
    dico_index_t *item;
    for (item = dico_index[code]; item; item = item->NEXT)
        if (item->INDEX == index)
        {
            *hit = item;
            return 1;
        }

    /* creation si absent */
    item = safe_malloc (sizeof (dico_index_t), "lookup_dico_index");
    item->INDEX = index;
    item->NEXT = dico_index[code];
    dico_index[code] = item;
    *hit = item;
    return 0;
}

/*
 * Fonctions et Variables <extern> (utilisables par tous)
 * -----------------------------------------------------------------------------
 */

/* creation d'une variable (commentaire dans var.h) */
var_t *cons_var (char *name, unsigned index, unsigned value)
{
    var_t *var;
    dico_name_t *hit_name;
    dico_index_t *hit_index;
    /* verifie l'unicite de name en utilisant le dictionnaire de nom */
    assert ((lookup_dico_name (name, &hit_name) == 0),
            "cons_var: nom deja utilise : %s", name);
    /* verifie l'unicite de l'index en utilisant le dictionnaire d'index */
    assert ((lookup_dico_index (index, &hit_index) == 0),
            "cons_var: index deja utilise : %d", index);
    /* verifie la validit de la valeur */
    assert (((value == 0) || (value == 1)), "valeur non Booleene");
    /* creation */
    var = safe_malloc (sizeof (var_t), "cons_var");
    var->NAME = hit_name->NAME;
    var->INDEX = index;
    var->VALUE = value;
    hit_name->VAR = var;
    hit_index->VAR = var;
    return var;
}

/* acces a une variable par son index (commentaire dans var.h) */
var_t *get_var_index (unsigned index)
{
    dico_index_t *hit_index;
    assert (lookup_dico_index (index, &hit_index),
            "get_var_index : variable d'index inconnu : %d", index);
    return hit_index->VAR;
}

/* acces a une variable par son nom (commentaire dans var.h) */
var_t *get_var_name (char *name)
{
    dico_name_t *hit_name;
    assert (lookup_dico_name (name, &hit_name),
            "get_var_name : variable de nom inconnu: %s", name);
    return hit_name->VAR;
}
