; ------------------------------------------------------------------------------
; programme : boutonit
; Date      : 20050324:0414 
; Version   : 2.1
; Auteurs   : franck
; Notes     : suppose que le Quartz est  20MHz
; ------------------------------------------------------------------------------
; inversion du portD<7> a chaque appui du bouton poussoir reli a INT
; ------------------------------------------------------------------------------

; ------------------------------------------------------------------------------ 
; CONFIGURATION GNRALE 
; ------------------------------------------------------------------------------ 
 
        list    p=16f877        ; defini le processeur cible 
        list    n=0             ; pour ne pas avoir de CTRL-L dans le .lst
        include "p16f877.inc"   ; declaration des noms de registres 
        errorlevel -302         ; pour eviter les messages d'erreur de bank
 
        ; Definition du registre de configuration du PIC 
        ; _CP_OFF   : le code n'est pas protege et peut etre relu 
        ; _WDT_OFF  : pas de timer watch dog 
        ; _PWRTE_ON : attente d'un dlai apres le power on 
        ; _HS_OSC   : oscillateur  quartz 
        ; _LVP_OFF  : pas de mode programmation basse tension 
        __CONFIG _CP_OFF & _WDT_OFF & _PWRTE_ON & _HS_OSC & _LVP_OFF 
 
; ------------------------------------------------------------------------------ 
; TOPOLOGIE MEMOIRE A L'USAGE DU SYSTEME ET DEFINITION DE CONSTANTES
; ------------------------------------------------------------------------------ 
; Les 4 registres W0  W3 sont des registres de travail, ils peuvent tre 
; utiliss pour faire des calculs temporaires ou pour passer des parametres 
; une routine ou rcuprer son rsultat. 
; ------------------------------------------------------------------------------ 
; Remarquez la mthode utilise pour l'allocation des registres dans les bancs.
; On utilise CBLOCK et on maintient  jour des variables _BANK0_, _BANK1_, etc
; qui doivent contenir la prochaine adresse libre dans le bank dsign. 
; Par exemple, pour faire plus loin dans le texte une nouvelle allocation dans 
; le bank0, il faudra faire :
;
; CBLOCK _BANK0_ 
;   ........ 
;   ex1  : 5     ; un registre quelconque sur 5 octets
; ENDC 
; variable _BANK0_ = ex1+5 ; le 5 vient du fait que le reg ex1 fait 5 octets
;
; le registre ex1 est juste un exemple, il est sur 5 octets pour vous montrer 
; qu'il faut bien faire attention  ce que _BANK0_ designe effectivement une 
; adresse libre.
; ------------------------------------------------------------------------------ 
 
        ; allocation des registres dans la zone partage
        ; ----------------------------------------------
        CBLOCK 0x70 
          W0            ; registres de travail
          W1            ;    " 
          W2            ;    " 
          W3            ;    " 
          EVENT         ; registre contenant des vnements indpendants
          _WBAK         ; sauvegarde de W      pour le gestionnaire d'interrupt.
          _STATUSBAK    ; sauvegarde de STATUS pour le gestionnaire d'interrupt.
        ENDC 
 
        ; allocation des registres dans les banks utilisateur
        ; ---------------------------------------------------
        variable _BANK0_ = 0x20  ; premier octet libre dans le bank0
        variable _BANK1_ = 0xA0  ; premier octet libre dans le bank1
        variable _BANK2_ = 0x110 ; premier octet libre dans le bank2
        variable _BANK3_ = 0x190 ; premier octet libre dans le bank3
        
        CBLOCK _BANK0_
          _PCLATHBAK    ; sauvegarde de PCLATH pour le gestionnaire d'interrupt.
          _W0           ; registres de travail temporaires utiliss exclusivement
          _W1           ; par le gestionnaire d'interruption
          _BP_VAL       ; succession des valeurs du bouton poussoir
        ENDC 
        variable _BANK0_ = _BP_VAL + 1  ; mise  jour du premier octet libre

        ; declaration des numros de bit du registre EVENT
        ;-------------------------------------------------
        #define APPUI   EVENT,0 ; passe  1 si le bouton a t appuy

; ------------------------------------------------------------------------------
; DEBUT DE PROGRAMME
; ------------------------------------------------------------------------------

        ORG     0
        call    INITIALISATION
        goto    MAIN    ; sauter au programme principal

; ------------------------------------------------------------------------------
; GESTIONNAIRE D'INTERRUPTION
; ------------------------------------------------------------------------------
; Remarques importantes:
;
; 1. le gestionnaire est toujours en 3 parties 
;    a. sauvegarde contexte, b. analyse des causes, c. restauration contexte
;    Il est suivi par les ISR. Nous avons prvu ici toutes les ISR possibles.
;    Il vous a mettre du code derniere les tiquettes ISR_..
;
; 2. On choisit de ne pas sauver le registre FSR  l'entre du gestionnaire
;    il ne faut donc pas l'utiliser dans les ISR (ou alors elle doivent le 
;    sauver et le restaurer seules).
;
; 3. les ISR sont les routines de traitement des interruptions.
;    - elles doivent tre courtes (pas de boucle, pas d'attente)
;    - elles doivent IMPERATIVEMENT contenir l'effacement du drapeau
;      sinon on retourne en interruption aussitot apres le retfie.
;    - elle doivent IMPERATIVEMENT revenir dans le bank0 avant de sortir.
;    - elles se terminent par un goto _ISR_RESTAURE (et pas par un return).
;      On ne traite qu'une interruption  la fois, car d'une part le risque 
;      d'interruption multiple est faible, et d'autre part on garantie que la 
;      priorit des interruptions est respecte (ordre du test des causes).
; 
; 4. pour l'analyse des causes, on choisit de rendre les tests systmatiques
;    et on s'aide d'une macro instruction qui effectue ces tests. Vous savez
;    que vous devez tester le bit enable d'une interruption avant de tester 
;    le drapeau de l'interruption. Malheureusement, hormis pour trois
;    interruptions, les registres enable se trouvent dans le bank1 (reg PIE1 et
;    PIE2) alors que les drapeaux sont dans le bank0 (PIR1 et PIR2). Afin de
;    reduire le nombre de lignes du gestionnaire et sa dure, on a choisit de
;    faire des copies temporaires des registres enable dans le bank0, mais ces
;    copies ne sont faites que si c'est ncessaire.
;    On utiliser le registre _W0 pour PIE1 et _W1 pour PIE2
;    Une fois dans l'ISR, ces registres sont utilisables pour autre chose
;    car on ne retourne pas dans l'analyse des causes.
; ------------------------------------------------------------------------------

   ; ----------------------------------------------------------------------
   ; MACRO DEDIEE AU TEST DES CAUSES D'INTERRUPTION
   ; ----------------------------------------------------------------------
   ; ISR : Interrupt Service Routine= adresse de la routine d'interruption
   ; IER : Interrupt Enable Register= numero du registre du bit enable
   ; IEB : Interrupt Enable Bit     = numero du bit enable dans IER 
   ; IFR : Interrupt Flag Register  = numero du registre du bit drapeau
   ; IFB : Interrupt Flag Bit       = numero du bit drapeau dans IFR 
   ; ENABLE  : drapeau d'activation du test de cause, 3 valeurs possibles
   ; ----------------------------------------------------------------------
   variable never_enable  = 0 ; interruption jamais utilise
   variable always_enable = 1 ; interruption utilise et non masquable
   variable maybe_enable  = 2 ; interruption utilise et masquable
   variable _pie1_copied  = 0 ;  1 quand la recopie de registre est faite
   variable _pie2_copied  = 0 ;  1 quand la recopie de registre est faite

ISRCALL macro   ISR,IER,IEB,IFR,IFB,ENABLE
          if (ENABLE!=never_enable)   ; si l'interruption est utilise on entre
            if (ENABLE==maybe_enable) ; sinon si l'int pas tjs util on teste ena
              variable _IER = IER     ; par defaut le registre enable est IER
              if (IER==PIE1)
                variable _IER = _W0   ; changement de registre enable
              endif
              if (IER==PIE2)
                variable _IER = _W1   ; changement de registre enable
              endif
              if ((IER==PIE1)&&(_pie1_copied==0))
                variable _pie1_copied = 1
                bsf     STATUS,RP0    ; passe du bank0 au bank1
                movf    PIE1,w        ; va chercher le registre PIE1
                bcf     STATUS,RP0    ; passe du bank0 au bank1
                movwf   _W0           ; et en fait une copie
              endif
              if ((IER==PIE2)&&(_pie2_copied==0))
                variable _pie2_copied = 1
                bsf     STATUS,RP0    ; passe du bank0 au bank1
                movf    PIE2,w        ; va chercher le registre PIE2
                bcf     STATUS,RP0    ; passe du bank0 au bank1
                movwf   _W1           ; et en fait une copie
              endif
              btfss   _IER, IEB ; on teste juste le bit enable
              goto    $+3       ; le bit enable n'est pas  1, on saute
            endif
            btfsc   IFR, IFB    ; saute  la routine de traitement si FLAG = 1
            goto    ISR         ; routine de traitement de T0
          endif
        endm

        ORG     4       ; adresse du gestionnaire d'interruption

    ; --------------------------------------------------------------------------
    ; SAUVEGARDE DES REGISTRES W, STATUS, FSR, PCLATH et clrf STATUS et PCLATH
    ; --------------------------------------------------------------------------

_ISR_SAUVE
        movwf   _WBAK           ; sauve le registre W
        swapf   STATUS,W        ; W <- STATUS [ 3:0 , 7:4 ]
        movwf   _STATUSBAK      ; sauve STATUS, sans toucher aux drapeaux
        clrf    STATUS          ; STATUS <- 0 donc BANK0
        movf    PCLATH,W
        movwf   _PCLATHBAK      ; sauve le registre PCLATH
        clrf    PCLATH          ; PCLATH <- 0 donc PAGE0

    ; --------------------------------------------------------------------------
    ; TEST DES 14 CAUSES D'INTERRUPTION
    ; --------------------------------------------------------------------------
    ; l'ordre dans lequel les causes sont testes peut tre modifi
    ; la macro ISRCALL n'ajoute du code que si le dernier arg est different
    ; de never_enable.
    ; --------------------------------------------------------------------------

_ISR_CAUSES
        ISRCALL ISR_T0,   INTCON, T0IE,   INTCON, T0IF,   always_enable
        ISRCALL ISR_INT,  INTCON, INTE,   INTCON, INTF,   maybe_enable
        ISRCALL ISR_RB,   INTCON, RBIE,   INTCON, RBIF,   never_enable
        ISRCALL ISR_PSP,  PIE1,   PSPIE,  PIR1,   PSPIF,  never_enable
        ISRCALL ISR_AD,   PIE1,   ADIE,   PIR1,   ADIF,   never_enable
        ISRCALL ISR_RC,   PIE1,   RCIE,   PIR1,   RCIF,   never_enable
        ISRCALL ISR_TX,   PIE1,   TXIE,   PIR1,   TXIF,   never_enable
        ISRCALL ISR_SSP,  PIE1,   SSPIE,  PIR1,   SSPIF,  never_enable
        ISRCALL ISR_CCP1, PIE1,   CCP1IE, PIR1,   CCP1IF, never_enable
        ISRCALL ISR_TMR2, PIE1,   TMR2IE, PIR1,   TMR2IF, never_enable
        ISRCALL ISR_TMR1, PIE1,   TMR1IE, PIR1,   TMR1IF, never_enable
        ISRCALL ISR_EE,   PIE2,   EEIE,   PIR2,   EEIF,   never_enable
        ISRCALL ISR_BCL,  PIE2,   BCLIE,  PIR2,   BCLIF,  never_enable
        ISRCALL ISR_CCP2, PIE2,   CCP2IE, PIR2,   CCP2IF, never_enable

    ; --------------------------------------------------------------------------
    ; RESTAURATION DES REGISTRES DE TRAVAIL
    ; --------------------------------------------------------------------------

_ISR_RESTAURE
        movf    _PCLATHBAK,W
        movwf   PCLATH          ; restaure PCLATH
        swapf   _STATUSBAK,W    ; W <- 0x7C+1 [ 7:4 , 3:0 ]
        movwf   STATUS          ; STATUS <- W
        swapf   _WBAK,F         ; 0x7C <- 0x7C [ 3:0 , 7:4 ]
        swapf   _WBAK,W         ; W <- 0x7C [ 7:4 , 3:0 ]
        retfie                  ; sortie avec GIE <- 1

    ; --------------------------------------------------------------------------
    ; (I)nterrupt (S)ervice (R)outine : Routines d'interruption
    ; --------------------------------------------------------------------------
    ; - quand une ISR est appele, STATUS pointe toujours sur le bank0
    ;   l'ISR doit doit effacer le drapeau et revenir dans le bank0 avant
    ;   de sortir par un goto _ISR_RESTAURE
    ; - Si vous devez ajouter une ISR vous mettez le code derrire l'tiquette
    ;   correspondante et vous changer le parametre never_enable par 
    ;   maybe_enable ou par always_enable
    ; --------------------------------------------------------------------------

ISR_T0  ; ------------------------------------------------ dbordement du TIMER0
        ; echantillonage des valeurs du bouton poussoir tous les 13ms
        ; ramement de l'interruption INT en cas de dtection d'un front
        ; montant vrai de sur INT (si les 3 dernieres valeurs lues sont 011) 
        ; ----------------------------------------------------------------------
        bcf     INTCON,T0IF     ; effacement du drapeau du TIMER0
        rrf     PORTB,W         ; PORTB0 -> Carry
        rlf     _BP_VAL,f       ; VALn-2<-VALn-1 ; VALn-1<-VALn ; VALn<-PORTB0
        movf    _BP_VAL,w       ; dtection du relachement
        andlw   B'111'          ; Si (( _BP_VAL & B'111' ) != B'011')
        xorlw   B'011'          ;
        btfss   STATUS,Z        ; si le bouton n'est pas lach depuis 2x13ms
        goto    _ISR_RESTAURE   ; sortie de l'ISR
        bcf     INTCON,INTF     ; effacement du drapeau (caus par les rebonds)
        bsf     INTCON,INTE     ; rarmement de l'interruption
        goto    _ISR_RESTAURE   ; sortie de l'ISR

ISR_INT ; ----------------------------------------------------- interruption INT
        ; attente d'un front descendant, emission d'un signal,
        ; desarment de l'interruption
        ; ----------------------------------------------------------------------
        bcf     INTCON,INTF     ; effacement du drapeau 
        bsf     APPUI           ; EVENT.APPUI <- 1 (action  faire)
        bcf     INTCON,INTE     ; dactive l'interruption INT
        goto    _ISR_RESTAURE

ISR_RB  ; ------------------------------ changement d'tat des broches RB0  RB3
ISR_PSP ; ---------------- lecture ou criture sur le port parallele PORTD,PORTE
ISR_AD  ; ----------------------------------------- fin de conversion ANALOGIQUE
ISR_RC  ; ---------------------------------- fin de reception d'une donne RS232
ISR_TX  ; ------------------------------------ fin d'emission d'une donne RS232
ISR_SSP ; ---------------------- fin de transfert lors d'un transfert SPI ou I2C
ISR_CCP1; ------------------ une comparaison ou une capture a t faite sur CCP1
ISR_TMR2; ----------------------------- les registres TMR2 et PIR2 ont t gaux
ISR_TMR1; ------------------------------------------------ dbordement du TIMER1
ISR_EE  ; ----------------------------------------- fin d'criture dans l'EEPROM
ISR_BCL ; ----------------------------------------------- collision I2C dtecte
ISR_CCP2; ------------------ une comparaison ou une capture a t faite sur CCP2

; ------------------------------------------------------------------------------
; TOPOLOGIE MEMOIRE UTILISATEUR
; ------------------------------------------------------------------------------

        CBLOCK _BANK0_
        ENDC

        CBLOCK _BANK1_
        ENDC

        CBLOCK _BANK2_
        ENDC

        CBLOCK _BANK3_
        ENDC

; ------------------------------------------------------------------------------
; initialisation generale
; ------------------------------------------------------------------------------

INITIALISATION

    ; --------------------------------------------------------------------------
    ; direction des entres sorties
    ; --------------------------------------------------------------------------

        BANKSEL TRISD           ; aller dans le bank du registre TRSID (bank1)
        bcf     TRISD,7         ; programme la broche 7 du port D en sortie

    ; --------------------------------------------------------------------------
    ; initialisation du TIMER0 : une interruption ts les 0x10000 cycles
    ; --------------------------------------------------------------------------

        BANKSEL OPTION_REG
        bcf     OPTION_REG,T0CS ; timer0 en mode timer
        bcf     OPTION_REG,PSA  ; PSA <- 0 Prescaler pour le timer0
        bsf     OPTION_REG,PS0  ; PS[2:0] <- 111, prescaler  256
        bsf     OPTION_REG,PS1
        bsf     OPTION_REG,PS2
        bcf     INTCON,T0IF     ; effacer le drapeau d'interruption
        bsf     INTCON,T0IE     ; autoriser l'interruption du TIMER0

    ; --------------------------------------------------------------------------
    ; initialisation de l'entree interruption externe INT
    ; --------------------------------------------------------------------------

        BANKSEL OPTION_REG
        bcf     OPTION_REG,NOT_RBPU ; active le pullup sur le port B
        bcf     OPTION_REG,INTEDG   ; surveiller le front descendant de INT
        bcf     INTCON,INTF     ; effacer le drapeau d'interruption
        bsf     INTCON,INTE     ; autoriser l'interruption du INT

    ; --------------------------------------------------------------------------
    ; initialisation des registres de travail et sortie de l'initialisation
    ; --------------------------------------------------------------------------

        clrf    STATUS          ; revient au bank0
        bcf     PORTD,7         ; mettre le PORTD bit 7 teint
        clrf    _BP_VAL         ; l'etat du bouton
        comf    _BP_VAL,f       ; tat libre du bouton tir par le pull-up
        clrf    EVENT           ; les signaux de synchronisation

        bsf     INTCON,PEIE     ; autoriser les interruptions periphriques
        retfie                  ; autoriser les interruptions et sortir de l'init

; ------------------------------------------------------------------------------
; programme principal
; ------------------------------------------------------------------------------

MAIN    
        btfss   APPUI           ; attendre que la variable APPUI  1
        goto    $-1
        bcf     APPUI           ; then APPUI <- 0 (vnement vu)

        movlw   B'10000000'     ; 
        xorwf   PORTD,f         ; on vient de voir l'appui, inverser le bit 7
        goto    MAIN            ; on boucle sur l'attente de l'appui

; ------------------------------------------------------------------------------
        END                     ; directive terminant un programme
