//////////////////////////////////////////////////////////////////////////////
// File: tsar_iob_cluster.cpp
// Author: Alain Greiner 
// Copyright: UPMC/LIP6
// Date : april 2013
// This program is released under the GNU public license
//
// Modified by: Cesar Fuguet
// Modified on: mars 2014
//////////////////////////////////////////////////////////////////////////////
// Cluster(0,0) & Cluster(xmax-1,ymax-1) contains the IOB0 & IOB1 components.
// These two clusters contain 6 extra components:
// - 1 vci_io_bridge (connected to the 3 networks.
// - 3 vci_dspin_wrapper for the IOB.
// - 2 dspin_local_crossbar for commands and responses. 
//////////////////////////////////////////////////////////////////////////////

#include "../include/tsar_iob_cluster.h"

#define tmpl(x) \
   template<typename vci_param_int      , typename vci_param_ext,\
            size_t   dspin_int_cmd_width, size_t   dspin_int_rsp_width,\
            size_t   dspin_ram_cmd_width, size_t   dspin_ram_rsp_width>\
            x TsarIobCluster<\
                  vci_param_int      , vci_param_ext,\
                  dspin_int_cmd_width, dspin_int_rsp_width,\
                  dspin_ram_cmd_width, dspin_ram_rsp_width>

namespace soclib { namespace caba  {

//////////////////////////////////////////////////////////////////////////
//                 Constructor
//////////////////////////////////////////////////////////////////////////
tmpl(/**/)::TsarIobCluster(struct ClusterParams& params) :
   soclib::caba::BaseModule(params.insname),
   p_clk("clk"),
   p_resetn("resetn") {

   assert((params.x_id < X_MAX) && (params.y_id < Y_MAX));

   size_t cid           = this->clusterId(params.x_id, params.y_id);
   size_t cluster_iob0  = this->clusterId(0, 0);
   size_t cluster_iob1  = this->clusterId(X_SIZE - 1, Y_SIZE - 1);
   size_t is_iob0       = (cid == cluster_iob0);
   size_t is_iob1       = (cid == cluster_iob1);
   bool   is_io_cluster = is_iob0 || is_iob1;

   size_t l_width = vci_param_int::S - X_WIDTH - Y_WIDTH;

   // Vectors of DSPIN ports for inter-cluster communications
   p_dspin_int_cmd_in =
      alloc_elems<DspinInput<dspin_int_cmd_width> >("p_int_cmd_in", 4, 3);
   p_dspin_int_cmd_out =
      alloc_elems<DspinOutput<dspin_int_cmd_width> >("p_int_cmd_out", 4, 3);
   p_dspin_int_rsp_in =
      alloc_elems<DspinInput<dspin_int_rsp_width> >("p_int_rsp_in", 4, 2);
   p_dspin_int_rsp_out =
      alloc_elems<DspinOutput<dspin_int_rsp_width> >("p_int_rsp_out", 4, 2);

   p_dspin_ram_cmd_in =
      alloc_elems<DspinInput<dspin_ram_cmd_width> >("p_ext_cmd_in", 4);
   p_dspin_ram_cmd_out =
      alloc_elems<DspinOutput<dspin_ram_cmd_width> >("p_ext_cmd_out", 4);
   p_dspin_ram_rsp_in =
      alloc_elems<DspinInput<dspin_ram_rsp_width> >("p_ext_rsp_in", 4);
   p_dspin_ram_rsp_out =
      alloc_elems<DspinOutput<dspin_ram_rsp_width> >("p_ext_rsp_out", 4);

   // ports in cluster_iob0 and cluster_iob1 only
   p_vci_iob_iox_ini   = NULL;
   p_vci_iob_iox_tgt   = NULL;
   p_dspin_iob_cmd_out = NULL;
   p_dspin_iob_rsp_in  = NULL;
   if ( is_io_cluster ) {
      // VCI ports from IOB to IOX network
      p_vci_iob_iox_ini = new soclib::caba::VciInitiator<vci_param_ext>;
      p_vci_iob_iox_tgt = new soclib::caba::VciTarget<vci_param_ext>; 

      // DSPIN ports from IOB to RAM network
      p_dspin_iob_cmd_out = new soclib::caba::DspinOutput<dspin_ram_cmd_width>;
      p_dspin_iob_rsp_in  = new soclib::caba::DspinInput<dspin_ram_rsp_width>;
   }

   // IRQ ports in cluster_iob0 only
   for ( size_t n = 0 ; n < 32 ; n++ ) {
      p_irq[n] = ( is_iob0 ) ? new sc_in<bool> : NULL;
   }

   ///////////////////////////////////////////////////////////////////////////
   //    Hardware components
   ///////////////////////////////////////////////////////////////////////////

   ////////////  PROCS
   for (size_t p = 0; p < NB_PROCS; p++) { 
      std::ostringstream s_proc;
      s_proc << "proc_" << params.x_id << "_" << params.y_id << "_" << p;
      proc[p] = new VciCcVCacheWrapperType (
            s_proc.str().c_str(),
            cid * NB_PROCS + p,
            params.mt_int,
            IntTab(cid,p),
            (cid << l_width) + p,
            8, 8,
            8, 8,
            params.l1_i_ways, params.l1_i_sets, 16,
            params.l1_d_ways, params.l1_d_sets, 16,
            4, 4,
            X_WIDTH, Y_WIDTH,
            params.frozen_cycles,
            params.debug_start_cycle, params.proc_debug_ok);

      proc[p]->set_dcache_paddr_ext_reset(cid);
      proc[p]->set_icache_paddr_ext_reset(cid);
   }

   ///////////  MEMC    
   std::ostringstream s_memc;
   s_memc << "memc_" << params.x_id << "_" << params.y_id;
   memc = new VciMemCacheType (
         s_memc.str().c_str(),
         params.mt_int,
         params.mt_ext,
         IntTab(cid, RAM_MEMC_INI_ID),
         IntTab(cid, INT_MEMC_TGT_ID),
         X_WIDTH,
         Y_WIDTH,
         params.memc_ways, params.memc_sets, 16,
         3,
         4096,
         8, 8, 8,
         params.debug_start_cycle,
         params.memc_debug_ok);

   std::ostringstream s_wi_memc;
   s_wi_memc << "memc_wi_" << params.x_id << "_" << params.y_id;
   memc_ram_wi = new VciExtDspinInitiatorWrapperType (
         s_wi_memc.str().c_str(),
         vci_param_int::S);

   ///////////   LOCAL ROM
   std::ostringstream s_brom;
   s_brom << "brom_" << params.x_id << "_" << params.y_id;
   brom = new VciSimpleRom<vci_param_int>(
         s_brom.str().c_str(),
         IntTab(cid, INT_BROM_TGT_ID),
         params.mt_int,
         params.loader,
         X_WIDTH + Y_WIDTH);

   // Multi-TTY controller
   mtty        = NULL;
   if (NB_DEBUG_TTY_CHANNELS) {
      assert(NB_DEBUG_TTY_CHANNELS < 8);

      std::ostringstream s_mtty;
      s_mtty << "mtty_" << params.x_id << "_" << params.y_id;
      std::vector<std::string> vect_names;
      for( size_t tid = 0 ; tid < NB_DEBUG_TTY_CHANNELS ; tid++ ) {
         std::ostringstream term_name;
         term_name << s_mtty.str() << "_" << tid;
         vect_names.push_back(term_name.str().c_str());
      }
      mtty = new VciMultiTty<vci_param_int>(
            s_mtty.str().c_str(),
            IntTab(cid, INT_MTTY_TGT_ID),
            params.mt_int,
            vect_names);
   }

   ///////////   XICU
   std::ostringstream s_xicu;
   s_xicu << "xicu_" << params.x_id << "_" << params.y_id;
   xicu = new VciXicu<vci_param_int>(
         s_xicu.str().c_str(),
         params.mt_int,
         IntTab(cid, INT_XICU_TGT_ID),
         32, 32, 32,
         NB_PROCS);

   ////////////  MDMA
   std::ostringstream s_mdma;
   s_mdma << "mdma_" << params.x_id << "_" << params.y_id;
   mdma = new VciMultiDma<vci_param_int>(
         s_mdma.str().c_str(),
         params.mt_int,
         IntTab(cid, NB_PROCS),
         IntTab(cid, INT_MDMA_TGT_ID),
         64,
         NB_DMA_CHANNELS);

   ///////////  Direct LOCAL_XBAR(S)
   size_t nb_direct_initiators = NB_PROCS + 1;
   size_t nb_direct_targets    = 4;
   if (NB_DEBUG_TTY_CHANNELS) {
      nb_direct_targets++;
   }
   if ( is_io_cluster ) {
      nb_direct_initiators++;
      nb_direct_targets++;
   }

   std::ostringstream s_int_xbar_d;
   s_int_xbar_d << "int_xbar_cmd_d_" << params.x_id << "_" << params.y_id;
   int_xbar_d = new VciLocalCrossbar<vci_param_int>(
         s_int_xbar_d.str().c_str(),
         params.mt_int,
         cid,
         nb_direct_initiators,
         nb_direct_targets,
         0 );

   std::ostringstream s_int_dspin_ini_wrapper_gate_d;
   s_int_dspin_ini_wrapper_gate_d 
      << "int_dspin_ini_wrapper_gate_d_" << params.x_id << "_" << params.y_id;
   int_wi_gate_d = new VciIntDspinInitiatorWrapperType(
         s_int_dspin_ini_wrapper_gate_d.str().c_str(),
         X_WIDTH + Y_WIDTH + l_width);

   std::ostringstream s_int_dspin_tgt_wrapper_gate_d;
   s_int_dspin_tgt_wrapper_gate_d
      << "int_dspin_tgt_wrapper_gate_d_" << params.x_id << "_" << params.y_id;
   int_wt_gate_d = new VciIntDspinTargetWrapperType(
         s_int_dspin_tgt_wrapper_gate_d.str().c_str(),
         X_WIDTH + Y_WIDTH + l_width);

   ////////////  Coherence LOCAL_XBAR(S)
   std::ostringstream s_int_xbar_m2p_c;
   s_int_xbar_m2p_c << "int_xbar_m2p_c_" << params.x_id << "_" << params.y_id;
   int_xbar_m2p_c = new DspinLocalCrossbar<dspin_int_cmd_width>(
         s_int_xbar_m2p_c.str().c_str(),
         params.mt_int,
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH, l_width,
         1,
         NB_PROCS,
         2, 2,
         true,
         false,
         true);

   std::ostringstream s_int_xbar_p2m_c;
   s_int_xbar_p2m_c << "int_xbar_p2m_c_" << params.x_id << "_" << params.y_id;
   int_xbar_p2m_c = new DspinLocalCrossbar<dspin_int_rsp_width>(
         s_int_xbar_p2m_c.str().c_str(),
         params.mt_int,
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH, 0,
         NB_PROCS,
         1,
         2, 2,
         false,
         false,
         false);

   std::ostringstream s_int_xbar_clack_c;
   s_int_xbar_clack_c << "int_xbar_clack_c_" << params.x_id << "_"
                                             << params.y_id;
   int_xbar_clack_c = new DspinLocalCrossbar<dspin_int_cmd_width>(
         s_int_xbar_clack_c.str().c_str(),
         params.mt_int,
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH, l_width,
         1,
         NB_PROCS,
         1, 1,
         true,
         false,
         false);

   //////////////  INT ROUTER(S)
   std::ostringstream s_int_router_cmd;
   s_int_router_cmd << "router_cmd_" << params.x_id << "_" << params.y_id;
   int_router_cmd = new VirtualDspinRouter<dspin_int_cmd_width>(
         s_int_router_cmd.str().c_str(),
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH,
         3,
         4,4);

   std::ostringstream s_int_router_rsp;
   s_int_router_rsp << "router_rsp_" << params.x_id << "_" << params.y_id;
   int_router_rsp = new VirtualDspinRouter<dspin_int_rsp_width>(
         s_int_router_rsp.str().c_str(),
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH,
         2,
         4,4);

   //////////////  XRAM
   std::ostringstream s_xram;
   s_xram << "xram_" << params.x_id << "_" << params.y_id;
   xram = new VciSimpleRam<vci_param_ext>(
         s_xram.str().c_str(),
         IntTab(cid, RAM_XRAM_TGT_ID),
         params.mt_ext,
         params.loader,
         params.xram_latency);

   std::ostringstream s_wt_xram;
   s_wt_xram << "xram_wt_" << params.x_id << "_" << params.y_id;
   xram_ram_wt = new VciExtDspinTargetWrapperType(
         s_wt_xram.str().c_str(),
         vci_param_int::S);

   /////////////  RAM ROUTER(S)
   std::ostringstream s_ram_router_cmd;
   s_ram_router_cmd << "ram_router_cmd_" << params.x_id << "_" << params.y_id;
   ram_router_cmd = new DspinRouterTsar<dspin_ram_cmd_width>(
         s_ram_router_cmd.str().c_str(),
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH,
         4, 4,
         is_iob0, is_iob1,
         false,
         l_width);

   std::ostringstream s_ram_router_rsp;
   s_ram_router_rsp << "ram_router_rsp_" << params.x_id << "_" << params.y_id;
   ram_router_rsp = new DspinRouterTsar<dspin_ram_rsp_width>(
         s_ram_router_rsp.str().c_str(),
         params.x_id, params.y_id,
         X_WIDTH, Y_WIDTH,
         4, 4,
         is_iob0, is_iob1,
         true,
         l_width);

   ////////////////////// I/O  CLUSTER ONLY ///////////////////////
   iob        = NULL;
   iob_ram_wi = NULL;
   if ( is_io_cluster ) {
      ///////////  IO_BRIDGE
      size_t iox_local_id;
      bool   has_irqs;
      if ( is_iob0 ) {
         iox_local_id = 0;
         has_irqs     = true;
      }
      else {
         iox_local_id = 1;
         has_irqs     = false;
      }

      std::ostringstream s_iob;
      s_iob << "iob_" << params.x_id << "_" << params.y_id;   
      iob = new VciIoBridge<vci_param_int, vci_param_ext>(  
            s_iob.str().c_str(),
            params.mt_ext,
            params.mt_int,
            params.mt_iox,
            IntTab(cid, INT_IOBX_TGT_ID),
            IntTab(cid, INT_IOBX_INI_ID),
            IntTab(cid, iox_local_id ),
            has_irqs,
            16,
            8,
            8,
            params.debug_start_cycle,
            params.iob_debug_ok );

      std::ostringstream s_iob_ram_wi;
      s_iob_ram_wi << "iob_ram_wi_" << params.x_id << "_" << params.y_id;   
      iob_ram_wi = new VciExtDspinInitiatorWrapperType(
            s_iob_ram_wi.str().c_str(),
            vci_param_int::S);
   }

   ////////////////////////////////////
   // Connections are defined here
   ////////////////////////////////////

   // on coherence network : local srcid[proc] in [0...NB_PROCS-1]
   //                      : local srcid[memc] = NB_PROCS
   // In cluster_iob0, 32 HWI interrupts from external peripherals
   // are connected to the XICU ports p_hwi[0:31] 
   // In other clusters, no HWI interrupts are connected to XICU

   //////////////////////// internal CMD & RSP routers
   int_router_cmd->p_clk    (this->p_clk);
   int_router_cmd->p_resetn (this->p_resetn);
   int_router_rsp->p_clk    (this->p_clk);
   int_router_rsp->p_resetn (this->p_resetn);

   for (int i = 0; i < 4; i++) {
      for(int k = 0; k < 3; k++) {
         int_router_cmd->p_out[i][k] (this->p_dspin_int_cmd_out[i][k]);
         int_router_cmd->p_in[i][k]  (this->p_dspin_int_cmd_in[i][k]);
      }
      for(int k = 0; k < 2; k++) {
         int_router_rsp->p_out[i][k] (this->p_dspin_int_rsp_out[i][k]);
         int_router_rsp->p_in[i][k]  (this->p_dspin_int_rsp_in[i][k]);
      }
   }

   // local ports
   int_router_cmd->p_out[4][0] (signal_int_dspin_cmd_g2l_d);
   int_router_cmd->p_out[4][1] (signal_int_dspin_m2p_g2l_c);
   int_router_cmd->p_out[4][2] (signal_int_dspin_clack_g2l_c);
   int_router_cmd->p_in[4][0]  (signal_int_dspin_cmd_l2g_d);
   int_router_cmd->p_in[4][1]  (signal_int_dspin_m2p_l2g_c);
   int_router_cmd->p_in[4][2]  (signal_int_dspin_clack_l2g_c);

   int_router_rsp->p_out[4][0] (signal_int_dspin_rsp_g2l_d);
   int_router_rsp->p_out[4][1] (signal_int_dspin_p2m_g2l_c);
   int_router_rsp->p_in[4][0]  (signal_int_dspin_rsp_l2g_d);
   int_router_rsp->p_in[4][1]  (signal_int_dspin_p2m_l2g_c);

   ///////////////////// CMD & RSP VCI local crossbar direct
   int_xbar_d->p_clk             (this->p_clk);
   int_xbar_d->p_resetn          (this->p_resetn);
   int_xbar_d->p_initiator_to_up (signal_int_vci_l2g);
   int_xbar_d->p_target_to_up    (signal_int_vci_g2l);

   int_xbar_d->p_to_target[INT_MEMC_TGT_ID] (signal_int_vci_tgt_memc);
   int_xbar_d->p_to_target[INT_XICU_TGT_ID] (signal_int_vci_tgt_xicu);
   int_xbar_d->p_to_target[INT_BROM_TGT_ID] (signal_int_vci_tgt_brom);
   int_xbar_d->p_to_target[INT_MDMA_TGT_ID] (signal_int_vci_tgt_mdma);
   if (NB_DEBUG_TTY_CHANNELS) {
      int_xbar_d->p_to_target[INT_MTTY_TGT_ID] (signal_int_vci_tgt_mtty);
   }
   int_xbar_d->p_to_initiator[INT_MDMA_INI_ID] (signal_int_vci_ini_mdma);
   for (size_t p = 0; p < NB_PROCS; p++) {
      int_xbar_d->p_to_initiator[INT_PROC_INI_ID + p](
            signal_int_vci_ini_proc[p]);
   }

   if ( is_io_cluster ) {
      int_xbar_d->p_to_target[INT_IOBX_TGT_ID]    (signal_int_vci_tgt_iobx);
      int_xbar_d->p_to_initiator[INT_IOBX_INI_ID] (signal_int_vci_ini_iobx);
   }

   int_wi_gate_d->p_clk               (this->p_clk);
   int_wi_gate_d->p_resetn            (this->p_resetn);
   int_wi_gate_d->p_vci               (signal_int_vci_l2g);
   int_wi_gate_d->p_dspin_cmd         (signal_int_dspin_cmd_l2g_d);
   int_wi_gate_d->p_dspin_rsp         (signal_int_dspin_rsp_g2l_d);

   int_wt_gate_d->p_clk               (this->p_clk);
   int_wt_gate_d->p_resetn            (this->p_resetn);
   int_wt_gate_d->p_vci               (signal_int_vci_g2l);
   int_wt_gate_d->p_dspin_cmd         (signal_int_dspin_cmd_g2l_d);
   int_wt_gate_d->p_dspin_rsp         (signal_int_dspin_rsp_l2g_d);
   
   ////////////////////// M2P DSPIN local crossbar coherence
   int_xbar_m2p_c->p_clk              (this->p_clk);
   int_xbar_m2p_c->p_resetn           (this->p_resetn);
   int_xbar_m2p_c->p_global_out       (signal_int_dspin_m2p_l2g_c);
   int_xbar_m2p_c->p_global_in        (signal_int_dspin_m2p_g2l_c);
   int_xbar_m2p_c->p_local_in[0]      (signal_int_dspin_m2p_memc);
   for (size_t p = 0; p < NB_PROCS; p++) {
      int_xbar_m2p_c->p_local_out[p] (signal_int_dspin_m2p_proc[p]);
   }

   ////////////////////////// P2M DSPIN local crossbar coherence
   int_xbar_p2m_c->p_clk             (this->p_clk);
   int_xbar_p2m_c->p_resetn          (this->p_resetn);
   int_xbar_p2m_c->p_global_out      (signal_int_dspin_p2m_l2g_c);
   int_xbar_p2m_c->p_global_in       (signal_int_dspin_p2m_g2l_c);
   int_xbar_p2m_c->p_local_out[0]    (signal_int_dspin_p2m_memc);
   for (size_t p = 0; p < NB_PROCS; p++) {
      int_xbar_p2m_c->p_local_in[p]   (signal_int_dspin_p2m_proc[p]);
   }
                                      
   ////////////////////// CLACK DSPIN  local crossbar coherence
   int_xbar_clack_c->p_clk             (this->p_clk);
   int_xbar_clack_c->p_resetn          (this->p_resetn);
   int_xbar_clack_c->p_global_out      (signal_int_dspin_clack_l2g_c);
   int_xbar_clack_c->p_global_in       (signal_int_dspin_clack_g2l_c);
   int_xbar_clack_c->p_local_in[0]     (signal_int_dspin_clack_memc);
   for (size_t p = 0; p < NB_PROCS; p++) {
      int_xbar_clack_c->p_local_out[p] (signal_int_dspin_clack_proc[p]);
   }

   //////////////////////////////////// Processors
   for (size_t p = 0; p < NB_PROCS; p++) {
      proc[p]->p_clk          (this->p_clk);
      proc[p]->p_resetn       (this->p_resetn);
      proc[p]->p_vci          (signal_int_vci_ini_proc[p]);
      proc[p]->p_dspin_m2p    (signal_int_dspin_m2p_proc[p]);
      proc[p]->p_dspin_p2m    (signal_int_dspin_p2m_proc[p]);
      proc[p]->p_dspin_clack  (signal_int_dspin_clack_proc[p]);
      proc[p]->p_irq[0]       (signal_proc_it[p]);
      for ( size_t j = 1 ; j < 6 ; j++) {
         proc[p]->p_irq[j]    (signal_false);
      }
   }

   ///////////////////////////////////// XICU
   xicu->p_clk    (this->p_clk);
   xicu->p_resetn (this->p_resetn);
   xicu->p_vci    (signal_int_vci_tgt_xicu);
   for ( size_t p = 0 ; p < NB_PROCS ; p++) {
      xicu->p_irq[p] (signal_proc_it[p]);
   }
   for ( size_t i=0 ; i<32 ; i++) {
      if ( is_iob0 ) 
         xicu->p_hwi[i] (*(this->p_irq[i]));
      else  
         xicu->p_hwi[i] (signal_false);
   }                     

   ///////////////////////////////////// MEMC
   memc->p_clk              (this->p_clk);
   memc->p_resetn           (this->p_resetn);
   memc->p_vci_ixr          (signal_ram_vci_ini_memc);
   memc->p_vci_tgt          (signal_int_vci_tgt_memc);
   memc->p_dspin_p2m        (signal_int_dspin_p2m_memc);
   memc->p_dspin_m2p        (signal_int_dspin_m2p_memc);
   memc->p_dspin_clack      (signal_int_dspin_clack_memc);
   memc->p_irq              (signal_irq_memc);

   // wrapper to RAM network
   memc_ram_wi->p_clk       (this->p_clk);
   memc_ram_wi->p_resetn    (this->p_resetn);
   memc_ram_wi->p_dspin_cmd (signal_ram_dspin_cmd_memc_i);
   memc_ram_wi->p_dspin_rsp (signal_ram_dspin_rsp_memc_i);
   memc_ram_wi->p_vci       (signal_ram_vci_ini_memc);

   //////////////////////////////////// BROM
   brom->p_clk              (this->p_clk);
   brom->p_resetn           (this->p_resetn);
   brom->p_vci              (signal_int_vci_tgt_brom);

   if (NB_DEBUG_TTY_CHANNELS) {
      //////////////////////////////////// MTTY
      mtty->p_clk              (this->p_clk);
      mtty->p_resetn           (this->p_resetn);
      mtty->p_vci              (signal_int_vci_tgt_mtty);

      for ( size_t i=0 ; i < NB_DEBUG_TTY_CHANNELS ; i++ ) {
         mtty->p_irq[i] (signal_irq_mtty[i]);
      }
   }

   //////////////////////////////////// XRAM
   xram->p_clk              (this->p_clk);
   xram->p_resetn           (this->p_resetn);
   xram->p_vci              (signal_ram_vci_tgt_xram);

   // wrapper to RAM network
   xram_ram_wt->p_clk       (this->p_clk);
   xram_ram_wt->p_resetn    (this->p_resetn);
   xram_ram_wt->p_dspin_cmd (signal_ram_dspin_cmd_xram_t);
   xram_ram_wt->p_dspin_rsp (signal_ram_dspin_rsp_xram_t);
   xram_ram_wt->p_vci       (signal_ram_vci_tgt_xram);

   /////////////////////////////////// MDMA
   mdma->p_clk              (this->p_clk);
   mdma->p_resetn           (this->p_resetn);
   mdma->p_vci_target       (signal_int_vci_tgt_mdma);
   mdma->p_vci_initiator    (signal_int_vci_ini_mdma);
   for (size_t i = 0 ; i < NB_DMA_CHANNELS ; i++)
      mdma->p_irq[i]        (signal_irq_mdma[i]);

   //////////////////////////// RAM network CMD & RSP routers
   ram_router_cmd->p_clk       (this->p_clk);
   ram_router_cmd->p_resetn    (this->p_resetn);
   ram_router_rsp->p_clk       (this->p_clk);
   ram_router_rsp->p_resetn    (this->p_resetn);
   for( size_t n=0 ; n<4 ; n++) {
      ram_router_cmd->p_out[n] (this->p_dspin_ram_cmd_out[n]);
      ram_router_cmd->p_in[n]  (this->p_dspin_ram_cmd_in[n]);
      ram_router_rsp->p_out[n] (this->p_dspin_ram_rsp_out[n]);
      ram_router_rsp->p_in[n]  (this->p_dspin_ram_rsp_in[n]);
   }
   ram_router_cmd->p_out[4]    (signal_ram_dspin_cmd_xram_t);
   ram_router_cmd->p_in[4]     (signal_ram_dspin_cmd_memc_i);
   ram_router_rsp->p_out[4]    (signal_ram_dspin_rsp_memc_i);
   ram_router_rsp->p_in[4]     (signal_ram_dspin_rsp_xram_t);

   ///////////////////////// IOB exists only in cluster_iob0 & cluster_iob1. 
   if ( is_io_cluster ) {
      // IO bridge
      iob->p_clk         (this->p_clk);
      iob->p_resetn      (this->p_resetn);
      iob->p_vci_ini_iox (*(this->p_vci_iob_iox_ini));
      iob->p_vci_tgt_iox (*(this->p_vci_iob_iox_tgt));
      iob->p_vci_tgt_int (signal_int_vci_tgt_iobx);
      iob->p_vci_ini_int (signal_int_vci_ini_iobx);
      iob->p_vci_ini_ram (signal_ram_vci_ini_iobx);

      if ( is_iob0 ) {
         for ( size_t n = 0 ; n < 32 ; n++ ) {
            (*iob->p_irq[n]) (*(this->p_irq[n]));
         }
      }

      // initiator wrapper to RAM network
      iob_ram_wi->p_clk       (this->p_clk);
      iob_ram_wi->p_resetn    (this->p_resetn);
      iob_ram_wi->p_dspin_cmd (*(this->p_dspin_iob_cmd_out));
      iob_ram_wi->p_dspin_rsp (*(this->p_dspin_iob_rsp_in));
      iob_ram_wi->p_vci       (signal_ram_vci_ini_iobx);
   }
} // end constructor

tmpl(/**/)::~TsarIobCluster() {
   if (p_vci_iob_iox_ini)   delete p_vci_iob_iox_ini;
   if (p_vci_iob_iox_tgt)   delete p_vci_iob_iox_tgt;
   if (p_dspin_iob_cmd_out) delete p_dspin_iob_cmd_out;
   if (p_dspin_iob_rsp_in)  delete p_dspin_iob_rsp_in;
   if (iob)                 delete iob;
   if (iob_ram_wi)          delete iob_ram_wi;

   for (size_t n = 0 ; n < 32 ; n++) {
      if (p_irq[n]) delete p_irq[n];
   }

   for (size_t p = 0; p < NB_PROCS; p++) { 
      delete proc[p];
   }

   delete memc;
   delete memc_ram_wi;
   delete xicu;
   delete brom;
   delete mtty;
   delete mdma;
   delete int_xbar_d;
   delete int_xbar_m2p_c;
   delete int_xbar_p2m_c;
   delete int_xbar_clack_c;
   delete int_router_cmd;
   delete int_router_rsp;
   delete xram;
   delete xram_ram_wt;
   delete ram_router_cmd;
   delete ram_router_rsp;
}

}}


// Local Variables:
// tab-width: 3
// c-basic-offset: 3
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=3:tabstop=3:softtabstop=3

