//////////////////////////////////////////////////////////////////////////////
// File: tsar_iob_cluster.h
// Author: Alain Greiner
// Copyright: UPMC/LIP6
// Date : april 2013
// This program is released under the GNU public license
//////////////////////////////////////////////////////////////////////////////

#ifndef SOCLIB_CABA_TSAR_IOB_CLUSTER_H
#define SOCLIB_CABA_TSAR_IOB_CLUSTER_H

#include <systemc>
#include <sys/time.h>
#include <iostream>
#include <sstream>
#include <cstdlib>
#include <cstdarg>

#include "gdbserver.h"
#include "mapping_table.h"
#include "mips32.h"
#include "vci_simple_ram.h"
#include "vci_simple_rom.h"
#include "vci_xicu.h"
#include "vci_local_crossbar.h"
#include "dspin_local_crossbar.h"
#include "vci_dspin_initiator_wrapper.h"
#include "vci_dspin_target_wrapper.h"
#include "dspin_router_tsar.h"
#include "virtual_dspin_router.h"
#include "vci_multi_dma.h"
#include "vci_mem_cache.h"
#include "vci_cc_vcache_wrapper.h"
#include "vci_io_bridge.h"
#include "vci_multi_tty.h"
#include "hard_config.h"

///////////////////////////////////////////////////////////////////////
//     Number of channels for debug TTY (may be 0)
///////////////////////////////////////////////////////////////////////
#define NB_DEBUG_TTY_CHANNELS 1

///////////////////////////////////////////////////////////////////////
//     TGT_ID and INI_ID port indexing for INT local interconnect
///////////////////////////////////////////////////////////////////////

#define INT_MEMC_TGT_ID 0
#define INT_XICU_TGT_ID 1
#define INT_BROM_TGT_ID 2
#define INT_MDMA_TGT_ID 3
#define INT_MTTY_TGT_ID 4
#define INT_IOBX_TGT_ID (4 + (NB_DEBUG_TTY_CHANNELS ? 1 : 0))

#define INT_PROC_INI_ID 0 // from 0 to 7
#define INT_MDMA_INI_ID NB_PROCS
#define INT_IOBX_INI_ID (NB_PROCS + 1)

///////////////////////////////////////////////////////////////////////
//     TGT_ID and INI_ID port indexing for RAM local interconnect
///////////////////////////////////////////////////////////////////////

#define RAM_XRAM_TGT_ID 0

#define RAM_MEMC_INI_ID 0
#define RAM_IOBX_INI_ID 1

namespace soclib { namespace caba {

///////////////////////////////////////////////////////////////////////////
template<typename vci_param_int,
         typename vci_param_ext,
         size_t   dspin_int_cmd_width,
         size_t   dspin_int_rsp_width,
         size_t   dspin_ram_cmd_width,
         size_t   dspin_ram_rsp_width>
class TsarIobCluster
///////////////////////////////////////////////////////////////////////////
    : public soclib::caba::BaseModule
{

   public:

      // Ports
      sc_in<bool>   p_clk;
      sc_in<bool>   p_resetn;

      // Thes two ports are used to connect IOB to IOX nework in top cell
      soclib::caba::VciInitiator<vci_param_ext>* p_vci_iob_iox_ini;
      soclib::caba::VciTarget<vci_param_ext>*    p_vci_iob_iox_tgt;

      // These ports are used to connect IOB to RAM network in top cell
      soclib::caba::DspinOutput<dspin_ram_cmd_width>* p_dspin_iob_cmd_out;
      soclib::caba::DspinInput<dspin_ram_rsp_width>*  p_dspin_iob_rsp_in;

      // These ports are used to connect hard IRQ from external peripherals to
      // IOB0
      sc_in<bool>* p_irq[32];

      // These arrays of ports are used to connect the INT & RAM networks in
      // top cell
      soclib::caba::DspinOutput<dspin_int_cmd_width>** p_dspin_int_cmd_out;
      soclib::caba::DspinInput<dspin_int_cmd_width>**  p_dspin_int_cmd_in;
      soclib::caba::DspinOutput<dspin_int_rsp_width>** p_dspin_int_rsp_out;
      soclib::caba::DspinInput<dspin_int_rsp_width>**  p_dspin_int_rsp_in;

      soclib::caba::DspinOutput<dspin_ram_cmd_width>* p_dspin_ram_cmd_out;
      soclib::caba::DspinInput<dspin_ram_cmd_width>*  p_dspin_ram_cmd_in;
      soclib::caba::DspinOutput<dspin_ram_rsp_width>* p_dspin_ram_rsp_out;
      soclib::caba::DspinInput<dspin_ram_rsp_width>*  p_dspin_ram_rsp_in;

      // interrupt signals
      sc_signal<bool> signal_false;
      sc_signal<bool> signal_proc_it[8];
      sc_signal<bool> signal_irq_mdma[8];
      sc_signal<bool> signal_irq_mtty[8];
      sc_signal<bool> signal_irq_memc;

      // INT network DSPIN signals between DSPIN routers and DSPIN
      // local_crossbars
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_cmd_l2g_d;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_cmd_g2l_d;
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_rsp_l2g_d;
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_rsp_g2l_d;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_m2p_l2g_c;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_m2p_g2l_c;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_clack_l2g_c;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_clack_g2l_c;
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_p2m_l2g_c;
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_p2m_g2l_c;

      // INT network VCI signals between VCI components and VCI local crossbar
      VciSignals<vci_param_int> signal_int_vci_ini_proc[8];
      VciSignals<vci_param_int> signal_int_vci_ini_mdma;
      VciSignals<vci_param_int> signal_int_vci_ini_iobx;

      VciSignals<vci_param_int> signal_int_vci_tgt_memc;
      VciSignals<vci_param_int> signal_int_vci_tgt_xicu;
      VciSignals<vci_param_int> signal_int_vci_tgt_brom;
      VciSignals<vci_param_int> signal_int_vci_tgt_mtty;
      VciSignals<vci_param_int> signal_int_vci_tgt_mdma;
      VciSignals<vci_param_int> signal_int_vci_tgt_iobx;

      VciSignals<vci_param_int> signal_int_vci_l2g;
      VciSignals<vci_param_int> signal_int_vci_g2l;

      // Coherence DSPIN signals between DSPIN local crossbars and CC
      // components
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_m2p_memc;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_clack_memc;
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_p2m_memc;
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_m2p_proc[8];
      DspinSignals<dspin_int_cmd_width> signal_int_dspin_clack_proc[8];
      DspinSignals<dspin_int_rsp_width> signal_int_dspin_p2m_proc[8];

      // RAM network VCI signals between VCI components and VCI/DSPIN wrappers
      VciSignals<vci_param_ext> signal_ram_vci_ini_memc;
      VciSignals<vci_param_ext> signal_ram_vci_ini_iobx;
      VciSignals<vci_param_ext> signal_ram_vci_tgt_xram;

      // RAM network DSPIN signals between VCI/DSPIN wrappers and routers
      DspinSignals<dspin_ram_cmd_width> signal_ram_dspin_cmd_xram_t;
      DspinSignals<dspin_ram_rsp_width> signal_ram_dspin_rsp_xram_t;
      DspinSignals<dspin_ram_cmd_width> signal_ram_dspin_cmd_memc_i;
      DspinSignals<dspin_ram_rsp_width> signal_ram_dspin_rsp_memc_i;

      //////////////////////////////////////
      // Hardwate Components (pointers)
      //////////////////////////////////////
      typedef VciCcVCacheWrapper<vci_param_int, dspin_int_cmd_width,
              dspin_int_rsp_width, GdbServer<Mips32ElIss> >
              VciCcVCacheWrapperType;

      typedef VciMemCache<vci_param_int, vci_param_ext, dspin_int_rsp_width,
              dspin_int_cmd_width> VciMemCacheType;

      typedef VciDspinInitiatorWrapper<vci_param_int, dspin_int_cmd_width,
              dspin_int_rsp_width> VciIntDspinInitiatorWrapperType;

      typedef VciDspinTargetWrapper<vci_param_int, dspin_int_cmd_width,
              dspin_int_rsp_width> VciIntDspinTargetWrapperType;

      typedef VciDspinInitiatorWrapper<vci_param_ext, dspin_ram_cmd_width,
              dspin_ram_rsp_width> VciExtDspinInitiatorWrapperType;

      typedef VciDspinTargetWrapper<vci_param_ext, dspin_ram_cmd_width,
              dspin_ram_rsp_width> VciExtDspinTargetWrapperType;

      VciCcVCacheWrapperType*          proc[8];

      VciMemCacheType*                 memc;
      VciExtDspinInitiatorWrapperType* memc_ram_wi;

      VciXicu<vci_param_int>*          xicu;

      VciMultiDma<vci_param_int>*      mdma;

      VciSimpleRom<vci_param_int>*     brom;

      VciMultiTty<vci_param_int>*      mtty;

      VciLocalCrossbar<vci_param_int>*  int_xbar_d;
      VciIntDspinInitiatorWrapperType*  int_wi_gate_d;
      VciIntDspinTargetWrapperType*     int_wt_gate_d;
      
      DspinLocalCrossbar<dspin_int_cmd_width>* int_xbar_m2p_c;
      DspinLocalCrossbar<dspin_int_rsp_width>* int_xbar_p2m_c;
      DspinLocalCrossbar<dspin_int_cmd_width>* int_xbar_clack_c;

      VirtualDspinRouter<dspin_int_cmd_width>* int_router_cmd;
      VirtualDspinRouter<dspin_int_rsp_width>* int_router_rsp;

      VciSimpleRam<vci_param_ext>*  xram;
      VciExtDspinTargetWrapperType* xram_ram_wt;

      DspinRouterTsar<dspin_ram_cmd_width>* ram_router_cmd;
      DspinRouterTsar<dspin_ram_rsp_width>* ram_router_rsp;

      // IO Network Components (not instanciated in all clusters)

      VciIoBridge<vci_param_int, vci_param_ext>* iob;
      VciExtDspinInitiatorWrapperType*           iob_ram_wi;

      size_t m_procs;

      struct ClusterParams {
         sc_module_name insname;

         size_t x_id;
         size_t y_id;

         const soclib::common::MappingTable &mt_int;
         const soclib::common::MappingTable &mt_ext;
         const soclib::common::MappingTable &mt_iox;

         size_t memc_ways;
         size_t memc_sets;
         size_t l1_i_ways;
         size_t l1_i_sets;
         size_t l1_d_ways;
         size_t l1_d_sets;
         size_t xram_latency;

         const Loader& loader;

         uint32_t frozen_cycles;
         uint32_t debug_start_cycle;
         bool     memc_debug_ok;
         bool     proc_debug_ok;
         bool     iob_debug_ok;
      };

      // utility functions
      static uint32_t clusterId(size_t x_id, size_t y_id) {
         return ((x_id << Y_WIDTH) | y_id); 
      };

      // cluster constructor
      TsarIobCluster(struct ClusterParams& params);
      ~TsarIobCluster();
};

}}

#endif

// vim: filetype=cpp:expandtab:shiftwidth=3:tabstop=3:softtabstop=3
