/* -*- c++ -*-
 *
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Authors  : Cesar Armando Fuguet Tortolero 
 * Date     : jul 2015
 * Copyright: UPMC - LIP6
 */
#include <systemc>
#include <string>
#include "synthetic_dspin_network.h"

#if _OPENMP
#include <omp.h>
#endif

struct Args
{
    size_t x_size;
    size_t y_size;
    size_t load;
    size_t ncycles;
    bool faulty;
};


////////////////////////////////
// parse command line arguments
////////////////////////////////
void parse_args(Args *args, int argc, char **argv)
{
    for (int arg = 1; arg < argc; ++arg) {
        if ((strcmp(argv[arg], "-L") == 0) && ((arg + 1) < argc)) {
            args->load = strtol(argv[arg + 1], NULL, 0);
            arg++;
            continue;
        }
        if ((strcmp(argv[arg], "-X") == 0) && ((arg + 1) < argc)) {
            args->x_size = strtol(argv[arg + 1], NULL, 0);
            arg++;
            continue;
        }
        if ((strcmp(argv[arg], "-Y") == 0) && ((arg + 1) < argc)) {
            args->y_size = strtol(argv[arg + 1], NULL, 0);
            arg++;
            continue;
        }
        if ((strcmp(argv[arg], "-N") == 0) && ((arg + 1) < argc)) {
            args->ncycles = strtol(argv[arg + 1], NULL, 0);
            arg++;
            continue;
        }
        if (strcmp(argv[arg], "-F") == 0) {
            args->faulty = true;
            continue;
        }
    
        std::cout << "   Arguments are (key, value) couples." << std::endl;
        std::cout << "   The order is not important." << std::endl;
        std::cout << "   Accepted arguments are :" << std::endl << std::endl;
        std::cout << "     -L dspin generators' accepted load * 1000" << std::endl;
        std::cout << "     -X number of clusters per row" << std::endl;
        std::cout << "     -Y number of clusters per column" << std::endl;
        std::cout << "     -N simulation's cycles" << std::endl;
        std::cout << "     -F deactivate a router" << std::endl;
    }
}                        // end parse_args()


int sc_main(int argc, char **argv)
{
    using namespace soclib::caba;

#if _OPENMP
    omp_set_dynamic(false);
    omp_set_num_threads(1);
#endif


    ////////////////////////////////
    // parse command line arguments
    ////////////////////////////////
    Args args = {
        .x_size  = 5,
        .y_size  = 5,
        .load    = 10,
        .ncycles = 100000,
        .faulty  = false
    };
    parse_args(&args, argc, argv);


    ////////////////////////////
    // components instantiation
    ////////////////////////////
    sc_core::sc_clock signal_clk;
    sc_core::sc_signal<bool> signal_resetn;
    SyntheticDspinNetwork syntheticDspinNetwork("syntheticDspinNetwork",
                                                args.x_size,
                                                args.y_size,
                                                args.load);


    ///////////
    // netlist
    ///////////
    syntheticDspinNetwork.p_clk(signal_clk);
    syntheticDspinNetwork.p_resetn(signal_resetn);


    ////////////////////
    // start simulation
    ////////////////////
    sc_core::sc_start(sc_core::SC_ZERO_TIME);
    signal_resetn.write(0);
    syntheticDspinNetwork.reset();

    if (args.faulty) {
        // deactivate a router in the center of the mesh and create its
        // recovery cycle-free contour
        syntheticDspinNetwork.set_faulty_router(args.x_size / 2, args.y_size / 2);
    }

    sc_core::sc_start(1, sc_core::SC_NS);
    signal_resetn.write(1);
    sc_core::sc_start(args.ncycles, sc_core::SC_NS);


    ////////////////////
    // print statistics
    ////////////////////
    for (size_t x = 0; x < args.x_size; ++x) {
        for (size_t y = 0; y < args.y_size; ++y) {
            syntheticDspinNetwork.print_stats(x, y);
        }
    }

  
    return 0;
}                   // end sc_main()

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
