/**
 * @author  Cesar Armando Fuguet Tortolero
 * @date    24 May, 2015
 * @brief   This platform allows the validation of the reconfigurable routing
 *          algorithm on the DSPIN router component. The platform has been
 *          specifically designed to test the routing of broadcast packets.
 */
#include <iostream>
#include <systemc>
#include <cassert>
#include <cstdlib>

#include "dspin_router.h"
#include "dspin_router_config.h"
#include "dspin_packet_generator.h"
#include "alloc_elems.h"

#if _OPENMP
#include <omp.h>
#endif

/*
 * Platform constant parameters
 */
#define X_WIDTH 4
#define Y_WIDTH 4

#define FIFO_DEPTH 2
#define NFLITS 8
#define LOAD 1000
#define BROADCAST_PERIOD 1
#define DSPIN_WIDTH 39
#define MAX_PACKETS 100

/*
 * Platform default values
 */
#define X_SIZE 5
#define Y_SIZE 5

static inline int cluster(int x, int y)
{
    return (x << Y_WIDTH) | y;
}

static inline uint32_t configRouter(int reallocation_dir,
                                    int recovery_mode,
                                    int blackhole_pos)
{
    return ((reallocation_dir & 0x7) << 5) |
           ((recovery_mode    & 0x1) << 4) |
           (blackhole_pos     & 0xF);
}

int sc_main(int argc, char **argv)
{
    using namespace soclib::caba;
    using namespace soclib::common;

    typedef DspinPacketGenerator<DSPIN_WIDTH, DSPIN_WIDTH>
        DspinGeneratorType;
    typedef DspinRouter<DSPIN_WIDTH>
        DspinRouterType;
    typedef DspinSignals<DSPIN_WIDTH>
        DspinSignalType;

#if _OPENMP
    omp_set_dynamic(false);
    omp_set_num_threads(1);
#endif

    /* mesh size */
    int xSize = X_SIZE;
    int ySize = Y_SIZE;

    /* (x,y) coordinates of the initiator router */
    int xSrc = -1;
    int ySrc = -1;

    /* (x,y) coordinates of the faulty router */
    int xFaulty = -1;
    int yFaulty = -1;

    /* simulation cycles */
    size_t simCycles = 0;

    /* debug */
    bool debug = false;

    /* synthetic generator load */
    int load = LOAD;

    /* broadcast period */
    int bcp = BROADCAST_PERIOD;

    /* maximum number of packets per generator */
    size_t max = MAX_PACKETS;

    for (int n = 1; n < argc; n += 2) {
        if ((strcmp(argv[n], "-X") == 0) && ((n + 1) < argc)) {
            xSize = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-Y") == 0) && ((n + 1) < argc) ) {
            ySize = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-FX") == 0) && ((n + 1) < argc)) {
            xFaulty = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-FY") == 0) && ((n + 1) < argc) ) {
            yFaulty = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-SX") == 0) && ((n + 1) < argc)) {
            xSrc = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-SY") == 0) && ((n + 1) < argc) ) {
            ySrc = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-N") == 0) && ((n + 1) < argc) ) {
            simCycles = strtol(argv[n + 1], NULL, 0);
            assert(simCycles > 0);
            continue;
        }
        if ((strcmp(argv[n], "-L") == 0) && ((n + 1) < argc) ) {
            load = strtol(argv[n + 1], NULL, 0);
            assert(load > 0);
            continue;
        }
        if ((strcmp(argv[n], "-B") == 0) && ((n + 1) < argc) ) {
            bcp = strtol(argv[n + 1], NULL, 0);
            assert(bcp >= 0);
            continue;
        }
        if ((strcmp(argv[n], "-P") == 0) && ((n + 1) < argc) ) {
            max = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n--], "-G") == 0)) {
            debug = true;
            continue;
        }
    }

    assert (xFaulty < xSize );
    assert (yFaulty < ySize );
    assert (xSrc < xSize );
    assert (ySrc < ySize );

    DspinGeneratorType ***dspinGenerator = new DspinGeneratorType**[xSize];
    DspinRouterType ***dspinRouter = new DspinRouterType**[xSize];
    for (int x = 0; x < xSize; ++x) {
        dspinGenerator[x] = new DspinGeneratorType*[ySize];
        dspinRouter[x] = new DspinRouterType*[ySize];
        for (int y = 0; y < ySize; ++y) {
            const bool BROADCAST_SUPPORTED = true;
            const bool CONFIGURATION_SUPPORTED = true;
            std::ostringstream routerStr;
            routerStr << "dspinRouter["<< x << "][" << y << "]";
            dspinRouter[x][y] =
                new DspinRouterType(routerStr.str().c_str(), x, y,
                                    X_WIDTH, Y_WIDTH,
                                    FIFO_DEPTH, FIFO_DEPTH,
                                    BROADCAST_SUPPORTED,
                                    CONFIGURATION_SUPPORTED);

            if ((x == xFaulty) && (y == yFaulty)) {
                dspinRouter[x][y]->set_disable_mask(0x1F);
            }

            int ld = 0;
            const int SRCID = cluster(x,y);
            bool all = (xSrc == -1) || (ySrc == -1);
            if (all || (cluster(x,y) == cluster(xSrc,ySrc))) {
                ld = load;
            }
            if ((x == xFaulty) && (y == yFaulty)) {
                ld = 0;
            }
            if (simCycles > 0) {
                max = 0;
            }
            std::ostringstream generatorStr;
            generatorStr << "dspinGenerator["<< x << "][" << y << "]";
            dspinGenerator[x][y] =
                new DspinGeneratorType(generatorStr.str().c_str(),
                                       SRCID, NFLITS,
                                       ld, 8,
                                       bcp,
                                       X_WIDTH, Y_WIDTH,
                                       X_SIZE, Y_SIZE,
                                       max);
        }
    }

    const int H = xSize - 1;
    const int Y = ySize - 1;
    sc_clock signal_clk("clk");
    sc_core::sc_signal<bool> signal_resetn("signal_resetn");
    DspinSignalType*** sDspinL =
        alloc_elems<DspinSignalType>("sDspinL", xSize, ySize, 2);
    DspinSignalType*** sDspinH =
        alloc_elems<DspinSignalType>("sDspinH", H + 2, ySize, 2);
    DspinSignalType*** sDspinV =
        alloc_elems<DspinSignalType>("sDspinV", xSize, Y + 2, 2);
    sc_signal<uint32_t> sConfigNONE("sConfigNONE");
    sc_signal<uint32_t> sConfigN("sConfigN");
    sc_signal<uint32_t> sConfigNW("sConfigNW");
    sc_signal<uint32_t> sConfigNE("sConfigNE");
    sc_signal<uint32_t> sConfigS("sConfigS");
    sc_signal<uint32_t> sConfigSW("sConfigSW");
    sc_signal<uint32_t> sConfigSE("sConfigSE");
    sc_signal<uint32_t> sConfigW("sConfigW");
    sc_signal<uint32_t> sConfigE("sConfigE");
    for (int x = 0; x < xSize; ++x) {
        for (int y = 0; y < ySize; ++y) {
            dspinGenerator[x][y]->p_clk(signal_clk);
            dspinGenerator[x][y]->p_resetn(signal_resetn);
            dspinGenerator[x][y]->p_out(sDspinL[x][y][0]);
            dspinGenerator[x][y]->p_in(sDspinL[x][y][1]);

            dspinRouter[x][y]->p_clk(signal_clk);
            dspinRouter[x][y]->p_resetn(signal_resetn);
            dspinRouter[x][y]->p_in[0](sDspinV[x][y + 1][1]);
            dspinRouter[x][y]->p_out[0](sDspinV[x][y + 1][0]);
            dspinRouter[x][y]->p_in[1](sDspinV[x][y][0]);
            dspinRouter[x][y]->p_out[1](sDspinV[x][y][1]);
            dspinRouter[x][y]->p_in[2](sDspinH[x + 1][y][1]);
            dspinRouter[x][y]->p_out[2](sDspinH[x + 1][y][0]);
            dspinRouter[x][y]->p_in[3](sDspinH[x][y][0]);
            dspinRouter[x][y]->p_out[3](sDspinH[x][y][1]);
            dspinRouter[x][y]->p_in[4](sDspinL[x][y][0]);
            dspinRouter[x][y]->p_out[4](sDspinL[x][y][1]);

            if ((xFaulty < 0) || (yFaulty < 0)) {
                dspinRouter[x][y]->bind_recovery_port(sConfigNONE);
                continue;
            }

            if (x == (xFaulty + 1)) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigNE);
                    std::cout << "config NE" << std::endl;
                    continue;
                }
                if (y == yFaulty) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigE);
                    std::cout << "config E" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigSE);
                    std::cout << "config SE" << std::endl;
                    continue;
                }
            }
            if (x == xFaulty) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigN);
                    std::cout << "config N" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigS);
                    std::cout << "config S" << std::endl;
                    continue;
                }
            }
            if (x == (xFaulty - 1)) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigNW);
                    std::cout << "config NW" << std::endl;
                    continue;
                }
                if (y == yFaulty) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigW);
                    std::cout << "config W" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigSW);
                    std::cout << "config SW" << std::endl;
                    continue;
                }
            }
            dspinRouter[x][y]->bind_recovery_port(sConfigNONE);
        }
    }

    srandom(3);
    sc_start(sc_core::SC_ZERO_TIME);
    signal_resetn = 0;

    /* initialize the configuration signals */
    sConfigNONE.write(configRouter(0, 0, NORMAL));

    // requests to the deactivated segment are dropped
    sConfigN.write(configRouter(REQ_SOUTH, 1, N_OF_X));
    sConfigNE.write(configRouter(REQ_WEST, 1, NE_OF_X));
    sConfigE.write(configRouter(REQ_WEST, 1, E_OF_X));
    sConfigSE.write(configRouter(REQ_WEST, 1, SE_OF_X));
    sConfigS.write(configRouter(REQ_NORTH, 1, S_OF_X));
    sConfigSW.write(configRouter(REQ_EAST, 1, SW_OF_X));
    sConfigW.write(configRouter(REQ_EAST, 1, W_OF_X));
    sConfigNW.write(configRouter(REQ_EAST, 1, NW_OF_X));

    /* initialize mesh boundary signals */
    for (int x = 0; x < xSize; ++x) {
        sDspinV[x][0][0].write = false;
        sDspinV[x][0][1].read = true;
        sDspinV[x][ySize][0].read = true;
        sDspinV[x][ySize][1].write = false;
    }
    for (int y = 0; y < ySize; ++y) {
        sDspinH[0][y][0].write = false;
        sDspinH[0][y][1].read = true;
        sDspinH[xSize][y][0].read = true;
        sDspinH[xSize][y][1].write = false;
    }

    sc_start(sc_core::sc_time(5, SC_NS));
    signal_resetn = 1;

    size_t n;
    if (debug) {
        for(n = 0; n < simCycles; ++n) {
            std::cout << std::endl;
            std::cout << "##########################################" << std::endl;
            std::cout << "Simulation cycle " << n << std::endl;
            std::cout << "##########################################" << std::endl;
            std::cout << std::endl;
            sc_start(sc_core::sc_time(1, SC_NS));
            for (int x = 0; x < xSize; ++x) {
                for (int y = 0; y < ySize; ++y) {
                    dspinRouter[x][y]->print_trace();
                    dspinGenerator[x][y]->print_trace();
                }
            }
        }
    } else {
        if (simCycles > 0) {
            sc_start(sc_core::sc_time(simCycles, SC_NS));
        }
        else {
            const int period = 3000;
            for (n = 0; n < (max * 200); n += period) {
                if ((xSrc == -1) || (ySrc == -1)) {
                    std::cout << "When using a packet limit, there must be one ";
                    std::cout << "source only" << std::endl;
                    std::exit(1);
                }

                sc_start(sc_core::sc_time(period, SC_NS));

                uint32_t pkts = dspinGenerator[xSrc][ySrc]->get_sent_packets();
                if (pkts >= max) break;
            }
            sc_start(sc_core::sc_time(period, SC_NS));
            std::cout << "Simulated cycles: " << n + period << std::endl;
        }
    }

    for (int x = 0; x < xSize; ++x) {
        for (int y = 0; y < ySize; ++y) {
            dspinGenerator[x][y]->print_stats();
        }
    }

    return 0;
}

/*
 * vim: ts=4 : sw=4 : sts=4 : et
 */
