/**
 * @author  Cesar Armando Fuguet Tortolero
 * @date    24 May, 2015
 * @brief   This platform allows the validation of the reconfigurable routing
 *          algorithm on the DSPIN router component. The platform has been
 *          specifically designed to test the routing of broadcast packets.
 */
#include <iostream>
#include <systemc>
#include <cassert>

#include "dspin_router.h"
#include "dspin_router_config.h"
#include "dspin_packet_generator.h"
#include "alloc_elems.h"

#if _OPENMP
#include <omp.h>
#endif

/*
 * Platform constant parameters
 */
#define X_WIDTH 4
#define Y_WIDTH 4

#define FIFO_DEPTH 8
#define NFLITS 2
#define LOAD 1000
#define BROADCAST_PERIOD 3
#define DSPIN_WIDTH 39

/*
 * Platform default values
 */
#define X_SIZE 4
#define Y_SIZE 4

static inline int cluster(int x, int y)
{
    return (x << Y_WIDTH) | y;
}

static inline uint32_t configRouter(int bypass_mode,
                                    int reallocation_dir,
                                    int blackhole_pos)
{
    return (bypass_mode << 7) | (reallocation_dir << 4) | blackhole_pos;
}

int sc_main(int argc, char **argv)
{
    using namespace soclib::caba;
    using namespace soclib::common;

    typedef DspinPacketGenerator<DSPIN_WIDTH, DSPIN_WIDTH>
        DspinGeneratorType;
    typedef DspinRouter<DSPIN_WIDTH>
        DspinRouterType;
    typedef DspinSignals<DSPIN_WIDTH>
        DspinSignalType;

#if _OPENMP
    omp_set_dynamic(false);
    omp_set_num_threads(1);
#endif

    /* mesh size */
    int xSize = X_SIZE;
    int ySize = Y_SIZE;

    /* (x,y) coordinates of the initiator router */
    int xSrc = -1;
    int ySrc = -1;

    /* (x,y) coordinates of the faulty router */
    int xFaulty = -1;
    int yFaulty = -1;

    /* enable the DSPIN router's debug */
    int debug = false;

    /* number of simulation cycles */
    int simCycles = 100000;
    for (int n = 1; n < argc; n = n + 2) {
        if ((strcmp(argv[n], "-X") == 0) && ((n + 1) < argc)) {
            xSize = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-Y") == 0) && ((n + 1) < argc) ) {
            ySize = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-FX") == 0) && ((n + 1) < argc)) {
            xFaulty = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-FY") == 0) && ((n + 1) < argc) ) {
            yFaulty = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-SX") == 0) && ((n + 1) < argc)) {
            xSrc = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-SY") == 0) && ((n + 1) < argc) ) {
            ySrc = strtol(argv[n + 1], NULL, 0);
            continue;
        }
        if ((strcmp(argv[n], "-N") == 0) && ((n + 1) < argc) ) {
            simCycles = strtol(argv[n + 1], NULL, 0);
            assert(simCycles > 0);
            continue;
        }
        if ((strcmp(argv[n], "-DEBUG") == 0)) {
            debug = true;
            continue;
        }
    }

    assert (xFaulty < xSize );
    assert (yFaulty < ySize );
    assert (xSrc < xSize );
    assert (ySrc < ySize );

    DspinGeneratorType ***dspinGenerator = new DspinGeneratorType**[xSize];
    DspinRouterType ***dspinRouter = new DspinRouterType**[xSize];
    for (int x = 0; x < xSize; ++x) {
        dspinGenerator[x] = new DspinGeneratorType*[ySize];
        dspinRouter[x] = new DspinRouterType*[ySize];
        for (int y = 0; y < ySize; ++y) {
            const bool BROADCAST_SUPPORTED = true;
            const bool CONFIGURATION_SUPPORTED = true;
            std::ostringstream routerStr;
            routerStr << "dspinRouter["<< x << "][" << y << "]";
            dspinRouter[x][y] =
                new DspinRouterType(routerStr.str().c_str(), x, y,
                                    X_WIDTH, Y_WIDTH,
                                    FIFO_DEPTH, FIFO_DEPTH,
                                    BROADCAST_SUPPORTED,
                                    CONFIGURATION_SUPPORTED);

            if ((x == xFaulty) && (y == yFaulty)) {
                dspinRouter[x][y]->set_disable_mask(0x1F);
            }

            int broadcast_period = 0;
            int load = 0;
            const int SRCID = cluster(x,y);
            bool all = (xSrc == -1) && (ySrc == -1);
            if (all || (cluster(x,y) == cluster(xSrc,ySrc))) {
               broadcast_period = BROADCAST_PERIOD;
               load = LOAD;
            }
            std::ostringstream generatorStr;
            generatorStr << "dspinGenerator["<< x << "][" << y << "]";
            dspinGenerator[x][y] =
                new DspinGeneratorType(generatorStr.str().c_str(),
                                       SRCID, NFLITS,
                                       load, FIFO_DEPTH,
                                       broadcast_period);
        }
    }

    const int H = xSize - 1;
    const int Y = ySize - 1;
    sc_clock signal_clk("clk");
    sc_core::sc_signal<bool> signal_resetn("signal_resetn");
    DspinSignalType*** sDspinL =
        alloc_elems<DspinSignalType>("sDspinL", xSize, ySize, 2);
    DspinSignalType*** sDspinH =
        alloc_elems<DspinSignalType>("sDspinH", H + 2, ySize, 2);
    DspinSignalType*** sDspinV =
        alloc_elems<DspinSignalType>("sDspinV", xSize, Y + 2, 2);
    sc_signal<uint32_t> sConfigNONE("sConfigNONE");
    sc_signal<uint32_t> sConfigN("sConfigN");
    sc_signal<uint32_t> sConfigNW("sConfigNW");
    sc_signal<uint32_t> sConfigNE("sConfigNE");
    sc_signal<uint32_t> sConfigS("sConfigS");
    sc_signal<uint32_t> sConfigSW("sConfigSW");
    sc_signal<uint32_t> sConfigSE("sConfigSE");
    sc_signal<uint32_t> sConfigW("sConfigW");
    sc_signal<uint32_t> sConfigE("sConfigE");
    for (int x = 0; x < xSize; ++x) {
        for (int y = 0; y < ySize; ++y) {
            dspinGenerator[x][y]->p_clk(signal_clk);
            dspinGenerator[x][y]->p_resetn(signal_resetn);
            dspinGenerator[x][y]->p_out(sDspinL[x][y][0]);
            dspinGenerator[x][y]->p_in(sDspinL[x][y][1]);

            dspinRouter[x][y]->p_clk(signal_clk);
            dspinRouter[x][y]->p_resetn(signal_resetn);
            dspinRouter[x][y]->p_in[0](sDspinV[x][y + 1][1]);
            dspinRouter[x][y]->p_out[0](sDspinV[x][y + 1][0]);
            dspinRouter[x][y]->p_in[1](sDspinV[x][y][0]);
            dspinRouter[x][y]->p_out[1](sDspinV[x][y][1]);
            dspinRouter[x][y]->p_in[2](sDspinH[x + 1][y][1]);
            dspinRouter[x][y]->p_out[2](sDspinH[x + 1][y][0]);
            dspinRouter[x][y]->p_in[3](sDspinH[x][y][0]);
            dspinRouter[x][y]->p_out[3](sDspinH[x][y][1]);
            dspinRouter[x][y]->p_in[4](sDspinL[x][y][0]);
            dspinRouter[x][y]->p_out[4](sDspinL[x][y][1]);

            if (x == (xFaulty + 1)) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigNE);
                    std::cout << "config NE" << std::endl;
                    continue;
                }
                if (y == yFaulty) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigE);
                    std::cout << "config E" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigSE);
                    std::cout << "config SE" << std::endl;
                    continue;
                }
            }
            if (x == xFaulty) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigN);
                    std::cout << "config N" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigS);
                    std::cout << "config S" << std::endl;
                    continue;
                }
            }
            if (x == (xFaulty - 1)) {
                if (y == (yFaulty + 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigNW);
                    std::cout << "config NW" << std::endl;
                    continue;
                }
                if (y == yFaulty) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigW);
                    std::cout << "config W" << std::endl;
                    continue;
                }
                if (y == (yFaulty - 1)) {
                    dspinRouter[x][y]->bind_recovery_port(sConfigSW);
                    std::cout << "config SW" << std::endl;
                    continue;
                }
            }
            dspinRouter[x][y]->bind_recovery_port(sConfigNONE);
        }
    }

    sc_start(sc_core::SC_ZERO_TIME);
    signal_resetn = 0;

    /* initialize the configuration signals */
    sConfigNONE.write(configRouter(0, REQ_NOP, BH_NONE));
    sConfigN.write(configRouter(1, REQ_SOUTH, BH_N));
    sConfigNE.write(configRouter(1, REQ_WEST, BH_NE));
    sConfigE.write(configRouter(1, REQ_WEST, BH_E));
    sConfigSE.write(configRouter(1, REQ_WEST, BH_SE));
    sConfigS.write(configRouter(1, REQ_NORTH, BH_S));
    sConfigSW.write(configRouter(1, REQ_EAST, BH_SW));
    sConfigW.write(configRouter(1, REQ_EAST, BH_W));
    sConfigNW.write(configRouter(1, REQ_EAST, BH_NW));

    /* initialize mesh boundary signals */
    for (int x = 0; x < xSize; ++x) {
        sDspinV[x][0][0].write = false;
        sDspinV[x][0][1].read = true;
        sDspinV[x][ySize][0].read = true;
        sDspinV[x][ySize][1].write = false;
    }
    for (int y = 0; y < ySize; ++y) {
        sDspinH[0][y][0].write = false;
        sDspinH[0][y][1].read = true;
        sDspinH[xSize][y][0].read = true;
        sDspinH[xSize][y][1].write = false;
    }

    sc_start(sc_core::sc_time(5, SC_NS));
    signal_resetn = 1;

    for(int i = 0; i < simCycles; ++i) {
        if (!debug) {
            sc_start(sc_core::sc_time(simCycles, SC_NS));
            break;
        }
        std::cout << std::endl;
        std::cout << "##########################################" << std::endl;
        std::cout << "Simulation cycle " << i << std::endl;
        std::cout << "##########################################" << std::endl;
        std::cout << std::endl;
        sc_start(sc_core::sc_time(1, SC_NS));
        for (int x = 0; x < xSize; ++x) {
            for (int y = 0; y < ySize; ++y) {
                dspinRouter[x][y]->print_trace();
            }
        }
    }
    for (int x = 0; x < xSize; ++x) {
        for (int y = 0; y < ySize; ++y) {
            dspinGenerator[x][y]->print_stats();
        }
    }

    return 0;
}

/*
 * vim: ts=4 : sw=4 : sts=4 : et
 */
