#!/usr/bin/python

import onerun
import os
import random
from math import ceil
from sys import exit

class RunArguments:
    def __init__(self, name, x, y):
        self.path = os.getcwd()
        self.outpath = name
        self.x = x
        self.y = y
        self.nprocs = 4
        self.compileonly = False
        self.batchmode = True
        self.faultyrouter = []
        self.faultymask = 0xF
        self.faultycore = []
        self.debug = None

        self.xmax = self.x - 1
        self.ymax = self.y - 1
        self.pmax = self.nprocs - 1

        # init random number generator
        random.seed()

    def add_faultyrouter(self, x):
        if x in self.faultyrouter:
            return False

        self.faultyrouter.append(x)
        return True

    def add_faultycore(self, x):
        if x in self.faultycore:
            return False

        self.faultycore.append(x)
        return True

    def init_faultyrouter(self, x):
        if x == None: return
        self.faultyrouter = x

    def init_faultycore(self, x):
        if x == None: return
        self.faultycore = x

    def get_faultycount(self, total, pct):
        """
            Returns the number of faulty elements in a population
            Arguments:
                total -- Total number of elements
                pct -- Percentage of faulty elements
        """
        return int(ceil(total * float(pct)/100))

    def get_faultycorecount(self, pct):
        """
            Returns the number of faulty cores in the platform
            Arguments:
                pct -- Percentage of faulty cores
        """
        return self.get_faultycount(self.get_totalcores(), pct)

    def get_faultyroutercount(self, pct):
        """
            Returns the number of faulty routers in the platform
            Arguments:
                pct -- Percentage of faulty routers
        """
        return self.get_faultycount(self.get_totalclusters(), pct)

    def get_totalclusters(self):
        return self.x * self.y

    def get_totalcores(self):
        return self.get_totalclusters() * self.nprocs

for xsize, ysize in ((3, 3), (4, 4), (4, 8), (8, 8), (8, 16), (16, 16)):
    # --------------------------------------------------------------------
    # simulation without faults
    # --------------------------------------------------------------------
    cfgname = 'output_{0}_{1}'.format(xsize, ysize)
    args = RunArguments(cfgname, xsize, ysize)
    onerun.run(args)

    for pct in (10, 20, 30, 40, 50, 60):
        # --------------------------------------------------------------------
        # simulation with random faulty cores
        # --------------------------------------------------------------------
        cfgname = 'output_core_{0}_{1}_{2}'.format(xsize, ysize, pct)
        args = RunArguments(cfgname, xsize, ysize)

        faultycount = args.get_faultycorecount(pct)
        print "{0}/{1} faulty cores\n".\
            format(faultycount, args.get_totalcores())

        n = 0
        while n < faultycount:
            cx = random.randint(0, args.xmax)
            cy = random.randint(0, args.ymax)
            cl = random.randint(0, args.pmax)
            if args.add_faultycore((cx, cy, cl)): n += 1

        onerun.run(args)

        # --------------------------------------------------------------------
        # simulation with random faulty routers
        # --------------------------------------------------------------------
        cfgname = 'output_router_{0}_{1}_{2}'.format(xsize, ysize, pct)
        args = RunArguments(cfgname, xsize, ysize)

        faultycount = args.get_faultyroutercount(pct)
        print "{0}/{1} faulty routers\n".\
            format(faultycount, args.get_totalclusters())

        n = 0
        while n < faultycount:
            cx = random.randint(0, args.xmax)
            cy = random.randint(0, args.ymax)
            if args.add_faultyrouter((cx, cy)): n += 1

        onerun.run(args)

        # --------------------------------------------------------------------
        # simulation with random faulty routers and cores
        # --------------------------------------------------------------------
        cfgname = 'output_mixed_{0}_{1}_{2}'.format(xsize, ysize, pct)
        args = RunArguments(cfgname, xsize, ysize)

        total = args.get_totalclusters() + args.get_totalcores()
        faultycount = args.get_faultycount(total, pct)
        print "{0}/{1} faulty routers/cores\n".\
                format(faultycount, total)

        n = 0
        while n < faultycount:
            cx = random.randint(0, args.xmax)
            cy = random.randint(0, args.ymax)
            if (random.random() < 0.05):    # faulty router: 5% of probability
                if not args.add_faultyrouter((cx, cy)):
                    continue

            else:                           # faulty core: 95% of probability
                cl = random.randint(0, args.pmax)
                if not args.add_faultycore((cx, cy, cl)):
                    continue

            n += 1

        onerun.run(args)

