/**
 * \file    io.h
 * \date    5 September 2012
 * \author  Cesar Fuguet
 *
 * Utility functions to write or read memory mapped hardware registers
 */
#ifndef IO_H
#define IO_H

#include <cpu.h>

/**
 * \brief Read an 32 bits memory mapped hardware register
 * 
 * \note interrupts should be disabled while executing this function
 */
static inline unsigned int ioread32(uint64_t paddr)
{
    uint32_t value;
    uint32_t msb = paddr >> CLUSTER_OFFSET_BITS;
    uint32_t lsb = paddr & ((1ULL << CLUSTER_OFFSET_BITS)-1);

    asm volatile("mfc2   $8,     $24   \n"  /* t0 <= DATA_PADDR_EXT  */
                 "mtc2   %2,     $24   \n"  /* DATA_PADDR_EXT <= msb */
                 "lw     %0,     0(%1) \n"  /* value <= *paddr       */
                 "mtc2   $8,     $24   \n"  /* DATA_PADDR_EXT <= 0   */
                 : "=r"(value)
                 : "r"(lsb), "r"(msb)
                 : "$8", "memory");

    return value;
}

/**
 * \brief Write an 32 bits memory mapped hardware register
 *
 * \note interrupts should be disabled while executing this function
 */
static inline void iowrite32(uint64_t paddr, uint32_t value)
{
    uint32_t msb = paddr >> CLUSTER_OFFSET_BITS;
    uint32_t lsb = paddr & ((1ULL << CLUSTER_OFFSET_BITS)-1);

    asm volatile("mfc2   $8,     $24   \n"  /* t0 <= DATA_PADDR_EXT  */
                 "mtc2   %2,     $24   \n"  /* DATA_PADDR_EXT <= msb */
                 "sw     %0,     0(%1) \n"  /* value <= *paddr       */
                 "mtc2   $8,     $24   \n"  /* DATA_PADDR_EXT <= 0   */
                 "sync                 \n"
                 : /* no outputs */
                 : "r"(value), "r"(lsb), "r"(msb)
                 : "$8", "memory");
}

#endif

/*
 * vim: tabstop=4 : softtabstop=4 : shiftwidth=4 : expandtab
 */
