/**
 * \file    stdio.c
 * \date    July 8, 2014
 * \author  Cesar Fuguet
 *
 * \brief   input/output functions
 */
#include <stdio.h>
#include <stdint.h>
#include <tty.h>
#include <cpu.h>

static const char HexaTab[] = "0123456789ABCDEF";

void puts(const char *buffer)
{
    sputs(buffer, 256);
}

int getc(char *c)
{
    return tty_getc(c);
}

void putc(const char c)
{
    if (c == '\n') {
        tty_putc('\r');
    }
    tty_putc(c);
}

void sputs(const char *buffer, unsigned int length)
{
    unsigned int n;
    for (n = 0; n < length; n++) {
        if (buffer[n] == 0) {
            break;
        }
        putc(buffer[n]);
    }
}

void putx(unsigned int val)
{
    char buf[9];
    unsigned int c;

    buf[8] = 0;
    for (c = 0; c < 8; c++) {
        buf[7 - c] = HexaTab[val & 0xF];
        val >>= 4;
    }
    puts(buf);
}

void putd(unsigned int val)
{
    char buf[11];
    unsigned int i;
    unsigned int first = 0;

    buf[10] = 0;
    for (i = 0; i < 10; i++) {
        if ((val == 0) && (i != 0)) {
            break;
        }
        buf[9 - i] = HexaTab[val % 10];
        first = 9 - i;
        val /= 10;
    }
    puts(&buf[first]);
}

void printf(const char *format, ...)
{
    va_list ap;
    va_start(ap, format);

    while (1) {
        while (*format) {
            unsigned int i;
            for (i = 0; format[i] && (format[i] != '%'); i++) ;
            if (i) {
                sputs(format, i);
                format += i;
            }
            if (*format == '%') {
                format++;
                goto printf_arguments;
            }
        }
        va_end(ap);
        return;

      printf_arguments:
        switch (*format++) {
        case 'c':              /* char conversion */
            {
                char arg = (char)va_arg(ap, int);
                putc(arg);
                break;
            }

        case 'd':              /* 32 bits decimal signed  */
            {
                int arg = va_arg(ap, int);
                if (arg < 0) {
                    putc('-');
                    arg = -arg;
                }
                putd(arg);
                break;
            }

        case 'u':              /* 32 bits decimal unsigned  */
            {
                unsigned int arg = va_arg(ap, unsigned int);
                putd(arg);
                break;
            }

        case 'x':              /* 32 bits hexadecimal unsigned */
            {
                unsigned int arg = va_arg(ap, unsigned int);
                puts("0x");
                putx(arg);
                break;
            }

        case 'l':              /* 64 bits hexadecimal unsigned */
            {
                uint64_t val = va_arg(ap, uint64_t);
                puts("0x");
                putx(val >> 32);
                putx(val & ((1ULL << 32) - 1));
                break;
            }

        case 's':              /* string */
            {
                char *arg = va_arg(ap, char *);
                puts(arg);
                break;
            }

        default:
            goto return_error;
        }                       /* end switch (format) */
    }                           /* end while(1) */

  return_error:
    puts("\n\n[ERROR] in printf()\n");
    exit();
}                               /* end printf */

/**
 * exit function
 */
void exit()
{
    cpu_wait();
}

/*
 * vim: tabstop=4 : softtabstop=4 : shiftwidth=4 : expandtab
 */
