#!/usr/bin/env python

import onerun
import os
import random
from math import ceil
from sys import exit

class RunArguments:
    def __init__(self, name, x, y):
        self.path = os.getcwd()
        self.outpath = name
        self.x = x
        self.y = y
        self.nprocs = 4
        self.compileonly = False
        self.batchmode = True
        self.faultyrouter = []
        self.faultymask = 0xF
        self.faultycore = []
        self.debug = None

        self.xmax = self.x - 1
        self.ymax = self.y - 1
        self.pmax = self.nprocs - 1

        # init random number generator
        random.seed()

    def add_faultyrouter(self, x):
        if x in self.faultyrouter:
            return False

        self.faultyrouter.append(x)
        return True

    def add_faultycore(self, x):
        if x in self.faultycore:
            return False

        self.faultycore.append(x)
        return True

    def init_faultyrouter(self, x):
        if x == None: return
        self.faultyrouter = x

    def init_faultycore(self, x):
        if x == None: return
        self.faultycore = x

    def get_faultycount(self, total, pct):
        """
            Returns the number of faulty elements in a population
            Arguments:
                total -- Total number of elements
                pct -- Percentage of faulty elements
        """
        return int(ceil(total * float(pct)/100))

    def get_faultycorecount(self, pct):
        """
            Returns the number of faulty cores in the platform
            Arguments:
                pct -- Percentage of faulty cores
        """
        return self.get_faultycount(self.get_totalcores(), pct)

    def get_faultyroutercount(self, pct):
        """
            Returns the number of faulty routers in the platform
            Arguments:
                pct -- Percentage of faulty routers
        """
        return self.get_faultycount(self.get_totalclusters(), pct)

    def get_totalclusters(self):
        return self.x * self.y

    def get_totalcores(self):
        return self.get_totalclusters() * self.nprocs

# activate the different kinds of simulation
SIM_FAULTFREE = False
SIM_FAULTCORES = False
SIM_FAULTROUTERS = True
SIM_FAULTMIXED = False

# probability of faulty router (for mixed simul)
SIM_FAULTMIXED_PROBROUTER = 0.05 # 5%

# NoC index for faulty routers
CMD = 0
RSP = 1

for xsize, ysize in [(4, 4), (4, 8), (8, 8), (8, 16), (16, 16)]:
    if SIM_FAULTFREE:
        # ---------------------------------------------------------------------
        # simulation without faults
        # ---------------------------------------------------------------------
        cfgname = 'output_{0}_{1}'.format(xsize, ysize)
        args = RunArguments(cfgname, xsize, ysize)
        onerun.run(args)

    for pct in xrange(5, 60, 5): # 5, 10, ..., 55
        if SIM_FAULTCORES:
            # -----------------------------------------------------------------
            # simulation with random faulty cores
            # -----------------------------------------------------------------
            cfgname = 'output_core_{0}_{1}_{2}'.format(xsize, ysize, pct)
            args = RunArguments(cfgname, xsize, ysize)

            faultycount = args.get_faultycorecount(pct)
            print "{0}/{1} faulty cores\n".\
                  format(faultycount, args.get_totalcores())

            n = 0
            while n < faultycount:
                cx = random.randint(0, args.xmax)
                cy = random.randint(0, args.ymax)
                cl = random.randint(0, args.pmax)
                if args.add_faultycore((cx, cy, cl)): n += 1

            onerun.run(args)

        if SIM_FAULTROUTERS:
            # -----------------------------------------------------------------
            # simulation with random faulty routers
            # -----------------------------------------------------------------
            cfgname = 'output_router_{0}_{1}_{2}'.format(xsize, ysize, pct)
            args = RunArguments(cfgname, xsize, ysize)

            faultycount = args.get_faultyroutercount(pct)
            print "{0}/{1} faulty routers\n".\
                  format(faultycount, args.get_totalclusters())

            n = 0
            while n < faultycount:
                cx = random.randint(0, args.xmax)
                cy = random.randint(0, args.ymax)
                if args.add_faultyrouter((CMD, cx, cy)): n += 1

            onerun.run(args)

        if SIM_FAULTMIXED:
            # -----------------------------------------------------------------
            # simulation with random faulty routers and cores
            # -----------------------------------------------------------------
            cfgname = 'output_mixed_{0}_{1}_{2}'.format(xsize, ysize, pct)
            args = RunArguments(cfgname, xsize, ysize)

            total = args.get_totalclusters() + args.get_totalcores()
            faultycount = args.get_faultycount(total, pct)

            print "\nprobability of faulty routers: {0}".\
                  format(SIM_FAULTMIXED_PROBROUTER)
            fr = 0
            fc = 0
            while (fr + fc) < faultycount:
                cx = random.randint(0, args.xmax)
                cy = random.randint(0, args.ymax)
                if (random.random() < SIM_FAULTMIXED_PROBROUTER):
                    # add faulty router
                    fr += 1
                    while not args.add_faultyrouter((CMD, cx, cy)):
                        cx = random.randint(0, args.xmax)
                        cy = random.randint(0, args.ymax)

                else:
                    # add faulty core
                    fc += 1
                    cl = random.randint(0, args.pmax)
                    while not args.add_faultycore((cx, cy, cl)):
                        cx = random.randint(0, args.xmax)
                        cy = random.randint(0, args.ymax)
                        cl = random.randint(0, args.pmax)

            print "faulty routers / faulty cores = {0} / {1}".format(fr, fc)
            onerun.run(args)

# vim: ts=4 : sts=4 : sw=4 : et
