#!/usr/bin/env python

import dsx
import soclib
import soclib.component
import os

class Platform:
    def __init__(self, cluster_ctor, proc_count, binaries):
        self.pf = soclib.Architecture(cell_size = 4,
                                      plen_size = 8,
                                      addr_size = 32,
                                      rerror_size = 1,
                                      clen_size = 1,
                                      rflag_size = 1,
                                      srcid_size = 14,
                                      pktid_size = 4,
                                      trdid_size = 4,
                                      wrplen_size = 1
                                      )
        mt = self.pf.create('common:mapping_table', 'mtp',
                            addr_size = 32,
                            addr_bits = [8], srcid_bits = [8],
                            cacheability_mask = 0xff000000)
        self.mtp = mt

        # create a dictionary containing
        # loader = x, loader2 = y, ...
        bins = {}
        for i, bin in enumerate(binaries):
            k = 'binary'
            if i:
                k += str(i+1)
            bins[k] = bin
        self.loader = self.pf.create('common:loader', 'loader', **bins)

        self.ringp = self.pf.create('caba:vci_vgmn', 'ringp', 
                       mt = self.mtp,
                       min_latency = 1,
                       fifo_depth = 8, )

        self.cluster = cluster_ctor(self.pf,
                                    self.ringp,
                                    self.mtp,
                                    proc_count = proc_count,
                                    cluster_no = 0,
                                    cluster_base = 0x00000000,
                                    )

        self.proc_count = proc_count

    def simulator(self):
        self.pf.generate(soclib.PfDriver(outdir = 'cnc%d'%self.proc_count))

    def device_tree(self):
        from dsx.contrib.dts_platform.platform_to_dts import PlatformToDts
        import dsx.contrib.dts_platform.platform_handlers
        dtsgen = PlatformToDts(self.pf, self.mtp)
        return dtsgen.create_device_tree()

    def dts(self, output):
        from dsx.device_tree.dts import Driver
        dt = self.device_tree()
        dt.generate(Driver(
            outdir = '.',
            parent = None,
            output_file = output,
            ),)

    def dtb(self, output):
        from dsx.device_tree.dtb import Driver
        dt = self.device_tree()
        dt.generate(Driver(
            outdir = '.',
            parent = None,
            output_file = output,
            ),)

def main(cpus):
    from cnc_cluster import Cluster
    hard = Platform(Cluster, cpus, binaries = ["boot.bin", "kernel@0x00a00000:D"])
    hard.simulator()
    hard.dts(os.path.basename(__file__)[:-2]+'dts')
    hard.dtb(os.path.basename(__file__)[:-2]+'dtb')

if __name__ == '__main__':
    import sys
    cpus = 1
    if len(sys.argv) > 1:
        cpus = int(sys.argv[1])
    main(cpus)
