//////////////////////////////////////////////////////////////////////////////
// File: simple_cluster.cpp
// Author: Alain Greiner 
// Copyright: UPMC/LIP6
// Date : march 2013
// This program is released under the GNU public license
//////////////////////////////////////////////////////////////////////////////

#include "../include/simple_cluster.h"

namespace soclib {
namespace caba  {

//////////////////////////////////////////////////////////////////////////
//                 Constructor
//////////////////////////////////////////////////////////////////////////
template<typename vci_param, int cmd_width, int rsp_width>
SimpleCluster<vci_param, cmd_width, rsp_width>::SimpleCluster(
         sc_module_name    name,
         size_t            x_local,
         size_t            y_local,
         size_t            x_width,
         size_t            y_width,
         size_t            load_d,
         size_t            plen_d,
         size_t            load_ini_c,
         size_t            plen_ini_c,
         size_t            load_tgt_c,
         size_t            plen_tgt_c,
         size_t            bcp_ini,
         size_t            bcp_tgt )
            : soclib::caba::BaseModule(name),
            p_clk("clk"),
            p_resetn("resetn")
{
    // Vectors of ports definition
    p_cmd_in        = alloc_elems<DspinInput<cmd_width> >("p_cmd_in", 2, 4);
    p_cmd_out       = alloc_elems<DspinOutput<cmd_width> >("p_cmd_out", 2, 4);
    p_rsp_in        = alloc_elems<DspinInput<rsp_width> >("p_rsp_in", 2, 4);
    p_rsp_out       = alloc_elems<DspinOutput<rsp_width> >("p_rsp_out", 2, 4);

    // Empty mapping table (required by dspin_local_crossbar)
    MappingTable mt( 32, IntTab(1), IntTab (1), 0xFF000000);

    /////////////////////////////////////////
    //    Hardware Components definition 
    /////////////////////////////////////////

    // local crossbar CMD on coherence network
    std::ostringstream s2;
    s2 << "xbar_cmd_c_" << x_local << "_" << y_local;
    xbar_cmd_c = new DspinLocalCrossbar<cmd_width>(
                     s2.str().c_str(),
                     mt,                           // unused mapping table
                     x_local, y_local,             // cluster coordinates
                     x_width, y_width, 0,          // l_width == 0
                     1, 1,                         // one local source & one local dest
                     2, 2,                         // fifo depths  
                     false,                        // don't use local routing table 
                     true );                       // broacast supported

    std::cout << "- build xbar_cmd_c_" << x_local << "_" << y_local
              << " / flit_width = " << cmd_width << std::endl;

    // local crossbar RSP on coherence network
    std::ostringstream s3;
    s3 << "xbar_rsp_c_" << x_local << "_" << y_local;
    xbar_rsp_c = new DspinLocalCrossbar<rsp_width>(
                     s3.str().c_str(),
                     mt,                           // unused mapping table
                     x_local, y_local,             // cluster coordinates
                     x_width, y_width, 0,          // l_width == 0
                     1, 1,                         // one local source & one local dest
                     2, 2,                         // fifo depths  
                     false,                        // don't use local routing table 
                     true );                       // broadcast supported

    std::cout << "- build xbar_rsp_c_" << x_local << "_" << y_local
              << " / flit_width = " << rsp_width << std::endl;

    // local crossbar CMD on direct network
    std::ostringstream s0;
    s0 << "xbar_cmd_d_" << x_local << "_" << y_local;
    xbar_cmd_d = new DspinLocalCrossbar<cmd_width>(
                     s0.str().c_str(),
                     mt,                           // unused mapping table
                     x_local, y_local,             // cluster coordinates
                     x_width, y_width, 0,          // l_width = 0
                     1, 1,                         // one local source & one local dest
                     2, 2,                         // fifo depths  
                     false,                        // don't use local routing table 
                     false );                      // no broacast

    std::cout << "- build xbar_cmd_d_" << x_local << "_" << y_local 
              << " / flit_width = " << cmd_width << std::endl;


    // local crossbar RSP on direct network
    std::ostringstream s1;
    s1 << "xbar_rsp_d_" << x_local << "_" << y_local;
    xbar_rsp_d = new DspinLocalCrossbar<rsp_width>(
                     s1.str().c_str(),
                     mt,                           // unused mapping table
                     x_local, y_local,             // cluster coordinates
                     x_width, y_width, 0,          // l_width == 0
                     1, 1,                         // one local source & one local dest
                     2, 2,                         // fifo depths  
                     false,                        // don't use local routing table 
                     false );                      // no broacast

    std::cout << "- build xbar_rsp_d_" << x_local << "_" << y_local 
              << " / flit_width = " << rsp_width << std::endl;

    //  VCI Initiator on direct network   
    size_t srcid = (x_local << y_width) | y_local;
    std::ostringstream sid;
    sid << "ini_d_" << x_local << "_" << y_local;
    ini_d = new VciSyntheticInitiator<vci_param>(
                      sid.str().c_str(),
                      srcid,                 // VCI SRCID
                      plen_d,                // packet length
                      load_d,                // requested load
                      1024 );                // fifo depth

    std::ostringstream swid;
    swid << "w_ini_d_" << x_local << "_" << y_local;
    w_ini_d = new VciDspinInitiatorWrapper<vci_param,cmd_width,rsp_width>(
                     swid.str().c_str(),
                     x_width + y_width );

    std::cout << "- build ini_d_" << x_local << "_" << y_local << std::endl;

    //  VCI Target on direct network
    size_t tgtid = (x_local << y_width) | y_local;
    std::ostringstream std;
    std << "tgt_d_" << x_local << "_" << y_local;
    tgt_d = new VciSyntheticTarget<vci_param>(
                      std.str().c_str(),
                      tgtid,                 // MSB bits of VCI address
                      x_width,               // X field width
                      y_width,               // Y field width
                      0 );                   // L field width

    std::ostringstream swtd;
    swtd << "w_tgt_d_" << x_local << "_" << y_local;
    w_tgt_d = new VciDspinTargetWrapper<vci_param,cmd_width,rsp_width>(
                     swtd.str().c_str(),
                     x_width + y_width );

    std::cout << "- build tgt_d_" << x_local << "_" << y_local << std::endl;
    
    // DSPIN Sender on cc40 network / Receiver on cc33
    size_t s40_srcid = (x_local << y_width) | y_local;
    std::ostringstream s40;
    s40 << "ini_c_" << x_local << "_" << y_local;
    ini_c = new DspinPacketGenerator<cmd_width, rsp_width>(
                      s40.str().c_str(),
                      s40_srcid,            // ini component identifier
                      plen_ini_c,           // packet length (number of flits)
                      load_ini_c,           // requested load * 1000
                      1024,                 // fifo depth
                      bcp_ini );            // broadcast period

    std::cout << "- build ini_c_" << x_local << "_" << y_local << std::endl;

    //  DSPIN Sender on cc33 network / Receiver on cc40
    size_t s33_srcid = (x_local << y_width) | y_local;
    std::ostringstream s33;
    s33 << "tgt_c_" << x_local << "_" << y_local;
    tgt_c = new DspinPacketGenerator<rsp_width, cmd_width>(
                      s33.str().c_str(),
                      s33_srcid,            // ini component identifier
                      plen_tgt_c,           // packet length (number of flits)
                      load_tgt_c,           // requested load * 1000
                      1024,                 // fifo depth
                      bcp_tgt );            // broadcast period

    std::cout << "- build tgt_c_" << x_local << "_" << y_local << std::endl;

    // router CMD
    std::ostringstream rcmd;
    rcmd << "router_cmd_" << x_local << "_" << y_local;
    router_cmd = new VirtualDspinRouter<cmd_width>(
                     rcmd.str().c_str(),
                     x_local,y_local,              // coordinate in the mesh
                     x_width, y_width,             // x & y fields width
                     4,4);                         // input & output fifo depths

    std::cout << "- build router_cmd_" << x_local << "_" << y_local << std::endl;

    // router RSP
    std::ostringstream rrsp;
    rrsp << "router_rsp_" << x_local << "_" << y_local;
    router_rsp = new VirtualDspinRouter<rsp_width>(
                     rrsp.str().c_str(),
                     x_local,y_local,              // coordinate in the mesh
                     x_width, y_width,             // x & y fields width
                     4,4);                         // input & output fifo depths

    std::cout << "- build router_cmd_" << x_local << "_" << y_local << std::endl;

    ////////////////////////////////////
    // Connections are defined here
    ////////////////////////////////////

    // CMD ROUTER and RSP ROUTER
    router_cmd->p_clk                        (this->p_clk);
    router_cmd->p_resetn                     (this->p_resetn);
    router_rsp->p_clk                        (this->p_clk);
    router_rsp->p_resetn                     (this->p_resetn);

    for (int x = 0; x < 2; x++)
    {
        for(int y = 0; y < 4; y++)
        {
            router_cmd->p_out[x][y]          (this->p_cmd_out[x][y]);
            router_cmd->p_in[x][y]           (this->p_cmd_in[x][y]);
            router_rsp->p_out[x][y]          (this->p_rsp_out[x][y]);
            router_rsp->p_in[x][y]           (this->p_rsp_in[x][y]);
        }
    }

    router_cmd->p_out[0][4]                  (signal_dspin_cmd_g2l_d);
    router_cmd->p_out[1][4]                  (signal_dspin_cmd_g2l_c);
    router_cmd->p_in[0][4]                   (signal_dspin_cmd_l2g_d);
    router_cmd->p_in[1][4]                   (signal_dspin_cmd_l2g_c);

    router_rsp->p_out[0][4]                  (signal_dspin_rsp_g2l_d);
    router_rsp->p_out[1][4]                  (signal_dspin_rsp_g2l_c);
    router_rsp->p_in[0][4]                   (signal_dspin_rsp_l2g_d);
    router_rsp->p_in[1][4]                   (signal_dspin_rsp_l2g_c);

    std::cout << "- CMD & RSP routers connected" << std::endl;

    // CMD local crossbar direct
    xbar_cmd_d->p_clk                        (this->p_clk);
    xbar_cmd_d->p_resetn                     (this->p_resetn);
    xbar_cmd_d->p_global_out                 (signal_dspin_cmd_l2g_d);
    xbar_cmd_d->p_global_in                  (signal_dspin_cmd_g2l_d);
    xbar_cmd_d->p_local_out[0]               (signal_dspin_tgt_cmd_d);
    xbar_cmd_d->p_local_in[0]                (signal_dspin_ini_cmd_d);
        
    std::cout << "- CMD Direct crossbar connected" << std::endl;

    // RSP local crossbar direct
    xbar_rsp_d->p_clk                        (this->p_clk);
    xbar_rsp_d->p_resetn                     (this->p_resetn);
    xbar_rsp_d->p_global_out                 (signal_dspin_rsp_l2g_d);
    xbar_rsp_d->p_global_in                  (signal_dspin_rsp_g2l_d);
    xbar_rsp_d->p_local_in[0]                (signal_dspin_tgt_rsp_d);
    xbar_rsp_d->p_local_out[0]               (signal_dspin_ini_rsp_d);

    std::cout << "- RSP Direct crossbar connected" << std::endl;

    // CMD local crossbar coherence
    xbar_cmd_c->p_clk                        (this->p_clk);
    xbar_cmd_c->p_resetn                     (this->p_resetn);
    xbar_cmd_c->p_global_out                 (signal_dspin_cmd_l2g_c);
    xbar_cmd_c->p_global_in                  (signal_dspin_cmd_g2l_c);
    xbar_cmd_c->p_local_out[0]               (signal_dspin_tgt_cmd_c);
    xbar_cmd_c->p_local_in[0]                (signal_dspin_ini_cmd_c);
        
    std::cout << "- CMD coherence crossbar connected" << std::endl;

    // RSP local crossbar coherence
    xbar_rsp_c->p_clk                        (this->p_clk);
    xbar_rsp_c->p_resetn                     (this->p_resetn);
    xbar_rsp_c->p_global_out                 (signal_dspin_rsp_l2g_c);
    xbar_rsp_c->p_global_in                  (signal_dspin_rsp_g2l_c);
    xbar_rsp_c->p_local_in[0]                (signal_dspin_tgt_rsp_c);
    xbar_rsp_c->p_local_out[0]               (signal_dspin_ini_rsp_c);

    std::cout << "- RSP coherence crossbar connected" << std::endl;

    // VCI Initiator (and wrapper) on direct network
    ini_d->p_clk                             (this->p_clk);
    ini_d->p_resetn                          (this->p_resetn);
    ini_d->p_vci                             (signal_vci_ini);

    w_ini_d->p_clk                           (this->p_clk);
    w_ini_d->p_resetn                        (this->p_resetn);
    w_ini_d->p_dspin_cmd                     (signal_dspin_ini_cmd_d);
    w_ini_d->p_dspin_rsp                     (signal_dspin_ini_rsp_d);
    w_ini_d->p_vci                           (signal_vci_ini);

    std::cout << "- VCI initiators on direct network connected" << std::endl;

    // VCI Target (and wrapper) on direct network
    tgt_d->p_clk                             (this->p_clk);
    tgt_d->p_resetn                          (this->p_resetn);
    tgt_d->p_vci                             (signal_vci_tgt);

    w_tgt_d->p_clk                           (this->p_clk);
    w_tgt_d->p_resetn                        (this->p_resetn);
    w_tgt_d->p_dspin_cmd                     (signal_dspin_tgt_cmd_d);
    w_tgt_d->p_dspin_rsp                     (signal_dspin_tgt_rsp_d);
    w_tgt_d->p_vci                           (signal_vci_tgt);

    std::cout << "- VCI targets on direct network connected" << std::endl;

    // DSPIN initiators on coherence network
    ini_c->p_clk                             (this->p_clk);
    ini_c->p_resetn                          (this->p_resetn);
    ini_c->p_in                              (signal_dspin_ini_rsp_c);
    ini_c->p_out                             (signal_dspin_ini_cmd_c);

    std::cout << "- DSPIN initiators on coherence network connected" << std::endl;

    // DSPIN targets on coherence network
    tgt_c->p_clk                             (this->p_clk);
    tgt_c->p_resetn                          (this->p_resetn);
    tgt_c->p_in                              (signal_dspin_tgt_cmd_c);
    tgt_c->p_out                             (signal_dspin_tgt_rsp_c);

    std::cout << "- DSPIN targets on coherence network connected" << std::endl;

} // end constructor

}}    // end namespaces


// Local Variables:
// tab-width: 3
// c-basic-offset: 3
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=3:tabstop=3:softtabstop=3



