/* -*- c++ -*-
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Authors  : Abdelmalek SI MERABET
 *            Franck     WAJSBURT
 * Date     : Februrary 2013
 * Copyright: UPMC - LIP6
 */
#ifndef SOCLIB_CABA_DSPIN_ALLOC_RING_TARGET_FAST_H
#define SOCLIB_CABA_DSPIN_ALLOC_RING_TARGET_FAST_H

#include "generic_fifo.h"
#include "mapping_table.h"
#include "ring_signals_fast.h"
#include "dspin_interface.h"

#define T_DEBUG

namespace soclib { namespace caba {

namespace {

const char *ring_rsp_fsm_state_str_t[] = {
                "RSP_IDLE",
                "OWNER",
                "SENDING",
                "WAIT_PALLOC_END",
        };

#ifdef T_DEBUG

const char *ring_cmd_fsm_state_str_t[] = {
        "CMD_IDLE",
        "ALLOC",
        "NALLOC",
        "PALLOC2",
        "PALLOC1",
};
#endif
} // end namespace

template<typename vci_param, int ring_cmd_data_size, int ring_rsp_data_size>
class DspinLocalRingTargetFastC
{

typedef typename vci_param::fast_addr_t vci_addr_t;
typedef LocalRingSignals ring_signal_t; 
typedef DspinOutput<ring_cmd_data_size >  cmd_out_t;
typedef DspinInput<ring_rsp_data_size >   rsp_in_t;

private:
        
        enum ring_cmd_fsm_state_e {
        	CMD_IDLE,	 // waiting for first flit of a command packet
        	ALLOC,  	// next flit of a local cmd packet
        	NALLOC,         // next flit of a ring cmd packet
                PALLOC2,        // local target is receiving from init gate (ring is cmd_preempted) while target gate is allocated
                PALLOC1,        // local target is receiving from init gate, target gate has finished receiving (target gate is not allocated)        
        };
        
        // cmd token allocation fsm
        enum ring_rsp_fsm_state_e {
        	RSP_IDLE,	    
        	OWNER,  	
        	SENDING,  
	        WAIT_PALLOC_END,            
        };
        
        // structural parameters
	std::string   m_name;
        bool          m_alloc_target;

        // internal fifos 
        GenericFifo<uint64_t > m_cmd_fifo;     // fifo for the local command paquet
        GenericFifo<uint64_t > m_rsp_fifo;     // fifo for the local response paquet

        // routing table 
        soclib::common::AddressDecodingTable<vci_addr_t, int> m_rt;
        // locality table
        soclib::common::AddressDecodingTable<vci_addr_t, bool> m_lt;

        int           m_tgtid;


#ifdef T_DEBUG
        uint32_t            m_cpt;
#endif

        // internal registers
        sc_core::sc_signal<int>	        r_ring_cmd_fsm;	    // ring command packet FSM 
        sc_core::sc_signal<int>		r_ring_rsp_fsm;	    // ring response packet FSM
        
            
public :

#define __renRegTgt(x) x((((std::string) name)+"_" #x).c_str())

DspinLocalRingTargetFastC(
	const char     *name,
        bool            alloc_target,
        const int       &wrapper_fifo_depth,
        const soclib::common::MappingTable &mt,
        const soclib::common::IntTab &ringid,
        const int &tgtid)
     :  m_name(name),
        m_alloc_target(alloc_target),
        m_cmd_fifo(((std::string) name)+"m_cmd_fifo", wrapper_fifo_depth),
        m_rsp_fifo(((std::string) name)+"m_rsp_fifo", wrapper_fifo_depth),
        m_rt(mt.getRoutingTable<typename vci_param::fast_addr_t>(ringid)),
        m_lt(mt.getLocalityTable<typename vci_param::fast_addr_t>(ringid)),
        m_tgtid(tgtid),
        __renRegTgt(r_ring_cmd_fsm),
        __renRegTgt(r_ring_rsp_fsm)

{} //  end constructor

void reset()
{
        if(m_alloc_target)
        	r_ring_rsp_fsm = OWNER;
        else
        	r_ring_rsp_fsm = RSP_IDLE;
        
        r_ring_cmd_fsm = CMD_IDLE;
        m_cmd_fifo.init();
        m_rsp_fifo.init();

#ifdef T_DEBUG 
	m_cpt = 0;
#endif

   
}
void transition(const cmd_out_t &p_cmd_out, const rsp_in_t &p_rsp_in, const ring_signal_t p_ring_in, bool &tgt_cmd_val, rsp_str &tgt_rsp)       
{

	bool      cmd_fifo_put = false;
	uint64_t  cmd_fifo_data = 0;
	
	bool      rsp_fifo_get = false;
	bool      rsp_fifo_put = false;
	uint64_t  rsp_fifo_data = 0;

//////////// DSPIN FIFOS ACCESS /////////////////////////

	if (p_rsp_in.write) {
#ifdef T_DEBUG
std::cout << sc_time_stamp() << " -- " << m_name << " -- DSPIN FIFO" 
          << " -- rsp in rok : " << p_rsp_in.write
          << " --  in data  : " << std::hex << p_rsp_in.data.read()
          << " --  fifo wok : "  << m_rsp_fifo.wok()          
          << std::endl;
#endif

		rsp_fifo_data = (uint64_t) p_rsp_in.data.read();
		rsp_fifo_put =  m_rsp_fifo.wok();
	}

	bool cmd_fifo_get = p_cmd_out.read;
  
//////////// RING RSP FSM  /////////////////////////
        
	switch( r_ring_rsp_fsm ) 
	{
		case RSP_IDLE: 
  
#ifdef T_DEBUG 
if(m_rsp_fifo.rok())
        std::cout << std::dec << sc_time_stamp() << " - " << m_name
                  << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_t[r_ring_rsp_fsm]
                  << " - fifo ROK : " << m_rsp_fifo.rok()
                  << " - in grant : " << p_ring_in.rsp_grant  
                  << " - in palloc : " << p_ring_in.rsp_palloc
                  << " - in wok : " << p_ring_in.rsp_r
                  << " - fifo data : " << std::hex << m_rsp_fifo.read()
                  << std::endl;  
#endif 

			if ( p_ring_in.rsp_grant && m_rsp_fifo.rok() ) 

				r_ring_rsp_fsm = SENDING;           
                
		break;

		case OWNER: 
                { 

#ifdef T_DEBUG 
if(m_rsp_fifo.rok())
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_t[r_ring_rsp_fsm]
                          << " - in rsp grant : " << p_ring_in.rsp_grant
                          << " - in wok : " <<  p_ring_in.rsp_r
                          << " - fifo rok : " <<  m_rsp_fifo.rok()
                          << " - fifo data  : " <<  std::hex << m_rsp_fifo.read()
                          << std::endl;  
#endif

                        bool eop = ( (int) ((m_rsp_fifo.read() >> (ring_rsp_data_size - 1) ) & 0x1) == 1);

                        if ( m_rsp_fifo.rok() && eop && p_ring_in.rsp_r )  
			{
				rsp_fifo_get = true;
                                if ( p_ring_in.rsp_grant )
				        r_ring_rsp_fsm = OWNER;
                                else
				        r_ring_rsp_fsm = RSP_IDLE;
			}  
 
                        if ( m_rsp_fifo.rok() && (!eop || !p_ring_in.rsp_r)) 
			{
				rsp_fifo_get = p_ring_in.rsp_r;
				r_ring_rsp_fsm = SENDING;
			}

                        if ( !m_rsp_fifo.rok() && !p_ring_in.rsp_grant )
				r_ring_rsp_fsm = RSP_IDLE; 

                        if ( !m_rsp_fifo.rok() && p_ring_in.rsp_grant )
				r_ring_rsp_fsm = OWNER;
                }
		break;

		case SENDING: 
 
                // stay here while !release + rsp_preempt 
                // release = fifo_rok.ring_in.wok.fifo_data.eop
#ifdef T_DEBUG
if(m_rsp_fifo.rok())
        std::cout << std::dec << sc_time_stamp() << " - " << m_name
                  << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_t[r_ring_rsp_fsm] 
                  << " - fifo ROK : " << m_rsp_fifo.rok()
                  << " - in grant : " << p_ring_in.rsp_grant  
                  << " - in preempt : " << p_ring_in.rsp_preempt
                  << " - in palloc : " << p_ring_in.rsp_palloc
                  << " - in wok : " << p_ring_in.rsp_r
                  << " - fifo data : " << std::hex << m_rsp_fifo.read()
                  << std::endl;
#endif                       
                        if(p_ring_in.rsp_preempt) break;

			if(m_rsp_fifo.rok() && p_ring_in.rsp_r)
			{
				rsp_fifo_get = true;  
                                bool eop = ((int) (m_rsp_fifo.read() >> (ring_rsp_data_size - 1) ) & 0x1) == 1;
                                
				if (eop && !p_ring_in.rsp_palloc && p_ring_in.rsp_grant)
                                { 
						r_ring_rsp_fsm = OWNER;  
                                }
                                else if  (eop && !p_ring_in.rsp_palloc && !p_ring_in.rsp_grant)          
                                        {   
						r_ring_rsp_fsm = RSP_IDLE; 
                                        }
                                        else if (eop && p_ring_in.rsp_palloc) 
                                                {
						        r_ring_rsp_fsm = WAIT_PALLOC_END;
				                }
			}   
   
		break;

		case WAIT_PALLOC_END:
                // stay here and keep token till Init Gate last flit
                bool eop = ((int) (p_ring_in.rsp_data >> (ring_rsp_data_size - 1) ) & 0x1) == 1;
#ifdef T_DEBUG
if(p_ring_in.rsp_w)
        std::cout << std::dec << sc_time_stamp() << " - " << m_name
                  << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_t[r_ring_rsp_fsm] 
                  << " - palloc : " << p_ring_in.rsp_palloc
                  << " - in grant : " << p_ring_in.rsp_grant  
                  << " - in preempt : " << p_ring_in.rsp_preempt
                  << " - in rok : " << p_ring_in.rsp_w
                  << " - in wok : " << p_ring_in.rsp_r
                  << " - eop : " << eop
                  << std::endl;
#endif                        
                        if(p_ring_in.rsp_w && p_ring_in.rsp_r && eop) // last flit from Init gate
			{
				if (p_ring_in.rsp_grant)
					r_ring_rsp_fsm = OWNER;  
                                else            
					r_ring_rsp_fsm = CMD_IDLE; 
				        
			}   
   
		break;
	} // end switch ring cmd fsm

/////////// RING CMD FSM ////////////////////////
	switch( r_ring_cmd_fsm ) 
	{

		case CMD_IDLE:  
		{ // for variable scope

			vci_addr_t rtgtid = (vci_addr_t) (((p_ring_in.cmd_data >> (ring_cmd_data_size-vci_param::S-1)) << (vci_param::N-vci_param::S)));
			bool islocalm = m_lt[rtgtid]  && (m_rt[rtgtid] == m_tgtid);              // multicast
			bool isbrdcst = (p_ring_in.cmd_data & 0x1) == 0x1;
			bool islocalb = ( (int) ((p_ring_in.cmd_data >> 1 ) & 0xF) == m_tgtid); // broadcast
                        bool eop      = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1); 

			if(p_ring_in.cmd_w)
			{
#ifdef T_DEBUG
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_t[r_ring_cmd_fsm]
                          << " - in preempt : " << p_ring_in.cmd_preempt
                          << " - in rok : " << p_ring_in.cmd_w
                          << " - in data : " << std::hex << p_ring_in.cmd_data
                          << " - in wok : " << p_ring_in.cmd_r
                          << " - rtgtid : " << rtgtid
                          << " - fifo wok : " << m_cmd_fifo.wok()
                          << " - islocalm : " << islocalm
                          << " - islocalb : " << islocalb
                          << " - eop : " << eop
                          << std::endl;
#endif
				if ( (isbrdcst && islocalb) || (!isbrdcst && islocalm) )
				{
                               		
                              		cmd_fifo_put   = m_cmd_fifo.wok();
                              		cmd_fifo_data  = p_ring_in.cmd_data;

					if (eop && m_cmd_fifo.wok())
						r_ring_cmd_fsm = CMD_IDLE;
					else
						r_ring_cmd_fsm = ALLOC;
	
				}

				else
				{
                                        if (eop && p_ring_in.cmd_r)
                                                r_ring_cmd_fsm = CMD_IDLE;
                                        else
                                                r_ring_cmd_fsm = NALLOC;
				}
			}
			else
				r_ring_cmd_fsm = CMD_IDLE;
                }

		break;

		case ALLOC:  
#ifdef T_DEBUG
if(p_ring_in.cmd_w)
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_t[r_ring_cmd_fsm]
                          << " - in preempt : " << p_ring_in.cmd_preempt
                          << " - in rok : " << p_ring_in.cmd_w
                          << " - in data : " << std::hex << p_ring_in.cmd_data
                          << " - in wok : " << p_ring_in.cmd_r
                          << " - fifo wok : " << m_cmd_fifo.wok()
                          << std::endl;
#endif
                {
                        bool eop = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1);


                 	if ( p_ring_in.cmd_w && m_cmd_fifo.wok() && eop )
                        { 

				cmd_fifo_put  = true;
				cmd_fifo_data = p_ring_in.cmd_data;
			     	r_ring_cmd_fsm = CMD_IDLE;
			     		
                        }
                        
                 	else // !p_ring_in.cmd_w || !m_cmd_fifo.wok() || !eop
                        { 

				cmd_fifo_put  = p_ring_in.cmd_w && m_cmd_fifo.wok();
				cmd_fifo_data = p_ring_in.cmd_data;
			     	r_ring_cmd_fsm = ALLOC;
			     		
                        }                        
                } 
		break;

		case NALLOC:  
                // if preempt (init gate is preempting ring while target gate is allocated), 
                // if first flit of a new packet, is it for this local target ? 
#ifdef T_DEBUG
if(p_ring_in.cmd_preempt || p_ring_in.cmd_w)
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_t[r_ring_cmd_fsm]
                          << " - preempt : " << p_ring_in.cmd_preempt
                          << " - palloc : " << p_ring_in.cmd_palloc
                          << " - header : " << p_ring_in.cmd_header
                          << " - in rok : " << p_ring_in.cmd_w
                          << " - in data : " << std::hex << p_ring_in.cmd_data
                          << " - in wok : " << p_ring_in.cmd_r
                          << " - fifo wok : " << m_cmd_fifo.wok()
                          << std::endl;
#endif              
                {
			bool eop = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1);

                        
                        if(p_ring_in.cmd_preempt) 
                        {
                                // cmd_header : sent by init gate 
                                if(p_ring_in.cmd_header)
                                { 
					vci_addr_t rtgtid = (vci_addr_t) (((p_ring_in.cmd_data >> (ring_cmd_data_size-vci_param::S-1)) << (vci_param::N-vci_param::S)));
					bool islocalm = m_lt[rtgtid]  && (m_rt[rtgtid] == m_tgtid);              // multicast
					bool isbrdcst = (p_ring_in.cmd_data & 0x1) == 0x1;
					bool islocalb = ( (int) ((p_ring_in.cmd_data >> 1 ) & 0xF) == m_tgtid); // broadcast
        				bool eop      = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1);


					if ( (isbrdcst && islocalb) || (!isbrdcst && islocalm) )
					{
                        			
                        			cmd_fifo_put   = m_cmd_fifo.wok();
                        			cmd_fifo_data  = p_ring_in.cmd_data;

                                                if (!eop) 
                                                        r_ring_cmd_fsm = PALLOC2;

						//if (eop && m_cmd_fifo.wok())
						//	r_ring_cmd_fsm = CMD_IDLE;
						//else
						//	r_ring_cmd_fsm = PALLOC2;
	
					}

					else
					{
                        		        if (eop && p_ring_in.cmd_r)
                        		                r_ring_cmd_fsm = CMD_IDLE;
                        		        else
                        		                r_ring_cmd_fsm = NALLOC;
					}

                                }
                                else // preempt && !header 
                                {
                                        // palloc=1, means TG is not allocated, thus, if last flit, all targets must be in IDLE state
                                        if(eop && p_ring_in.cmd_r && (p_ring_in.cmd_palloc == 1))        
                                                r_ring_cmd_fsm = CMD_IDLE;
                                        else
                                                r_ring_cmd_fsm = NALLOC;

                                }

                        }

                        else // !preempt
                        {
                                if(p_ring_in.cmd_w && eop && p_ring_in.cmd_r && !p_ring_in.cmd_palloc)
                                        r_ring_cmd_fsm = CMD_IDLE;
                                else 
                                        r_ring_cmd_fsm = NALLOC;

                        }

                }
		break;

                case PALLOC2:
#ifdef T_DEBUG
if(p_ring_in.cmd_preempt || p_ring_in.cmd_w)
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_t[r_ring_cmd_fsm]
                          << " - in preempt : " << p_ring_in.cmd_preempt
                          << " - in rok : " << p_ring_in.cmd_w
                          << " - in data : " << std::hex << p_ring_in.cmd_data
                          << " - in wok : " << p_ring_in.cmd_r
                          << " - fifo wok : " << m_cmd_fifo.wok()
                          << std::endl;
#endif
                {
                         bool eop = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1);

                        if(p_ring_in.cmd_preempt)      
                        {
                                cmd_fifo_put  = m_cmd_fifo.wok();
                                cmd_fifo_data = p_ring_in.cmd_data;
                                
                                if(eop && m_cmd_fifo.wok())
                                        r_ring_cmd_fsm = NALLOC;
                                else
                                        r_ring_cmd_fsm = PALLOC2;
                                break;
                        }

                        if(p_ring_in.cmd_w && eop && p_ring_in.cmd_r)
                        {
			        r_ring_cmd_fsm = PALLOC1;
                        }
                 	                        
                } 
		break;                       

                case PALLOC1:
#ifdef T_DEBUG
    std::cout << std::dec << sc_time_stamp() << " - " << m_name
                          << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_t[r_ring_cmd_fsm]
                          << " - in preempt : " << p_ring_in.cmd_preempt
                          << " - in rok : " << p_ring_in.cmd_w
                          << " - in data : " << std::hex << p_ring_in.cmd_data
                          << " - in wok : " << p_ring_in.cmd_r
                          << " - fifo wok : " << m_cmd_fifo.wok()
                          << std::endl;
#endif
                {
                         bool eop = ( (int) ((p_ring_in.cmd_data >> (ring_cmd_data_size - 1) ) & 0x1) == 1);

                 	if ( p_ring_in.cmd_preempt && m_cmd_fifo.wok() && eop )
                        { 

				cmd_fifo_put  = true;
				cmd_fifo_data = p_ring_in.cmd_data;
                                r_ring_cmd_fsm = CMD_IDLE;
			     		
                        }
                        
                 	else // !p_ring_in.cmd_w || !m_cmd_fifo.wok() || !eop 
                        { 

				cmd_fifo_put  = p_ring_in.cmd_preempt && m_cmd_fifo.wok();
				cmd_fifo_data = p_ring_in.cmd_data;
			     	r_ring_cmd_fsm = PALLOC1;
			     		
                        }                        
                } 
		break;

	} // end switch cmd fsm 

    ////////////////////////
    //  fifos update      //
   ////////////////////////
//-- to keep trace on ring traffic : a valid command is received by the target
        tgt_rsp.rspval  = rsp_fifo_get;
        tgt_rsp.flit    = m_rsp_fifo.read();
        tgt_rsp.state   = ring_rsp_fsm_state_str_t[r_ring_rsp_fsm];

	tgt_cmd_val = cmd_fifo_put;
	//tgt_rsp_val = rsp_fifo_get;
// local cmd fifo update
	if ( cmd_fifo_put && cmd_fifo_get ) m_cmd_fifo.put_and_get(cmd_fifo_data);
	else if (  cmd_fifo_put && !cmd_fifo_get ) m_cmd_fifo.simple_put(cmd_fifo_data);
	else if ( !cmd_fifo_put && cmd_fifo_get ) m_cmd_fifo.simple_get();
// local rsp fifo update
	if (  rsp_fifo_put &&  rsp_fifo_get ) m_rsp_fifo.put_and_get(rsp_fifo_data);
	else if (  rsp_fifo_put && !rsp_fifo_get ) m_rsp_fifo.simple_put(rsp_fifo_data);
	else if ( !rsp_fifo_put &&  rsp_fifo_get ) m_rsp_fifo.simple_get();
 

#ifdef T_DEBUG
	m_cpt +=1;
#endif
}  // end Transition()
  
///////////////////////////////////////////////////////////////////
void genMoore(cmd_out_t &p_cmd_out, rsp_in_t &p_rsp_in)
///////////////////////////////////////////////////////////////////
{
	p_cmd_out.write = m_cmd_fifo.rok();
	p_cmd_out.data  = (sc_dt::sc_uint<ring_cmd_data_size>) m_cmd_fifo.read();

	p_rsp_in.read = m_rsp_fifo.wok();

} // end genMoore

///////////////////////////////////////////////////////////////////
void update_ring_signals(ring_signal_t p_ring_in, ring_signal_t &p_ring_out)
///////////////////////////////////////////////////////////////////
{

	switch( r_ring_rsp_fsm ) 
	{
		case RSP_IDLE:
			p_ring_out.rsp_grant = !m_rsp_fifo.rok() && p_ring_in.rsp_grant;

                        p_ring_out.rsp_palloc  = p_ring_in.rsp_palloc;
                        p_ring_out.rsp_preempt = p_ring_in.rsp_preempt;

	        	p_ring_out.rsp_w    = p_ring_in.rsp_w;
		        p_ring_out.rsp_data = p_ring_in.rsp_data;

		break;

		case OWNER:
                {
                        bool eop = ( (int) ((m_rsp_fifo.read() >> (ring_rsp_data_size - 1) ) & 0x1) == 1);
			p_ring_out.rsp_grant = (!m_rsp_fifo.rok() || (eop && p_ring_in.rsp_r)) ; 

                        p_ring_out.rsp_palloc  = 0;                   
                        p_ring_out.rsp_preempt = 0;                     

        		p_ring_out.rsp_w    =  m_rsp_fifo.rok();
	        	p_ring_out.rsp_data =  m_rsp_fifo.read(); 

                }
		break;

		case SENDING: 
                { 
                        bool eop = ( (int) ((m_rsp_fifo.read() >> (ring_rsp_data_size - 1) ) & 0x1) == 1);

                        p_ring_out.rsp_palloc  = p_ring_in.rsp_palloc;
                        p_ring_out.rsp_preempt = p_ring_in.rsp_preempt;

                        // if preempt : Init Gate is sending to Local Target, ring preempted
                        if (p_ring_in.rsp_preempt) 
                        {
                		p_ring_out.rsp_w     = p_ring_in.rsp_w;
	                	p_ring_out.rsp_data  = p_ring_in.rsp_data;
                                p_ring_out.rsp_grant = 0;
                        }
                        else
                        {
	        	        p_ring_out.rsp_w     = m_rsp_fifo.rok();
		                p_ring_out.rsp_data  = m_rsp_fifo.read();
                                p_ring_out.rsp_grant = m_rsp_fifo.rok() && p_ring_in.rsp_r && eop && !p_ring_in.rsp_palloc;
                        }
                }
		break; 

               case WAIT_PALLOC_END: 
                {
                        bool eop = ((int) (p_ring_in.rsp_data >> (ring_rsp_data_size - 1) ) & 0x1) == 1;

                        p_ring_out.rsp_grant = p_ring_in.rsp_w && p_ring_in.rsp_r && eop; 

                        p_ring_out.rsp_palloc  = p_ring_in.rsp_palloc;
                        p_ring_out.rsp_preempt = p_ring_in.rsp_preempt; 

               		p_ring_out.rsp_w       = p_ring_in.rsp_w; //p_ring_in.rsp_preempt; 
                	p_ring_out.rsp_data    = p_ring_in.rsp_data;
                }        
		break;
	} // end switch

        p_ring_out.rsp_header  = p_ring_in.rsp_header;
        p_ring_out.rsp_r       = p_ring_in.rsp_r;

	p_ring_out.cmd_w       = p_ring_in.cmd_w;
	p_ring_out.cmd_data    = p_ring_in.cmd_data;

	p_ring_out.cmd_grant   = p_ring_in.cmd_grant;

        p_ring_out.cmd_preempt = p_ring_in.cmd_preempt;
        p_ring_out.cmd_header  = p_ring_in.cmd_header;
        p_ring_out.cmd_palloc  = p_ring_in.cmd_palloc;

	switch( r_ring_cmd_fsm ) 
	{
		case CMD_IDLE:
		{
			vci_addr_t rtgtid = (vci_addr_t) (((p_ring_in.cmd_data >> (ring_cmd_data_size-vci_param::S-1)) << (vci_param::N-vci_param::S)));
			bool islocalm = m_lt[rtgtid]  && (m_rt[rtgtid] == m_tgtid); // multicast
			bool isbrdcst = (p_ring_in.cmd_data & 0x1) == 0x1;
			bool islocalb = ( (int) ((p_ring_in.cmd_data >> 1 ) & 0xF) == m_tgtid); // broadcast
                        
			if (p_ring_in.cmd_w && ((isbrdcst && islocalb) || (!isbrdcst && islocalm))) 
			{
                               	p_ring_out.cmd_r =  m_cmd_fifo.wok();
                       	} 
       			else
                       	{
                               	p_ring_out.cmd_r =  p_ring_in.cmd_r; 
                       	}  

		}
		break;

		case ALLOC:
                        p_ring_out.cmd_r =  m_cmd_fifo.wok(); 	
		break;

		case NALLOC:
		{
			vci_addr_t rtgtid = (vci_addr_t) (((p_ring_in.cmd_data >> (ring_cmd_data_size-vci_param::S-1)) << (vci_param::N-vci_param::S)));
			bool islocalm = m_lt[rtgtid]  && (m_rt[rtgtid] == m_tgtid); // multicast
			bool isbrdcst = (p_ring_in.cmd_data & 0x1) == 0x1;
			bool islocalb = ( (int) ((p_ring_in.cmd_data >> 1 ) & 0xF) == m_tgtid); // broadcast

                        if (p_ring_in.cmd_preempt && p_ring_in.cmd_header && ((isbrdcst && islocalb) || (!isbrdcst && islocalm)))
                        	p_ring_out.cmd_r  = m_cmd_fifo.wok();
                        else 
                        	p_ring_out.cmd_r  =  p_ring_in.cmd_r; 

                }
		break; 

		case PALLOC2:
                        if(p_ring_in.cmd_preempt)
                                p_ring_out.cmd_r  = m_cmd_fifo.wok(); 	
                        else 
                                p_ring_out.cmd_r  = p_ring_in.cmd_r;
		break;

		case PALLOC1:
                        p_ring_out.cmd_r  = m_cmd_fifo.wok(); 	
                break;

	} // end switch

} // end update_ring_signals 
  
};

}} // end namespace
#endif // SOCLIB_CABA_DSPIN_ALLOC_RING_TARGET_FAST_H

