#!/usr/bin/env python

from arch_classes import *

#########################################################################################
#   file   : arch_info.py 
#   date   : august 2016
#   author : Alain Greiner
#
#   This file describes the <tsar_generic_iob> architecture for ALMOS-MKH.
#########################################################################################
#  This python script defines a specific instance of ithe "tsar_generic_iob" architecture
#  for the ALMOS-MKH operating system. It is used to generate the "hard_config.h" file, 
#  used to configure the hardware architecture, and the "arch_info.bin" file, used by 
#  the ALMOS-MK bootloader.
#
#  The constructor prototype format is imposed by the genarch.py application,
#  and should not be modified.
#
#  The "tsar_generic_iob" architecture includes 7 external peripherals, accessed
#  through an IOB components located in cluster [0,0] or in cluster [x_size-1, y_size-1].
#  Available peripherals are: TTY, IOC, FBF, ROM, NIC, CMA, PIC.
#  All clusters contain (nb_cores) processors, one L2 cache, one XCU, and
#  one optional hardware coprocessor connected to a MWMR controller.
#
#  As the "tsar_generic_iob" architecture is generic, the following parameters 
#  are defined as constructor arguments and can be redefined in the Makefile when 
#  a new kernel image is generated :
#  - x_size         : number of clusters in a row (from 1 to 16)
#  - y_size         : number of clusters in a column (from & to 16)
#  - nb_cores       : number of processors per cluster (from 1 to 4)
#  - nb_ttys        : number of TTY channels (can be from 1 to 8)
#  - nb_nics        : number of NIC channels (from 1 to 2)
#  - ioc_type       : can be IOC_BDV, IOC_HBA, IOC_SDC, IOC_SPI
#  - txt_type       : IOB architecture use TXT_TTY 
#  - fbf_type       : IOB architecture use FBF_SCL
#  - sys_clk        : IOB architecture is SystemC only
#
#  The other hardware parameters are defined below:
#  - x_width        : number of bits for x coordinate
#  - y_width        : number of bits for y coordinate
#  - p_width        : number of bits for local processor index
#  - paddr_width    : number of bits for physical address
#  - irqs_per_core  : number of input IRQs per processor
#  - io_cxy         : IO cluster identifier
#  - boot_cxy       : boot cluster identifier
#  - cache_line     : number of bytes in cache line (in 16,32,64)
#  - devices_max    : max number of internal devices per cluster
#  - fbf_width      : number of lines = number of pixels
########################################################################################

############################
def arch( x_size        = 2,
          y_size        = 2,
          nb_cores      = 2,
          nb_ttys       = 3,
          nb_nics       = 1, 
          ioc_type      = 'IOC_BDV',
          txt_type      = 'TXT_TTY',
          fbf_type      = 'FBF_SCL',
          sys_clk       = 50000 ): 

    ### architecture constants

    p_width       = 2
    x_width       = 4
    y_width       = 4
    paddr_width   = 40
    irqs_per_core = 4           
    io_cxy        = ((x_size-1)<<y_width) + (y_size-1)   # upper right cluster
    boot_cxy      = 0
    cache_line    = 64
    devices_max   = 16  
    fbf_width     = 128

    ### constructor parameters checking

    assert( (x_size == 1) or (x_size == 2) or (x_size == 4)
             or (x_size == 8) or (x_size == 16) )

    assert( (y_size == 1) or (y_size == 2) or (y_size == 4)
             or (y_size == 8) or (y_size == 16) )

    assert( nb_cores <= 4 )

    assert( (nb_ttys >= 1) and (nb_ttys <= 8) )

    assert( (nb_nics >= 1) and (nb_nics <= 2) )

    assert( ioc_type in ['IOC_BDV','IOC_HBA','IOC_SDC','IOC_SPI','IOC_RDK'] )

    assert( (io_cxy == 0) or (io_cxy == ((x_size-1)<<y_width) + (y_size-1)) ) 

    assert( ((boot_cxy >> y_width) < x_size) and ((boot_cxy & ((1<<y_width)-1)) < y_size) )

    assert( (cache_line == 16) or (cache_line == 32) or (cache_line == 64)  )
 
    ### define platform name

    platform_name = 'tsar_iob_%d_%d_%d' % ( x_size, y_size , nb_cores )

    ### define physical segments replicated in all clusters

    ram_base = 0x0000000000
    ram_size = 0x4000000                   # 64 Mbytes

    xcu_base = 0x00B0000000
    xcu_size = 0x1000                      # 4 Kbytes

    dma_base = 0x00B1000000
    dma_size = 0x1000                      # 4 Kbytes 

    mmc_base = 0x00B2000000
    mmc_size = 0x1000                      # 4 Kbytes

    ### define physical segments for external peripherals
    ## These segments are only defined in cluster_io

    ioc_base  = 0x00B3000000               
    ioc_size  = 0x1000                     # 4 Kbytes

    tty_base  = 0x00B4000000
    tty_size  = 0x8000                     # 4 Kbytes * 8 channels

    nic_base  = 0x00B5000000
    nic_size  = 0x4000                     # 4 Kbytes * 4 channels

    fbf_base  = 0x00B7000000
    fbf_size  = 0x400000 + 0x1000          # 4 Mbytes + 4 Kbytes 

    pic_base  = 0x00B8000000
    pic_size  = 0x1000                     # 4 Kbytes

    iob_base  = 0x00BE000000
    iob_size  = 0x1000                     # 4 Kbytes

    rom_base  = 0x00BFC00000
    rom_size  = 0x4000                     # 16 Kbytes

    ############################
    ### call Header constructor
    ############################

    archi = Archinfo( name           = platform_name,
                      x_size         = x_size,
                      y_size         = y_size,
                      cores_max      = nb_cores,
                      devices_max    = devices_max,
                      paddr_width    = paddr_width,
                      x_width        = x_width,
                      y_width        = y_width,
                      irqs_per_core  = irqs_per_core,
                      io_cxy         = io_cxy,          
                      boot_cxy       = boot_cxy,
                      cache_line     = cache_line,
                      reset_address  = rom_base, 
                      p_width        = p_width )

    ####################################################
    ### construct hardware components for each cluster
    ####################################################

    for x in xrange( x_size ):
        for y in xrange( y_size ):
            cxy    = (x << y_width) + y;
            offset = cxy << (paddr_width - x_width - y_width)

            # define internal devices
            ram = archi.addDevice( ptype    = 'RAM_SCL' , 
                                   base     = ram_base + offset, 
                                   size     = ram_size )

            xcu = archi.addDevice( ptype    = 'ICU_XCU', 
                                   base     = xcu_base + offset, 
                                   size     = xcu_size,
                                   channels = 1, 
                                   arg0     = 16,              # number of HWIs
                                   arg1     = 16,              # number of WTIs
                                   arg2     = 16,              # number of PTIs
                                   arg3     = 16 )             # number of output IRQs

            mmc = archi.addDevice( ptype    = 'MMC_TSR',
                                   base     = mmc_base + offset,
                                   size     = mmc_size )

            archi.addIrq( dstdev = xcu, port = 0, srcdev = mmc )

            dma = archi.addDevice( ptype    = 'DMA_SCL',
                                   base     = dma_base + offset,
                                   size     = dma_size,
                                   channels = nb_cores )

            for i in xrange( nb_cores ):
                archi.addIrq( dstdev = xcu, port = i+1 , srcdev = dma, channel = i )

            # define external devices
            if( cxy == io_cxy ):

                pic = archi.addDevice( ptype    ='PIC_TSR',
                                       base     = pic_base + offset,
                                       size     = pic_size, 
                                       arg0     = 32 )         # number of input IRQs

                iob = archi.addDevice( ptype    = 'IOB_TSR', 
                                       base     = iob_base + offset,
                                       size     = iob_size )

                ioc = archi.addDevice( ptype    = ioc_type,
                                       base     = ioc_base + offset,
                                       size     = ioc_size )

                tty = archi.addDevice( ptype    = 'TXT_TTY',
                                       base     = tty_base + offset,
                                       size     = tty_size, 
                                       channels = nb_ttys )

                nic = archi.addDevice( ptype    = 'NIC_CBF',
                                       base     = nic_base + offset, 
                                       size     = nic_size, 
                                       channels = nb_nics )

                fbf = archi.addDevice( ptype    = 'FBF_SCL',
                                       base     = fbf_base + offset,
                                       size     = fbf_size, 
                                       arg0     = fbf_width,
                                       arg1     = fbf_width )

                rom = archi.addDevice( ptype    = 'ROM_SCL',
                                       base     = rom_base + offset, 
                                       size     = rom_size )

                # we describe the largest config : (nb_nics = 4) & (nb_ttys = 8)
                archi.addIrq( dstdev = pic, port = 0 , srcdev = nic, channel = 0 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 1 , srcdev = nic, channel = 1 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 2 , srcdev = nic, channel = 2 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 3 , srcdev = nic, channel = 3 , is_rx = True )

                archi.addIrq( dstdev = pic, port = 4 , srcdev = nic, channel = 0 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 5 , srcdev = nic, channel = 1 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 6 , srcdev = nic, channel = 2 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 7 , srcdev = nic, channel = 3 , is_rx = False )

                archi.addIrq( dstdev = pic, port = 12, srcdev = ioc )

                archi.addIrq( dstdev = pic, port = 16, srcdev = tty, channel = 0 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 17, srcdev = tty, channel = 1 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 18, srcdev = tty, channel = 2 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 19, srcdev = tty, channel = 3 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 20, srcdev = tty, channel = 4 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 21, srcdev = tty, channel = 5 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 22, srcdev = tty, channel = 6 , is_rx = True )
                archi.addIrq( dstdev = pic, port = 23, srcdev = tty, channel = 7 , is_rx = True )

                archi.addIrq( dstdev = pic, port = 24, srcdev = tty, channel = 0 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 25, srcdev = tty, channel = 1 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 26, srcdev = tty, channel = 2 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 27, srcdev = tty, channel = 3 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 28, srcdev = tty, channel = 4 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 29, srcdev = tty, channel = 5 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 30, srcdev = tty, channel = 6 , is_rx = False )
                archi.addIrq( dstdev = pic, port = 31, srcdev = tty, channel = 7 , is_rx = False )

            # define cores
            for p in xrange ( nb_cores ):
                core = archi.addCore( (x<<(y_width+p_width)) + (y<<p_width) + p,  # hardware id
                                      (x<<y_width) + y,                           # cluster 
                                       p )                                        # local index

    return archi

################################# platform test ####################################

if __name__ == '__main__':

    archi = arch()

    print archi.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

