
/* -*- c++ -*-
 * File 	: vci_synthetic_initiator.cpp
 * Date 	: 23/12/2010
 * Copyright 	: UPMC / LIP6
 * Authors 	: Christophe Choichillon
 * Version	: 2.0
 *
 * SOCLIB_LGPL_HEADER_BEGIN
 *
 * This file is part of SoCLib, GNU LGPLv2.1.
 *
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 *
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * SOCLIB_LGPL_HEADER_END
 *
 * Maintainers: christophe.choichillon@lip6.fr
 */

#include "../include/vci_synthetic_initiator.h"
#include <iostream>


#define DETERMINISTIC

namespace soclib { namespace caba {


#define tmpl(x) template<typename vci_param> x VciSyntheticInitiator<vci_param>

  //using soclib::common::uint32_log2;	
  
  ////////////////////////////////
  // 	Constructor 
  ////////////////////////////////

  tmpl(/**/)::VciSyntheticInitiator( 
      sc_module_name name,
      const soclib::common::MappingTable &mt,
      const soclib::common::IntTab       &vci_index,
      const uint32_t length,    // Packet length (flit numbers)
      const uint32_t rho,       // Packets ratio on the network
    //  const float  rho,       // Packets ratio on the network
      const uint32_t depth,     // Fifo depth
      const uint32_t xmesh,	
      const uint32_t ymesh,
      const uint32_t bc_period, // Broadcast period, if no broadcast => 0
      const uint32_t xmin, 
      const uint32_t xmax,
      const uint32_t ymin,
      const uint32_t ymax
      )

    : soclib::caba::BaseModule(name),

    p_clk("clk"),
    p_resetn("resetn"),
    p_vci("vci_ini"),

    m_srcid( mt.indexForId(vci_index) ),
    //  FIFOs 
    m_length(length),
    m_rho(rho),
    m_depth(depth),
    m_xmesh(xmesh),
    m_ymesh(ymesh),
    m_bc_period(bc_period),
    m_xmin(xmin),
    m_xmax(xmax),
    m_ymin(ymin),
    m_ymax(ymax),
    m_date_fifo("m_date_fifo", depth),
    m_local_seed(m_srcid),
    r_cmd_fsm("r_cmd_fsm")
    {

      r_req_id = new sc_signal<uint64_t>*[tab_size];
      for(int i = 0; i < tab_size ; i++){
        r_req_id[i] = new sc_signal<uint64_t>[2];
      }

      SC_METHOD(transition);
      dont_initialize();
      sensitive << p_clk.pos();

      SC_METHOD(genMoore);
      dont_initialize();
      sensitive << p_clk.neg();

    } // end constructor


  /////////////////////////////////
  tmpl(/**/)::~VciSyntheticInitiator()
    /////////////////////////////////
  {
  }

  ///////////////////////////////////
  tmpl(uint32_t)::destAdress()
  ///////////////////////////////////
  {
    return (uint32_t) (rand() % (m_xmesh * m_ymesh)) ;
  }


  ///////////////////////////////////
  tmpl(uint32_t)::destAdress(uint32_t *rand_seed)
  ///////////////////////////////////
  {
    return (uint32_t) (rand_r(rand_seed) % (m_xmesh * m_ymesh)) ;
  }

  
  //////////////////////////////////
  tmpl(void)::print_trace()
  //////////////////////////////////
  {
  	const char* state_cmd_str[] = { "IDLE",
				        "SINGLE_SEND",
					"BC_SEND"};

	const char* state_bc_rsp_str[] = {"IDLE",
					  "WAIT_RSP"};

	std::cout << "Vci_Synthetic_Initiator " << name()
		  << " : " << std::dec << m_cpt_cycles << " cycles " 
		  << " : state_cmd_fsm = " << state_cmd_str[r_cmd_fsm] 
		  << " : state_rsp_fsm = " << state_bc_rsp_str[r_bc_rsp_fsm] 
		  << " Adresse to send : " << std::hex << m_address_to_send 
		  << " Number of broadcast to receive : " << std::dec << r_bc_nrsp.read() 
		  << " Number of packets sent : " << std::dec << m_npackets << m_id_to_send << std::endl;
	for(int i = 0; i < (1<<vci_param::T) ; i++){
	  std::cout << "ID : " << i << " " << r_req_id[i][0].read() << " " << r_req_id[i][0].read() << std::endl;
	}
  }

  //////////////////////////////////
  tmpl(void)::printStats()
  //////////////////////////////////
  {
  	std::cout << name() << " : "<< std::dec << m_cpt_cycles << " cycles, " << m_npackets << " packets sent" << std::endl;
	std::cout << ((double)m_latency1/(double)m_npackets) << " | " << ((double)m_latency2/(double)m_npackets) << std::endl;
	if(m_bc_period)
	  std::cout << ((double)m_latency_bc/(double)m_nb_bc) << std::endl;
  }

  //////////////////////////////////
  tmpl(void)::transition()
  //////////////////////////////////
  {
    //  RESET          
    if ( ! p_resetn.read() ) {
      // Initializing seed for random numbers generation
#ifndef DETERMINISTIC
      srand(time(NULL));
#endif

      // Initializing FSMs
      r_cmd_fsm = VCI_IDLE;

      r_bc_rsp_fsm = BC_RSP_IDLE;

      // Initializing FIFOs 
      m_date_fifo.init();

      // Initializing the stats
      m_latency1 = 0 ;
      m_latency2 = 0 ;
      // Activity counters
      m_cpt_cycles		= 0;
      m_npackets		= 0;
      
      m_start_latency_bc	= 0;
      m_latency_bc		= 0;
      m_nb_bc			= 0;
      m_id_to_send		= -1;

      r_broadcast_req		= false;

      r_broadcast_rsp		= false;

      r_bc_nrsp			= 0;

      for(int i = 0; i < tab_size; i++){
   	r_req_id[i][0] = 0;
   	r_req_id[i][1] = 0;
	//std::cout << name() << " bla bla " << i << std::endl;
      }

      return;
    }

    bool    date_fifo_put = false;
    bool    date_fifo_get = false;



    // FSM controling effective requests send
    switch ( r_cmd_fsm.read() ) {
      //////////////////
      case VCI_IDLE:
        {
	  if (m_date_fifo.rok()){
	    if (r_broadcast_req.read() && !r_broadcast_rsp.read()){
	      m_address_to_send = 0x3 | (0x7c1f << vci_param::N-20) ;
	      r_cmd_fsm = VCI_BC_SEND ;
	    } else {
	      for(int i = 0; i < tab_size; i++){
	        if(r_req_id[i][0] == 0){
		  m_id_to_send = i;
		  break;
		}else{
		  m_id_to_send = -1;
		}
	      }
	      if(m_id_to_send == -1){
	        r_cmd_fsm = VCI_IDLE ;
		break;
	      } else {
	        r_cmd_fsm = VCI_SINGLE_SEND ;
	      }
#ifdef DETERMINISTIC
	      m_address_to_send = destAdress(&m_local_seed) << (vci_param::N)-(soclib::common::uint32_log2((uint32_t)m_xmesh)+soclib::common::uint32_log2((uint32_t)m_ymesh));
#else
	      m_address_to_send = destAdress() << (vci_param::N)-(soclib::common::uint32_log2((uint32_t)m_xmesh)+soclib::common::uint32_log2((uint32_t)m_ymesh));
#endif
	      m_count = 0;
	    }
	  }
          break;
        }
        //////////////////
      case VCI_SINGLE_SEND:
        {
	  if (p_vci.cmdack.read()){
	    if (m_count == m_length-1) {
	      r_req_id[(int)m_id_to_send][0] = m_date_fifo.read();
	      r_req_id[m_id_to_send][1] = m_cpt_cycles;
	      date_fifo_get = true;
	      r_cmd_fsm = VCI_IDLE ;
	    } else {
	      r_cmd_fsm = VCI_SINGLE_SEND ;
	      m_count++;
	    }
	  }
          break;
        }
      ///////////////////
      case VCI_BC_SEND:
        {
	  if (p_vci.rspval.read()) {
	    r_bc_nrsp = (m_xmax - m_xmin) * (m_ymax - m_ymin) ;
	    m_start_latency_bc = m_cpt_cycles;
	    date_fifo_get = true;
	    r_broadcast_rsp = true;
	    r_bc_rsp_fsm = VCI_IDLE;
            break;
	  }
        }

    } // end switch vci_fsm

    switch(r_bc_rsp_fsm.read()){
      ///////////////////
      case BC_RSP_IDLE:
        {
	  if (p_vci.rspval.read() && r_broadcast_rsp.read()) {
	    r_bc_rsp_fsm = BC_RSP_WAIT_RSP;
            break;
	  }
        }
      ////////////////////
      case BC_RSP_WAIT_RSP:
        {
	  if (p_vci.rspval.read() && (p_vci.rpktid.read() == 1)){
	    if (r_bc_nrsp == 1) {
	      r_broadcast_req = false;
	      r_broadcast_rsp = false;
	      m_address_to_send = 0;
	      m_latency_bc = m_latency_bc + (m_cpt_cycles - m_start_latency_bc);
	      m_nb_bc++;
	      r_bc_rsp_fsm = BC_RSP_IDLE ;
	    } else {
	      r_bc_nrsp = r_bc_nrsp.read() - 1;;
	      r_bc_rsp_fsm = BC_RSP_WAIT_RSP ;
	    }
	  }
          break;
        }    
    }

    if(p_vci.rspval.read()){
      if((int)(p_vci.pktid.read()) == 0){
	m_latency1 = m_latency1 + (m_cpt_cycles - r_req_id[(int)(p_vci.trdid.read())][0]);
	m_latency2 = m_latency2 + (m_cpt_cycles - r_req_id[(int)(p_vci.trdid.read())][1]);
	m_npackets++;
        r_req_id[(int)(p_vci.trdid.read())][0] = 0;
        r_req_id[(int)(p_vci.trdid.read())][1] = 0;
      }
    }

    /////////////////// Filling fifo
    if( ( (uint64_t)(m_rho*m_cpt_cycles) > (uint64_t)(m_length*m_npackets*1000)) ){
      if (m_date_fifo.wok()){
        date_fifo_put = true ;
      } 
      if (m_bc_period){
	      if (!r_broadcast_req.read() && (m_cpt_cycles % m_bc_period)){
		      r_broadcast_req = true;
	      }
      }
    }

    if (date_fifo_put){
      if (date_fifo_get){
        m_date_fifo.put_and_get(m_cpt_cycles);
      } else {
        m_date_fifo.simple_put(m_cpt_cycles);
      }
    } else {
      if (date_fifo_get){
        m_date_fifo.simple_get();
      }
    }
   
    m_cpt_cycles++;

    return;

  } // end transition()

  /////////////////////////////
  tmpl(void)::genMoore()
    /////////////////////////////
  {
    ////////////////////////////////////////////////////////////
    // Command signals on the p_vci port
    ////////////////////////////////////////////////////////////
     p_vci.cmd        = vci_param::CMD_WRITE;   
     p_vci.be         = 0xF;                             
     p_vci.srcid      = m_srcid;   
     p_vci.pktid      = 0;      
     p_vci.cons       = false;        
     p_vci.wrap       = false;       
     p_vci.contig     = true;       
     p_vci.clen       = 0;         
     p_vci.cfixed     = false;           
     p_vci.rspack     = true;


    switch ( r_cmd_fsm.read() ) {

      //////////////////
      case VCI_IDLE:
        {
	  p_vci.cmdval  = false;                 
	  p_vci.address = 0; 
	  p_vci.plen    = 0;                                         
	  p_vci.wdata   = 0;                                       
	  p_vci.trdid   = 0;                  
	  p_vci.eop     = false;                                   
          break;
        }
        //////////////////
      case VCI_SINGLE_SEND:
        {
	  p_vci.cmdval  = true;                 
	  p_vci.address = (addr_t)(m_address_to_send + (m_count*4)); 
	  p_vci.plen    = m_length*4;                                         
	  p_vci.wdata   = 0;                                       
	  p_vci.trdid   = m_id_to_send;                  
	  if (m_count == m_length - 1 ) {
	    p_vci.eop     = true;                                   
	  } else {
	    p_vci.eop     = false;                                   
	  }
          break;
        }
        ///////////////////
      case VCI_BC_SEND:
        {
       	  p_vci.cmdval  = true;                 
	  p_vci.address = (addr_t) m_address_to_send; 
	  p_vci.plen    = 4;                                         
	  p_vci.wdata   = 0;                                       
          p_vci.pktid   = 1;      
	  p_vci.trdid   = 0;                  
	  p_vci.eop     = true;                                   
          break;
        }
    } // end switch vci_cmd_fsm

  } // end genMoore()

}} // end name space
