/* -*- c++ -*-
  * File : vci_vdspin_target_wrapper.h
  * Copyright (c) UPMC, Lip6
  * Author : Alain Greiner
  *
  * SOCLIB_LGPL_HEADER_BEGIN
  * 
  * This file is part of SoCLib, GNU LGPLv2.1.
  * 
  * SoCLib is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as published
  * by the Free Software Foundation; version 2.1 of the License.
  * 
  * SoCLib is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Lesser General Public License for more details.
  * 
  * You should have received a copy of the GNU Lesser General Public
  * License along with SoCLib; if not, write to the Free Software
  * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  * 02110-1301 USA
  * 
  * SOCLIB_LGPL_HEADER_END
  */

/////////////////////////////////////////////////////////////////////////
// This component performs the protocol translation between VCI & DSPIN,
// and can be used to connect a VCI target to a DSPIN netnork.
// It is implemented as two fully independant sub-components:
// - translation from a DSPIN CMD to a VCI CMD
// - translation from a VCI RSP to a DSPIN RSP
// Each subcomponent contains a FIFO containing DSPIN flits.
// For the DSPIN interfaces, the widths of the CMD & RSP flits
// are defined as template parameters for future evolutions,
// but the VCI to DSPIN translation makes the following assumptions:
// - DSPIN CMD flit width == 40 bits
// - DSPIN RSP flit width == 33 bits
// - VCI address width    <= 40 bits
// - VCI data             == 32 bits
// - VCI plen		  == 8  bits
// - VCI srcid		  <= 14 bits
// - VCI trdid		  <= 8  bits
// - VCI pktid field not transmitted
// - VCI rerror		  == 2 bits
////////////////////////////////////////////////////////////////////////

#ifndef VCI_VDSPIN_TARGET_WRAPPER_H_
#define VCI_VDSPIN_TARGET_WRAPPER_H_

#include <systemc>
#include <assert.h>
#include "caba_base_module.h"
#include "vci_initiator.h"
#include "generic_fifo.h"
#include "dspin_interface.h"
#include "static_fast_int.h"

namespace soclib { namespace caba {


template<typename vci_param, int dspin_cmd_width, int dspin_rsp_width>
class VciVdspinTargetWrapper
	: public soclib::caba::BaseModule
{

    // Command FSM 
    enum fsm_state_cmd{
    CMD_IDLE,
    CMD_BROADCAST,
    CMD_RW,
    CMD_READ,
    CMD_WDATA,
    };

    // Response FSM
    enum fsm_state_rsp{
    RSP_IDLE,
    RSP_READ,
    RSP_WRITE,
    };

protected:
    SC_HAS_PROCESS(VciVdspinTargetWrapper);

public:
    // ports
    sc_core::sc_in<bool>                             		p_clk;
    sc_core::sc_in<bool>                             		p_resetn;
    soclib::caba::DspinOutput<dspin_rsp_width>			p_dspin_out;
    soclib::caba::DspinInput<dspin_cmd_width>			p_dspin_in;
    soclib::caba::VciInitiator<vci_param>      			p_vci;

    // constructor / destructor
    VciVdspinTargetWrapper(	sc_module_name 			name, 
				size_t				cmd_fifo_depth,
				size_t				rsp_fifo_depth);
private:
    // internal registers
    sc_core::sc_signal<int>        				r_cmd_fsm;
    sc_core::sc_signal<sc_uint<dspin_cmd_width>	>		r_cmd_buf0;
    sc_core::sc_signal<sc_uint<dspin_cmd_width>	>		r_cmd_buf1;
    sc_core::sc_signal<int>        				r_rsp_fsm;
    sc_core::sc_signal<size_t>					r_flit_count;

    // fifos cmd and rsp
    soclib::caba::GenericFifo<sc_uint<dspin_cmd_width> >  	r_fifo_cmd;
    soclib::caba::GenericFifo<sc_uint<dspin_rsp_width> >  	r_fifo_rsp;

    // methods systemc
    void transition();
    void genMoore();

public:
    void print_trace();

};

}} // end namespace
               
#endif // VCI_VDSPIN_TARGET_WRAPPER_H_

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4
