
__doc__ = '''
This file is a Cluster library. It contains classes implementing the
netlist of a cluster, for different tsar versions.
'''

class Cluster:
    '''
    A generic netlist of a cluster, which must be subclassed to
    implement caches&dma instanciation
    '''
    
    def __init__(self, pf, ringp, mtp, proc_count, cluster_no, cluster_base):
        self.pf = pf
        self.ringp = ringp
        self.mtp = mtp
        self.cluster_no = cluster_no

        self.generate(proc_count, cluster_base)

    def generate(self, proc_count, cluster_base):
        '''
        The core netlist, where caches and components are created
        '''
        self.cpu = []
        for i in range(proc_count):
            c = self.create_cpu(i,
                                cluster_base
                                + 0x10200000
                                + 0x01000000 * (i+1)
                                )
            self.cpu.append(c)

        xram = self.create_ram( segments = [(cluster_base, 0x02000000)] )
        rom  = self.create_rom( segments = [(0xbfc00000, 0x00400000)] )
        tty  = self.create_tty( addr = 0xd0200000 )
        dma  = self.create_dma( addr = 0xd1200000 )
        xicu = self.create_xicu( addr = 0xd2200000, hwi_count = 3, cpu_count = proc_count )

        xicu.hwi[0] // tty.irq[0]
        xicu.hwi[1] // dma.irq

        for p, cpu in enumerate(self.cpu):
            for i in range(6):
                xicu.irq[i+p*6] // cpu.irq[i]

    def create_rom(self, segments):
        name = 'rom%d'%self.cluster_no
        rom = self.pf.create('caba:vci_simple_ram', name, latency = 1)
        self.ringp.to_target.new() // rom.vci // self.pf.create("caba:vci_logger", 'rom_log', mt = self.mtp).vci
        for addr, size in segments:
            rom.addSegment(name, address = addr, size = size, cacheable = True)
        return rom

    def create_ram(self, segments):
        name = 'ram%d'%self.cluster_no
        ram = self.pf.create('caba:vci_simple_ram', name, latency = 1)
        self.ringp.to_target.new() // ram.vci // self.pf.create("caba:vci_logger", 'ram_log', mt = self.mtp).vci
        for addr, size in segments:
            ram.addSegment(name, address = addr, size = size, cacheable = True)
        return ram

    def create_tty(self, addr):
        name = 'tty%d'%self.cluster_no
        tty = self.pf.create('caba:vci_multi_tty', name, names = ['tty0'])
        self.ringp.to_target.new() // tty.vci // self.pf.create("caba:vci_logger", 'tty_log', mt = self.mtp).vci
        tty.addSegment(name, address = addr, size = 0x10, cacheable = False)
        return tty

    def create_xicu(self, addr, hwi_count, cpu_count):
        name = 'xicu%d'%self.cluster_no
        xicu = self.pf.create('caba:vci_xicu', name,
                              pti_count = cpu_count,
                              hwi_count = hwi_count,
                              wti_count = cpu_count,
                              irq_count = 6*cpu_count,
                              )
        self.ringp.to_target.new() // xicu.vci // self.pf.create("caba:vci_logger", 'xicu_log', mt = self.mtp).vci
        xicu.addSegment(name, address = addr, size = 0x1000, cacheable = False)
        return xicu

    def create_cpu(self, cpuid, addr):
            c = self.pf.create(
                'caba:vci_vcache_wrapper2',
                'proc_%d_%d' % (self.cluster_no, cpuid),
                iss_t = "common:mips32el",
                proc_id = cpuid,
                icache_ways = 4,
                icache_sets = 64,
                icache_words = 16,
                itlb_ways = 4,
                itlb_sets = 16,
                dcache_ways = 4,
                dcache_sets = 64,
                dcache_words = 16,
                dtlb_ways = 4,
                dtlb_sets = 16,
                write_buf_size = 16,
                )
            self.ringp.to_initiator.new() // c.vci
            return c

    def create_dma(self, addr):
        name = 'dma%d'%self.cluster_no
        dma = self.pf.create('caba:vci_dma', name, burst_size = 64)
        self.ringp.to_target.new() // dma.vci_target
        self.ringp.to_initiator.new() // dma.vci_initiator
        dma.addSegment(name, address = addr, size = 0x14, cacheable = False)
        return dma
