/**
 * \file    : sdcard.c
 * \date    : 30 August 2012
 * \author  : Cesar Fuguet
 *
 * This file defines the driver of a SD Card device using an SPI controller
 */

#include <sdcard.h>

#define in_reset        __attribute__((section (".reset")))
#define in_reset_data   __attribute__((section (".reset_data")))

/**
 * \param   sdcard: Initialized pointer to the block device
 *
 * \return  void
 *
 * \brief   Enable SD Card select signal
 */
in_reset static void _sdcard_enable(struct sdcard_dev * sdcard)
{
    spi_ss_assert(sdcard->spi, sdcard->slave_id);
}

/**
 * \param   sdcard: Initialized pointer to the block device
 *
 * \return  void
 *
 * \brief   Disable SD Card select signal
 */
in_reset static void _sdcard_disable(struct sdcard_dev * sdcard)
{
    spi_ss_deassert(sdcard->spi, sdcard->slave_id);
}

/**
 * \param   tick_count: SD Card clock ticks number
 *
 * \return  void
 *
 * \brief   Enable SD Card clock
 *          The tick count is byte measured (1 tick, 8 clock)
 */
in_reset static void _sdcard_gen_tick(struct sdcard_dev * sdcard, size_t tick_count)
{
    volatile register int i = 0;
    while(i++ < tick_count) spi_put_tx(sdcard->spi, 0xFF, 0);
}

/**
 * \param   sdcard: Initialized pointer to the block device
 *
 * \return  char from the SD card
 *
 * \brief   Get a byte from the SD Card
 */
in_reset static unsigned char _sdcard_receive_char(struct sdcard_dev * sdcard)
{
    _sdcard_gen_tick(sdcard, 1);

    return spi_get_rx(sdcard->spi, 0);
}

/**
 * \param   sdcard: Initialized pointer to the block device
 *
 * \return  sdcard response
 *
 * \brief   Wait for a valid response after the send of a command
 *          This function can return if one of the next two conditions are true:
 *           1. Bit valid received
 *           2. Timeout (not valid bit received after SDCARD_COMMAND_TIMEOUT
 *              wait ticks)
 */
in_reset static unsigned char _sdcard_wait_response(struct sdcard_dev * sdcard)
{
    unsigned char sdcard_rsp;
    register int  iter;

    iter       = 0;
    sdcard_rsp = _sdcard_receive_char(sdcard);
    while (
            (iter < SDCARD_COMMAND_TIMEOUT) &&
            !SDCARD_CHECK_R1_VALID(sdcard_rsp)
          )
    {
        sdcard_rsp = _sdcard_receive_char(sdcard);
        iter++;
    }

    return sdcard_rsp;
}

/**
 * \params  sdcard: Initialized pointer to the block device
 *
 * \return  void
 *
 * \brief   Wait data block start marker
 */
in_reset static void _sdcard_wait_data_block(struct sdcard_dev * sdcard)
{
	while (_sdcard_receive_char(sdcard) != 0xFE);
}

/**
 * \param   sdcard  : Initialized pointer to block device
 * \param   index   : SD card CMD index
 * \param   app     : Type of command, 0 for normal command or 1 for application specific
 * \param   args    : SD card CMD arguments
 *
 * \return  response first byte
 *
 * \brief   Send command to the SD card
 */
in_reset static int _sdcard_send_command    (
        struct sdcard_dev * sdcard ,
        int                 index  ,
        int                 app    ,
        void *              args   ,
        unsigned            crc7   )
{
    unsigned char sdcard_rsp;
    unsigned char * _args;

    _sdcard_gen_tick(sdcard, 5);  

    if (app == SDCARD_ACMD)
    {
        spi_put_tx(sdcard->spi, 0x40 | 55         , 0 );/* CMD and START bit */
        spi_put_tx(sdcard->spi, 0x00              , 0 );/* Argument[0]       */
        spi_put_tx(sdcard->spi, 0x00              , 0 );/* Argument[1]       */
        spi_put_tx(sdcard->spi, 0x00              , 0 );/* Argument[2]       */
        spi_put_tx(sdcard->spi, 0x00              , 0 );/* Argument[3]       */
        spi_put_tx(sdcard->spi, 0x01 | (crc7 << 1), 0 );/* END bit           */

        sdcard_rsp = _sdcard_wait_response(sdcard);
        if (SDCARD_CHECK_R1_ERROR(sdcard_rsp))
        {
            return sdcard_rsp;        
        }
    }

    _args = (unsigned char *) args;

    _sdcard_gen_tick(sdcard, 1);  

    spi_put_tx(sdcard->spi, 0x40 | index      , 0 );
    spi_put_tx(sdcard->spi, _args[0]          , 0 );
    spi_put_tx(sdcard->spi, _args[1]          , 0 );
    spi_put_tx(sdcard->spi, _args[2]          , 0 );
    spi_put_tx(sdcard->spi, _args[3]          , 0 );
    spi_put_tx(sdcard->spi, 0x01 | (crc7 << 1), 0 );

    return _sdcard_wait_response(sdcard);
}

in_reset int sdcard_dev_open(struct sdcard_dev * sdcard, struct spi_dev * spi, int ss)
{
    unsigned char args[4];
	unsigned char sdcard_rsp;
	unsigned int  iter;

	sdcard->spi      = spi;
    sdcard->slave_id = ss;

	/* 
     * Supply SD card ramp up time (min 74 cycles)
     */
	_sdcard_gen_tick(sdcard, 10);

    /* 
     * Assert slave select signal
     * Send CMD0 (Reset Command)
     * Deassert slave select signal
     */
	_sdcard_enable(sdcard);

    args[0] = 0;
    args[1] = 0;
    args[2] = 0;
    args[3] = 0;
	sdcard_rsp = _sdcard_send_command(sdcard, 0, SDCARD_CMD, args, 0x4A);
	if ( sdcard_rsp != 0x01 )
	{
		return sdcard_rsp;
	}

	_sdcard_disable(sdcard);
	_sdcard_enable(sdcard);

	iter = 0;
	while( iter++ < SDCARD_COMMAND_TIMEOUT )
	{
		sdcard_rsp = _sdcard_send_command(sdcard, 41, SDCARD_ACMD, args, 0x00);
		if( sdcard_rsp == 0x01 )
		{
			continue;
		}

		break;
	}

	_sdcard_disable(sdcard);

	return sdcard_rsp;
}

in_reset int sdcard_dev_read(struct sdcard_dev * sdcard, void * buf, size_t count)
{
    unsigned char args[4];
    unsigned char sdcard_rsp;
    register int  i;

    for (i = 0; i < 4; i++)
        args[i] = (sdcard->access_pointer >> (32 - (i+1)*8)) & 0xFF;

    _sdcard_enable(sdcard);

    sdcard_rsp = _sdcard_send_command(sdcard, 17, SDCARD_CMD, args, 0x00);
    if ( SDCARD_CHECK_R1_ERROR(sdcard_rsp) )
    {
        _sdcard_disable(sdcard);
        return sdcard_rsp;
    }

    _sdcard_wait_data_block(sdcard);

    for ( i = 0; i < count; i++ )
    {
        *((char *) buf + i) = _sdcard_receive_char(sdcard);
    }

    /*
     * Get the remainder of the block bytes and the CRC16 (comes
     * at the end of the data block)
     */
    while( i++ < (sdcard->block_length + 2) ) _sdcard_receive_char(sdcard);

    _sdcard_disable(sdcard);

    /*
     * Move the access pointer to the next block
     */
    sdcard->access_pointer += sdcard->block_length;

    return 0;
}

in_reset size_t sdcard_dev_write(struct sdcard_dev *sdcard, void * buf, size_t count)
{
	return 0;
}

in_reset void sdcard_dev_lseek(struct sdcard_dev * sdcard, size_t blk_pos)
{
    sdcard->access_pointer = sdcard->block_length * blk_pos;
}

in_reset size_t sdcard_dev_get_capacity(struct sdcard_dev * sdcard)
{
	return sdcard->capacity;
}

in_reset int sdcard_dev_set_blocklen(struct sdcard_dev * sdcard, size_t len)
{
    unsigned char args[4];
    unsigned char sdcard_rsp;
    register int i;

    for (i = 0; i < 4; i++)
        args[i] = (len >> (32 - (i+1)*8)) & 0xFF;

    _sdcard_enable(sdcard);

    sdcard_rsp = _sdcard_send_command(sdcard, 16, SDCARD_CMD, args, 0x00);
    if ( SDCARD_CHECK_R1_ERROR(sdcard_rsp) )
    {
        _sdcard_disable(sdcard);
        return sdcard_rsp;
    }

    _sdcard_disable(sdcard);

    sdcard->block_length = len;

	return 0;
}

#undef in_reset

/*
 * vim: tabstop=4 : shiftwidth=4 : expandtab : softtabstop=4
 */
