/**
 * \file    : boot_utils.c
 * \date    : August 2012
 * \author  : Cesar Fuguet
 *
 * Definition of some miscellaneous functions used in by the
 * pre-loader
 */

#include <boot_utils.h>

/**
 * memcpy( _dst, _src, size )
 *
 * Transfer data between to memory buffers.
 *
 * \param _dst   : Destination buffer base address 
 * \param _src   : Source buffer base address
 * \param size   : Number of bytes to transfer 
 *
 */
void * memcpy(void *_dst, const void *_src, unsigned int size)
{
    unsigned int *dst = _dst;
    const unsigned int *src = _src;
    if ( ! ((unsigned int)dst & 3) && ! ((unsigned int)src & 3) )
        while (size > 3) {
            *dst++ = *src++;
            size -= 4;
        }

    unsigned char *cdst = (unsigned char*) dst;
    unsigned char *csrc = (unsigned char*) src;

    while (size--) {
        *cdst++ = *csrc++;
    }
    return _dst;
}

/**
 * memset( _dst, value, size )
 *
 * Initialize memory buffers with predefined value.
 *
 * \param _dst   : Destination buffer base address 
 * \param value  : Initialization value 
 * \param size   : Number of bytes to initialize
 *
 */
void * memset(void *_dst, const int value, unsigned int size)
{
    char * dst = (char *) _dst;

    while(size--) *dst++ = (char) value;

    return _dst;
}

/*
 * Misc functions for the ELF format
 */

/**
 * boot_print_elf_phdr( elf_phdr_ptr )
 *
 * Print some fields of a ELF program header
 *
 * \param elf_phdr_ptr : Pointer to the ELF program header to print
 *
 */
void boot_print_elf_phdr(Elf32_Phdr * elf_phdr_ptr)
{
    boot_puts("- type   : ");
    boot_putx(elf_phdr_ptr->p_type);

    boot_puts("\n- offset : ");
    boot_putx(elf_phdr_ptr->p_offset);

    boot_puts("\n- vaddr  : ");
    boot_putx(elf_phdr_ptr->p_vaddr);

    boot_puts("\n- paddr  : ");
    boot_putx(elf_phdr_ptr->p_paddr);

    boot_puts("\n- filesz : ");
    boot_putx(elf_phdr_ptr->p_filesz);

    boot_puts("\n- memsz  : ");
    boot_putx(elf_phdr_ptr->p_memsz);

    boot_puts("\n- flags  : ");
    boot_putx(elf_phdr_ptr->p_flags);

    boot_puts("\n- align  : ");
    boot_putx(elf_phdr_ptr->p_align);
}

// vim: tabstop=4 : softtabstop=4 : shiftwidth=4 : expandtab
