///////////////////////////////////////////////////////////////////////////////
// File: top.cpp 
// Author: Alain Greiner 
// Copyright: UPMC/LIP6
// Date : august 2013
// This program is released under the GNU public license
///////////////////////////////////////////////////////////////////////////////
// This file define a generic TSAR architecture with an IO network emulating 
// an external bus (i.e. Hypertransport) to access external peripherals:
//
// - BROM : boot ROM
// - FBUF : Frame Buffer
// - MTTY : multi TTY (up to 15 channels)
// - MNIC : Network controller (up to 2 channels)
// - CDMA : Chained Buffer DMA controller (up to 4 channels)
// - BDEV : Dlock Device controler (1 channel)
// 
// The internal physical address space is 40 bits.
//
// It contains a 2D mesh of XMAX*YMAX clusters, and 3 networks:
//
// 1) the INT network supports Read/Write transactions
//    between processors and L2 caches or peripherals. 
//    (VCI ADDDRESS = 40 bits / VCI DATA width = 32 bits)
//    It supports also coherence transactions between L1 & L2 caches.
// 3) the RAM network is emulating the 3D network between L2 caches
//    and L3 caches, and is implemented as a 2D mesh between the L2 caches,
//    the two IO bridges and the physical RAMs disributed in all clusters.
//    (VCI ADDRESS = 40 bits / VCI DATA = 64 bits)
// 4) the IOX network connects the two IO bridge components to the
//    6 external peripheral controllers.
//    (VCI ADDDRESS = 40 bits / VCI DATA width = 64 bits)
// 
// The external peripherals IRQs are connected to the XICU component
// in cluster(0,0): therefore, the number of channels for the external 
// peripherals (MTTY, MNIC, CDMA) is limited by the number of IRQ ports...
//
// In all clusters, the IRQs are connected to XICU as follow:
// - IRQ_IN[0]  to IRQ_IN[3] not connected (reserved  for context switch)
// - IRQ_IN[4]  to IRQ_IN[7] are connected to 4 MDMA channels
//
// In external peripheral IRQs are connected in cluster(0,0) only:
// - IRQ_IN[8]  to IRQ_IN[9]  are connected to 2 NIC_RX channels.
// - IRQ_IN[10] to IRQ_IN[11] are connected to 2 NIC_TX channels.
// - IRQ_IN[12] to IRQ_IN[15] are connected to 4 CDMA channels
// - IRQ_IN[16] to IRQ_IN[30] are connected to 15 TTY channels
// - IRQ_IN[31]               is connected to BDEV
// 
// All clusters are identical, but cluster(0,0) and cluster(XMAX-1,YMAX-1)
// contain an extra IO bridge component. These IOB0 & IOB1 components are
// connected to the three networks (INT, RAM, IOX).
// The number of clusters cannot be larger than 256.
// The number of processors per cluster cannot be larger than 4.
// 
// - It uses two dspin_local_crossbar per cluster to implement the
//   local interconnect correponding to the INT network. 
// - It uses two dspin_local_crossbar per cluster to implement the
//   local interconnect correponding to the coherence INT network. 
// - It uses two virtual_dspin_router per cluster to implement
//   the INT network (routing both the direct and coherence trafic).
// - It uses two dspin_router per cluster to implement the RAM network.
// - It uses the vci_cc_vcache_wrapper.
// - It uses the vci_mem_cache.
// - It contains one vci_xicu and one vci_multi_dma per cluster.
// - It contains one vci_simple ram per cluster to model the L3 cache.
//
// The TsarIobCluster component is defined in files
// tsar_iob_cluster.* (with * = cpp, h, sd)
//
// The main hardware parameters must be defined in the hard_config.h file :
// - XMAX        : number of clusters in a row (power of 2)
// - YMAX        : number of clusters in a column (power of 2)
// - CLUSTER_SIZE     : size of the segment allocated to a cluster
// - NB_PROCS_MAX     : number of processors per cluster (power of 2)
// - NB_DMA_CHANNELS  : number of DMA channels per cluster (< 9)
// - NB_TTY_CHANNELS  : number of TTY channels in I/O network (< 16)
// - NB_NIC_CHANNELS  : number of NIC channels in I/O network (< 9)
// 
// Some secondary hardware parameters must be defined in this top.cpp file:
// - XRAM_LATENCY     : external ram latency 
// - MEMC_WAYS        : L2 cache number of ways
// - MEMC_SETS        : L2 cache number of sets
// - L1_IWAYS     
// - L1_ISETS    
// - L1_DWAYS   
// - L1_DSETS  
// - FBUF_X_SIZE      : width of frame buffer (pixels)
// - FBUF_Y_SIZE      : heigth of frame buffer (lines)
// - BDEV_SECTOR_SIZE : block size for block drvice
// - BDEV_IMAGE_NAME  : file pathname for block device 
// - NIC_RX_NAME      : file pathname for NIC received packets
// - NIC_TX_NAME      : file pathname for NIC transmited packets
// - NIC_TIMEOUT      : max number of cycles before closing a container
//
// General policy for 40 bits physical address decoding:
// All physical segments base addresses are multiple of 1 Mbytes
// (=> the 24 LSB bits = 0, and the 16 MSB bits define the target) 
// The (x_width + y_width) MSB bits (left aligned) define
// the cluster index, and the LADR bits define the local index:
//      | X_ID  | Y_ID  |---| LADR |     OFFSET          |
//      |x_width|y_width|---|  8   |       24            |
//
// General policy for 14 bits SRCID decoding:
// Each component is identified by (x_id, y_id, l_id) tuple.
//      | X_ID  | Y_ID  |---| L_ID |
//      |x_width|y_width|---|  6   |
/////////////////////////////////////////////////////////////////////////

#include <systemc>
#include <sys/time.h>
#include <iostream>
#include <sstream>
#include <cstdlib>
#include <cstdarg>
#include <stdint.h>

#include "gdbserver.h"
#include "mapping_table.h"

#include "tsar_iob_cluster.h"
#include "vci_chbuf_dma.h"
#include "vci_multi_tty.h"
#include "vci_multi_nic.h"
#include "vci_simple_rom.h"
#include "vci_block_device_tsar.h"
#include "vci_framebuffer.h"
#include "vci_iox_network.h"

#include "alloc_elems.h"

///////////////////////////////////////////////////
//      OS
///////////////////////////////////////////////////
#define USE_ALMOS 0

#define almos_bootloader_pathname "bootloader.bin"
#define almos_kernel_pathname     "kernel-soclib.bin@0xbfc10000:D"
#define almos_archinfo_pathname   "arch-info.bin@0xBFC08000:D"

///////////////////////////////////////////////////
//               Parallelisation
///////////////////////////////////////////////////
#define USE_OPENMP               0

#if USE_OPENMP
#include <omp.h>
#endif

///////////////////////////////////////////////////////////
//          DSPIN parameters           
///////////////////////////////////////////////////////////

#define dspin_int_cmd_width   39
#define dspin_int_rsp_width   32

#define dspin_ram_cmd_width   64
#define dspin_ram_rsp_width   64

///////////////////////////////////////////////////////////
//         VCI fields width  for the 3 VCI networks          
///////////////////////////////////////////////////////////

#define vci_cell_width_int    4
#define vci_cell_width_ext    8

#define vci_plen_width        8
#define vci_address_width     40
#define vci_rerror_width      1
#define vci_clen_width        1
#define vci_rflag_width       1
#define vci_srcid_width       14
#define vci_pktid_width       4
#define vci_trdid_width       4
#define vci_wrplen_width      1

////////////////////////////////////////////////////////////
//    Main Hardware Parameters values         
//////////////////////i/////////////////////////////////////

#include "giet_vm/hard_config.h"

////////////////////////////////////////////////////////////
//    Secondary Hardware Parameters values         
//////////////////////i/////////////////////////////////////

#define XMAX                  CLUSTER_X
#define YMAX                  CLUSTER_Y

#define XRAM_LATENCY          0

#define MEMC_WAYS             16
#define MEMC_SETS             256

#define L1_IWAYS              4
#define L1_ISETS              64

#define L1_DWAYS              4
#define L1_DSETS              64

#define FBUF_X_SIZE           128
#define FBUF_Y_SIZE           128

#define BDEV_SECTOR_SIZE      512
#define BDEV_IMAGE_NAME       "giet_vm/display/images.raw"

#define NIC_RX_NAME           "giet_vm/nic/rx_packets.txt"
#define NIC_TX_NAME           "giet_vm/nic/tx_packets.txt"
#define NIC_TIMEOUT           10000

#define NORTH                 0
#define SOUTH                 1
#define EAST                  2
#define WEST                  3

#define cluster(x,y)   ((y) + YMAX*(x))

////////////////////////////////////////////////////////////
//    Software to be loaded in ROM & RAM         
//////////////////////i/////////////////////////////////////

#define BOOT_SOFT_NAME        "giet_vm/soft.elf"

////////////////////////////////////////////////////////////
//     DEBUG Parameters default values         
//////////////////////i/////////////////////////////////////

#define MAX_FROZEN_CYCLES     10000

/////////////////////////////////////////////////////////
//    Physical segments definition
/////////////////////////////////////////////////////////

// Non replicated peripherals

#define BROM_BASE             0x00BFC00000      
#define BROM_SIZE             0x0000100000   // 1 M Kbytes

#define IOBX_BASE             0x00BE000000
#define IOBX_SIZE             0x0000001000   // 4 K Kbytes

#define BDEV_BASE             0x00B3000000      
#define BDEV_SIZE             0x0000001000   // 4 Kbytes

#define MTTY_BASE             0x00B4000000      
#define MTTY_SIZE             0x0000001000 * NB_TTY_CHANNELS  // 4 Kbytes

#define MNIC_BASE             0x00B5000000      
#define MNIC_SIZE             0x0000080000   // 512 Kbytes

#define CDMA_BASE             0x00B6000000      
#define CDMA_SIZE             0x0000001000 * (NB_NIC_CHANNELS * 2)  // 4 Kbytes per channel 

#define FBUF_BASE             0x00B7000000      
#define FBUF_SIZE             FBUF_X_SIZE * FBUF_Y_SIZE

// replicated segments : address is incremented by a cluster offset 
//     offset  = cluster(x,y) << (address_width-x_width-y_width);

#define XRAM_BASE             0x0000000000      
#define XRAM_SIZE             0x0010000000   // 256 Mbytes

#define XICU_BASE             0x00B0000000      
#define XICU_SIZE             0x0000001000   // 4 Kbytes

#define MDMA_BASE             0x00B1000000      
#define MDMA_SIZE             0x0000001000 * NB_DMA_CHANNELS // 4 Kbytes per channel  

#define MEMC_BASE             0x00B2000000      
#define MEMC_SIZE             0x0000001000   // 4 Kbytes 


////////////////////////////////////////////////////////////////////////
//          SRCID definition
////////////////////////////////////////////////////////////////////////
// All initiators are in the same indexing space (14 bits).
// The SRCID is structured in two fields:
// - The 10 MSB bits define the cluster index (left aligned)
// - The 4  LSB bits define the local index.
// Two different initiators cannot have the same SRCID, but a given 
// initiator can have two alias SRCIDs: 
// - Internal initiators (procs, mdma) are replicated in all clusters,
//   and each initiator has one single SRCID.
// - External initiators (bdev, cdma) are not replicated, but can be
//   accessed in 2 clusters : cluster_iob0 and cluster_iob1. 
//   They have the same local index, but two different cluster indexes.
// As cluster_iob0 and cluster_iob1 contain both internal initiators
// and external initiators, they must have different local indexes. 
// Consequence: For a local interconnect, the INI_ID port index
// is NOT equal to the SRCID local index, and the local interconnect
// must make a translation: SRCID => INI_ID (port index)
////////////////////////////////////////////////////////////////////////

#define PROC_LOCAL_SRCID             0    // from 0 to 7
#define MDMA_LOCAL_SRCID             8
#define IOBX_LOCAL_SRCID             9
#define BDEV_LOCAL_SRCID             10
#define CDMA_LOCAL_SRCID             11
#define MEMC_LOCAL_SRCID             12

////////////////////////////////////////////////////////////////////
//     TGT_ID and INI_ID port indexing for INT local interconnect
////////////////////////////////////////////////////////////////////

#define INT_MEMC_TGT_ID              0
#define INT_XICU_TGT_ID              1
#define INT_MDMA_TGT_ID              2
#define INT_IOBX_TGT_ID              3

#define INT_PROC_INI_ID              0   // from 0 to (NB_PROCS_MAX-1)
#define INT_MDMA_INI_ID              NB_PROCS_MAX
#define INT_IOBX_INI_ID              (NB_PROCS_MAX+1)

////////////////////////////////////////////////////////////////////
//     TGT_ID and INI_ID port indexing for RAM local interconnect
////////////////////////////////////////////////////////////////////

#define RAM_XRAM_TGT_ID              0

#define RAM_MEMC_INI_ID              0
#define RAM_IOBX_INI_ID              1

////////////////////////////////////////////////////////////////////
//     TGT_ID and INI_ID port indexing for I0X local interconnect
////////////////////////////////////////////////////////////////////

#define IOX_IOB0_TGT_ID              0    // don't change this value
#define IOX_IOB1_TGT_ID              1    // don't change this value
#define IOX_FBUF_TGT_ID              2
#define IOX_BDEV_TGT_ID              3
#define IOX_MNIC_TGT_ID              4
#define IOX_CDMA_TGT_ID              5
#define IOX_BROM_TGT_ID              6
#define IOX_MTTY_TGT_ID              7

#define IOX_IOB0_INI_ID              0    // Don't change this value 
#define IOX_IOB1_INI_ID              1    // Don't change this value 
#define IOX_BDEV_INI_ID              2     
#define IOX_CDMA_INI_ID              3  

/////////////////////////////////////////////////////////////////////
int _main(int argc, char *argv[])
/////////////////////////////////////////////////////////////////////
{
   using namespace sc_core;
   using namespace soclib::caba;
   using namespace soclib::common;


   char     soft_name[256]   = BOOT_SOFT_NAME;             // pathname: binary code
   size_t   ncycles          = 1000000000;                 // simulated cycles
   char     disk_name[256]   = BDEV_IMAGE_NAME;            // pathname: disk image
   char     nic_rx_name[256] = NIC_RX_NAME;                // pathname: rx packets file
   char     nic_tx_name[256] = NIC_TX_NAME;                // pathname: tx packets file
   ssize_t  threads_nr       = 1;                          // simulator's threads number
   bool     debug_ok         = false;                      // trace activated
   size_t   debug_period     = 1;                          // trace period
   size_t   debug_memc_id    = 0xFFFFFFFF;                 // index of traced memc   
   size_t   debug_proc_id    = 0xFFFFFFFF;                 // index of traced proc 
   bool     debug_iob        = false;                      // trace iob0 & iob1 when true
   uint32_t debug_from       = 0;                          // trace start cycle
   uint32_t frozen_cycles    = MAX_FROZEN_CYCLES;          // monitoring frozen processor
   size_t   cluster_iob0     = cluster(0,0);               // cluster containing IOB0
   size_t   cluster_iob1     = cluster(XMAX-1,YMAX-1);     // cluster containing IOB1
   size_t   block_size       = BDEV_SECTOR_SIZE;           // disk block size

   ////////////// command line arguments //////////////////////
   if (argc > 1)
   {
      for (int n = 1; n < argc; n = n + 2)
      {
         if ((strcmp(argv[n],"-NCYCLES") == 0) && (n+1<argc))
         {
            ncycles = atoi(argv[n+1]);
         }
         else if ((strcmp(argv[n],"-SOFT") == 0) && (n+1<argc) )
         {
            strcpy(soft_name, argv[n+1]);
         }
         else if ((strcmp(argv[n],"-DEBUG") == 0) && (n+1<argc) )
         {
            debug_ok = true;
            debug_from = atoi(argv[n+1]);
         }
         else if ((strcmp(argv[n],"-DISK") == 0) && (n+1<argc) )
         {
            strcpy(disk_name, argv[n+1]);
         }
         else if ((strcmp(argv[n],"-MEMCID") == 0) && (n+1<argc) )
         {
            debug_memc_id = atoi(argv[n+1]);
            assert( (debug_memc_id < (XMAX*YMAX) ) && 
                   "debug_memc_id larger than XMAX * YMAX" );
         }
         else if ((strcmp(argv[n],"-IOB") == 0) && (n+1<argc) )
         {
            debug_iob = atoi(argv[n+1]);
         }
         else if ((strcmp(argv[n],"-PROCID") == 0) && (n+1<argc) )
         {
            debug_proc_id = atoi(argv[n+1]);
            assert( (debug_proc_id < (XMAX * YMAX * NB_PROCS_MAX) ) && 
                   "debug_proc_id larger than XMAX * YMAX * NB_PROCS" );
         }
         else if ((strcmp(argv[n], "-THREADS") == 0) && ((n+1) < argc))
         {
            threads_nr = atoi(argv[n+1]);
            threads_nr = (threads_nr < 1) ? 1 : threads_nr;
         }
         else if ((strcmp(argv[n], "-FROZEN") == 0) && (n+1 < argc))
         {
            frozen_cycles = atoi(argv[n+1]);
         }
         else if ((strcmp(argv[n], "-PERIOD") == 0) && (n+1 < argc))
         {
            debug_period = atoi(argv[n+1]);
         }
         else
         {
            std::cout << "   Arguments are (key,value) couples." << std::endl;
            std::cout << "   The order is not important." << std::endl;
            std::cout << "   Accepted arguments are :" << std::endl << std::endl;
            std::cout << "     -SOFT pathname_for_embedded_soft" << std::endl;
            std::cout << "     -DISK pathname_for_disk_image" << std::endl;
            std::cout << "     -NCYCLES number_of_simulated_cycles" << std::endl;
            std::cout << "     -DEBUG debug_start_cycle" << std::endl;
            std::cout << "     -THREADS simulator's threads number" << std::endl;
            std::cout << "     -FROZEN max_number_of_lines" << std::endl;
            std::cout << "     -PERIOD number_of_cycles between trace" << std::endl;
            std::cout << "     -MEMCID index_memc_to_be_traced" << std::endl;
            std::cout << "     -PROCID index_proc_to_be_traced" << std::endl;
            std::cout << "     -IOBID  index_iob_to_be_traced" << std::endl;
            exit(0);
         }
      }
   }

   // checking hardware parameters
   assert( ( (XMAX == 1) or (XMAX == 2) or (XMAX == 4) or
             (XMAX == 8) or (XMAX == 16) ) and
           "The XMAX parameter must be 1, 2, 4, 8 or 16" );

   assert( ( (YMAX == 1) or (YMAX == 2) or (YMAX == 4) or
             (YMAX == 8) or (YMAX == 16) ) and
           "The YMAX parameter must be 1, 2, 4, 8 or 16" );

   assert( ( (NB_PROCS_MAX == 1) or (NB_PROCS_MAX == 2) or (NB_PROCS_MAX == 4) ) and
           "The NB_PROCS_MAX parameter must be 1, 2, 4" );

   assert( (NB_DMA_CHANNELS == 4) and
           "The NB_DMA_CHANNELS parameter must be 4" );

   assert( (NB_TTY_CHANNELS < 16) and
           "The NB_TTY_CHANNELS parameter must be smaller than 16" );

   assert( (NB_NIC_CHANNELS == 2) and
           "The NB_NIC_CHANNELS parameter must be 2" );

   std::cout << std::endl;
   std::cout << " - XMAX            = " << XMAX << std::endl;
   std::cout << " - YMAX            = " << YMAX << std::endl;
   std::cout << " - NB_PROCS_MAX    = " << NB_PROCS_MAX <<  std::endl;
   std::cout << " - NB_DMA_CHANNELS = " << NB_DMA_CHANNELS <<  std::endl;
   std::cout << " - NB_TTY_CHANNELS = " << NB_TTY_CHANNELS <<  std::endl;
   std::cout << " - NB_NIC_CHANNELS = " << NB_NIC_CHANNELS <<  std::endl;
   std::cout << " - MEMC_WAYS       = " << MEMC_WAYS << std::endl;
   std::cout << " - MEMC_SETS       = " << MEMC_SETS << std::endl;
   std::cout << " - RAM_LATENCY     = " << XRAM_LATENCY << std::endl;
   std::cout << " - MAX_FROZEN      = " << frozen_cycles << std::endl;

   std::cout << std::endl;

#if USE_OPENMP
   omp_set_dynamic(false);
   omp_set_num_threads(threads_nr);
   std::cerr << "Built with openmp version " << _OPENMP << std::endl;
#endif

   // Define VciParams objects
   typedef soclib::caba::VciParams<vci_cell_width_int,
                                   vci_plen_width,
                                   vci_address_width,
                                   vci_rerror_width,
                                   vci_clen_width,
                                   vci_rflag_width,
                                   vci_srcid_width,
                                   vci_pktid_width,
                                   vci_trdid_width,
                                   vci_wrplen_width> vci_param_int;

   typedef soclib::caba::VciParams<vci_cell_width_ext,
                                   vci_plen_width,
                                   vci_address_width,
                                   vci_rerror_width, 
                                   vci_clen_width,
                                   vci_rflag_width,
                                   vci_srcid_width,
                                   vci_pktid_width,
                                   vci_trdid_width,
                                   vci_wrplen_width> vci_param_ext;

   // Define parameters depending on mesh size
   size_t   x_width;
   size_t   y_width;

   if      (XMAX == 1) x_width = 0;
   else if (XMAX == 2) x_width = 1;
   else if (XMAX <= 4) x_width = 2;
   else if (XMAX <= 8) x_width = 3;
   else                x_width = 4;

   if      (YMAX == 1) y_width = 0;
   else if (YMAX == 2) y_width = 1;
   else if (YMAX <= 4) y_width = 2;
   else if (YMAX <= 8) y_width = 3;
   else                y_width = 4;

   /////////////////////////////////////////////////////////////////////
   // INT network mapping table
   // - two levels address decoding for commands
   // - two levels srcid decoding for responses
   // - NB_PROCS_MAX + 2 (MDMA, IOBX) local initiators per cluster
   // - 4 local targets (MEMC, XICU, MDMA, IOBX) per cluster
   /////////////////////////////////////////////////////////////////////
   MappingTable maptab_int( vci_address_width, 
                            IntTab(x_width + y_width, 16 - x_width - y_width), 
                            IntTab(x_width + y_width, vci_srcid_width - x_width - y_width), 
                            0x00FF000000);

   for (size_t x = 0; x < XMAX; x++)
   {
      for (size_t y = 0; y < YMAX; y++)
      {
         uint64_t offset = ((uint64_t)cluster(x,y)) 
                              << (vci_address_width-x_width-y_width);
         bool config = true;

         // the four following segments are defined in all clusters

         std::ostringstream    smemc_conf;
         smemc_conf << "int_seg_memc_conf_" << x << "_" << y;
         maptab_int.add(Segment(smemc_conf.str(), MEMC_BASE+offset, MEMC_SIZE,
                     IntTab(cluster(x,y),INT_MEMC_TGT_ID), true, config ));

         std::ostringstream    smemc_xram;
         smemc_xram << "int_seg_memc_xram_" << x << "_" << y;
         maptab_int.add(Segment(smemc_xram.str(), XRAM_BASE+offset, XRAM_SIZE,
                     IntTab(cluster(x,y),INT_MEMC_TGT_ID), true));

         std::ostringstream    sxicu;
         sxicu << "int_seg_xicu_" << x << "_" << y;
         maptab_int.add(Segment(sxicu.str(), XICU_BASE+offset, XICU_SIZE, 
                     IntTab(cluster(x,y),INT_XICU_TGT_ID), false));

         std::ostringstream    smdma;
         smdma << "int_seg_mdma_" << x << "_" << y;
         maptab_int.add(Segment(smdma.str(), MDMA_BASE+offset, MDMA_SIZE, 
                     IntTab(cluster(x,y),INT_MDMA_TGT_ID), false));

         // the following segments are only defined in cluster_iob0 or in cluster_iob1

         if ( (cluster(x,y) == cluster_iob0) or (cluster(x,y) == cluster_iob1) )  
         {
            std::ostringstream    siobx;
            siobx << "int_seg_iobx_" << x << "_" << y;
            maptab_int.add(Segment(siobx.str(), IOBX_BASE+offset, IOBX_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false, config ));

            std::ostringstream    stty;
            stty << "int_seg_mtty_" << x << "_" << y;
            maptab_int.add(Segment(stty.str(), MTTY_BASE+offset, MTTY_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false));

            std::ostringstream    sfbf;
            sfbf << "int_seg_fbuf_" << x << "_" << y;
            maptab_int.add(Segment(sfbf.str(), FBUF_BASE+offset, FBUF_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false));

            std::ostringstream    sbdv;
            sbdv << "int_seg_bdev_" << x << "_" << y;
            maptab_int.add(Segment(sbdv.str(), BDEV_BASE+offset, BDEV_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false));

            std::ostringstream    snic;
            snic << "int_seg_mnic_" << x << "_" << y;
            maptab_int.add(Segment(snic.str(), MNIC_BASE+offset, MNIC_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false));

            std::ostringstream    srom;
            srom << "int_seg_brom_" << x << "_" << y;
            maptab_int.add(Segment(srom.str(), BROM_BASE+offset, BROM_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), true));

            std::ostringstream    sdma;
            sdma << "int_seg_cdma_" << x << "_" << y;
            maptab_int.add(Segment(sdma.str(), CDMA_BASE+offset, CDMA_SIZE, 
                        IntTab(cluster(x,y), INT_IOBX_TGT_ID), false));
         }

         // This define the mapping between the SRCIDs
         // and the port index on the local interconnect.

         maptab_int.srcid_map( IntTab( cluster(x,y), MDMA_LOCAL_SRCID ), INT_MDMA_INI_ID );

         maptab_int.srcid_map( IntTab( cluster(x,y), IOBX_LOCAL_SRCID ), INT_IOBX_INI_ID );

         for ( size_t p = 0 ; p < NB_PROCS_MAX ; p++ )
         maptab_int.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+p ), INT_PROC_INI_ID+p );
      }
   }
   std::cout << "INT network " << maptab_int << std::endl;

    /////////////////////////////////////////////////////////////////////////
    // RAM network mapping table 
    // - two levels address decoding for commands
    // - two levels srcid decoding for responses
    // - 2 local initiators (MEMC, IOBX) per cluster 
    //   (IOBX component only in cluster_iob0 and cluster_iob1)
    // - 1 local target (XRAM) per cluster
    ////////////////////////////////////////////////////////////////////////
    MappingTable maptab_ram( vci_address_width,
                             IntTab(x_width+y_width, 16 - x_width - y_width),
                             IntTab(x_width+y_width, vci_srcid_width - x_width - y_width), 
                             0x00FF000000);

    for (size_t x = 0; x < XMAX; x++)
    {
        for (size_t y = 0; y < YMAX ; y++)
        { 
            uint64_t offset = ((uint64_t)cluster(x,y)) 
                                << (vci_address_width-x_width-y_width);

            std::ostringstream sxram;
            sxram << "ext_seg_xram_" << x << "_" << y;
            maptab_ram.add(Segment(sxram.str(), XRAM_BASE+offset, 
                           XRAM_SIZE, IntTab(cluster(x,y), 0), false));
        }
    }

    // This define the mapping between the initiators (identified by their
    // global SRCID) and the port index on the RAM local interconnect.
    // External initiator have two alias SRCID (iob0 / iob1)  

    maptab_ram.srcid_map( IntTab( cluster_iob0, CDMA_LOCAL_SRCID ), RAM_IOBX_INI_ID );
    maptab_ram.srcid_map( IntTab( cluster_iob1, CDMA_LOCAL_SRCID ), RAM_IOBX_INI_ID );

    maptab_ram.srcid_map( IntTab( cluster_iob0, BDEV_LOCAL_SRCID ), RAM_IOBX_INI_ID );
    maptab_ram.srcid_map( IntTab( cluster_iob1, BDEV_LOCAL_SRCID ), RAM_IOBX_INI_ID );

    maptab_ram.srcid_map( IntTab( cluster_iob1, MEMC_LOCAL_SRCID ), RAM_MEMC_INI_ID );

    std::cout << "RAM network " << maptab_ram << std::endl;

    ///////////////////////////////////////////////////////////////////////
    // IOX network mapping table  
    // - two levels address decoding for commands
    // - two levels srcid decoding for responses
    // - 4 initiators (IOB0, IOB1, BDEV, CDMA)
    // - 8 targets (IOB0, IOB1, BDEV, CDMA, MTTY, FBUF, BROM, MNIC)
    ///////////////////////////////////////////////////////////////////////
    MappingTable maptab_iox( vci_address_width, 
                             IntTab(x_width+y_width, 16 - x_width - y_width),
                             IntTab(x_width+y_width, vci_srcid_width - x_width - y_width), 
                             0x00FF000000);

    // compute base addresses for cluster_iob0 and cluster_iob1
    uint64_t iob0_base = ((uint64_t)cluster_iob0) << (vci_address_width - x_width - y_width); 
    uint64_t iob1_base = ((uint64_t)cluster_iob1) << (vci_address_width - x_width - y_width); 

    // Each peripheral can be accessed through two segments,
    // depending on the used IOB (IOB0 or IOB1).
    maptab_iox.add(Segment("iox_seg_mtty_0", MTTY_BASE + iob0_base, MTTY_SIZE, 
                   IntTab(cluster_iob0,IOX_MTTY_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_mtty_1", MTTY_BASE + iob1_base, MTTY_SIZE, 
                   IntTab(cluster_iob1,IOX_MTTY_TGT_ID), false));

    maptab_iox.add(Segment("iox_seg_fbuf_0", FBUF_BASE + iob0_base, FBUF_SIZE, 
                   IntTab(cluster_iob0,IOX_FBUF_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_fbuf_1", FBUF_BASE + iob1_base, FBUF_SIZE, 
                   IntTab(cluster_iob1,IOX_FBUF_TGT_ID), false));

    maptab_iox.add(Segment("iox_seg_bdev_0", BDEV_BASE + iob0_base, BDEV_SIZE, 
                   IntTab(cluster_iob0,IOX_BDEV_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_bdev_1", BDEV_BASE + iob1_base, BDEV_SIZE, 
                   IntTab(cluster_iob1,IOX_BDEV_TGT_ID), false));

    maptab_iox.add(Segment("iox_seg_mnic_0", MNIC_BASE + iob0_base, MNIC_SIZE, 
                   IntTab(cluster_iob0,IOX_MNIC_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_mnic_1", MNIC_BASE + iob1_base, MNIC_SIZE, 
                   IntTab(cluster_iob1,IOX_MNIC_TGT_ID), false));

    maptab_iox.add(Segment("iox_seg_cdma_0", CDMA_BASE + iob0_base, CDMA_SIZE, 
                   IntTab(cluster_iob0,IOX_CDMA_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_cdma_1", CDMA_BASE + iob1_base, CDMA_SIZE, 
                   IntTab(cluster_iob1,IOX_CDMA_TGT_ID), false));

    maptab_iox.add(Segment("iox_seg_brom_0", BROM_BASE + iob0_base, BROM_SIZE, 
                   IntTab(cluster_iob0,IOX_BROM_TGT_ID), false));
    maptab_iox.add(Segment("iox_seg_brom_1", BROM_BASE + iob1_base, BROM_SIZE, 
                   IntTab(cluster_iob1,IOX_BROM_TGT_ID), false));

    // Each physical RAM can be accessed through IOB0, or through IOB1.
    // if IOMMU is not activated, addresses are 40 bits (physical addresses),
    // and the choice depends on on address bit A[39].
    // if IOMMU is activated the addresses use only 32 bits (virtual addresses),
    // and the choice depends on address bit A[31].
    for (size_t x = 0; x < XMAX; x++)
    {
        for (size_t y = 0; y < YMAX ; y++)
        { 
            uint64_t offset = ((uint64_t)cluster(x,y)) 
                             << (vci_address_width-x_width-y_width);

            if ( x < (XMAX/2) ) // send command to XRAM through IOB0
            {
                std::ostringstream siob0;
                siob0 << "iox_seg_xram_" << x << "_" << y;
                maptab_iox.add(Segment(siob0.str(), offset, 0x80000000, 
                            IntTab(cluster_iob0,IOX_IOB0_TGT_ID), false));
            }
            else                // send command to XRAM through IOB1
            {
                std::ostringstream siob1;
                siob1 << "iox_seg_xram_" << x << "_" << y;
                maptab_iox.add(Segment(siob1.str(), offset, 0x80000000, 
                            IntTab(cluster_iob1,IOX_IOB1_TGT_ID), false));
            }
        }
    }
    // useful when IOMMU activated
    maptab_iox.add(Segment("iox_seg_xram    ", 0xc0000000, 0x40000000, 
                          IntTab(cluster_iob1,IOX_IOB1_TGT_ID), false));

    // This define the mapping between the initiators (identified by the SRCID)
    // and the port index on the IOX local interconnect.
    // External initiator have two alias SRCID (iob0 / iob1 access)  

    maptab_iox.srcid_map( IntTab( cluster_iob0, CDMA_LOCAL_SRCID ), IOX_CDMA_INI_ID );
    maptab_iox.srcid_map( IntTab( cluster_iob1, CDMA_LOCAL_SRCID ), IOX_CDMA_INI_ID );

    maptab_iox.srcid_map( IntTab( cluster_iob0, BDEV_LOCAL_SRCID ), IOX_BDEV_INI_ID );
    maptab_iox.srcid_map( IntTab( cluster_iob1, BDEV_LOCAL_SRCID ), IOX_BDEV_INI_ID );

    for (size_t x = 0; x < XMAX; x++)
    {
        for (size_t y = 0; y < YMAX ; y++)
        { 
            if ( x < (XMAX/2) )   // send response to proc or mdma through IOB0
            {
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID   ), IOX_IOB0_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+1 ), IOX_IOB0_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+2 ), IOX_IOB0_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+3 ), IOX_IOB0_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), MDMA_LOCAL_SRCID   ), IOX_IOB0_INI_ID );
            }
            else                  // send response to proc or mdma through IOB1
            {
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID   ), IOX_IOB1_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+1 ), IOX_IOB1_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+2 ), IOX_IOB1_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), PROC_LOCAL_SRCID+3 ), IOX_IOB1_INI_ID );
            maptab_iox.srcid_map( IntTab( cluster(x,y), MDMA_LOCAL_SRCID   ), IOX_IOB1_INI_ID );
            }
        }
    }

    std::cout << "IOX network " << maptab_iox << std::endl;

    ////////////////////
    // Signals
    ///////////////////

    sc_clock                    signal_clk("clk");
    sc_signal<bool>             signal_resetn("resetn");

    sc_signal<bool>             signal_unused_irq[32];
    sc_signal<bool>             signal_irq_bdev;
    sc_signal<bool>             signal_irq_mnic_rx[NB_NIC_CHANNELS];
    sc_signal<bool>             signal_irq_mnic_tx[NB_NIC_CHANNELS];
    sc_signal<bool>             signal_irq_mtty[NB_TTY_CHANNELS];
    sc_signal<bool>             signal_irq_cdma[NB_NIC_CHANNELS*2];

    // VCI signals for IOX network
    VciSignals<vci_param_ext>   signal_vci_ini_iob0("signal_vci_ini_iob0");
    VciSignals<vci_param_ext>   signal_vci_ini_iob1("signal_vci_ini_iob1");
    VciSignals<vci_param_ext>   signal_vci_ini_bdev("signal_vci_ini_bdev");
    VciSignals<vci_param_ext>   signal_vci_ini_cdma("signal_vci_ini_cdma");

    VciSignals<vci_param_ext>   signal_vci_tgt_iob0("signal_vci_tgt_iob0");
    VciSignals<vci_param_ext>   signal_vci_tgt_iob1("signal_vci_tgt_iob1");
    VciSignals<vci_param_ext>   signal_vci_tgt_mtty("signal_vci_tgt_mtty");
    VciSignals<vci_param_ext>   signal_vci_tgt_fbuf("signal_vci_tgt_fbuf");
    VciSignals<vci_param_ext>   signal_vci_tgt_mnic("signal_vci_tgt_mnic");
    VciSignals<vci_param_ext>   signal_vci_tgt_brom("signal_vci_tgt_brom");
    VciSignals<vci_param_ext>   signal_vci_tgt_bdev("signal_vci_tgt_bdev");
    VciSignals<vci_param_ext>   signal_vci_tgt_cdma("signal_vci_tgt_cdma");

   // Horizontal inter-clusters INT network DSPIN 
   DspinSignals<dspin_int_cmd_width>*** signal_dspin_int_cmd_h_inc =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_int_cmd_h_inc", XMAX-1, YMAX, 3);
   DspinSignals<dspin_int_cmd_width>*** signal_dspin_int_cmd_h_dec =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_int_cmd_h_dec", XMAX-1, YMAX, 3);
   DspinSignals<dspin_int_rsp_width>*** signal_dspin_int_rsp_h_inc =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_int_rsp_h_inc", XMAX-1, YMAX, 2);
   DspinSignals<dspin_int_rsp_width>*** signal_dspin_int_rsp_h_dec =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_int_rsp_h_dec", XMAX-1, YMAX, 2);

   // Vertical inter-clusters INT network DSPIN
   DspinSignals<dspin_int_cmd_width>*** signal_dspin_int_cmd_v_inc =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_int_cmd_v_inc", XMAX, YMAX-1, 3);
   DspinSignals<dspin_int_cmd_width>*** signal_dspin_int_cmd_v_dec =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_int_cmd_v_dec", XMAX, YMAX-1, 3);
   DspinSignals<dspin_int_rsp_width>*** signal_dspin_int_rsp_v_inc =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_int_rsp_v_inc", XMAX, YMAX-1, 2);
   DspinSignals<dspin_int_rsp_width>*** signal_dspin_int_rsp_v_dec =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_int_rsp_v_dec", XMAX, YMAX-1, 2);

   // Mesh boundaries INT network DSPIN 
   DspinSignals<dspin_int_cmd_width>**** signal_dspin_false_int_cmd_in =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_false_int_cmd_in", XMAX, YMAX, 4, 3);
   DspinSignals<dspin_int_cmd_width>**** signal_dspin_false_int_cmd_out =
      alloc_elems<DspinSignals<dspin_int_cmd_width> >("signal_dspin_false_int_cmd_out", XMAX, YMAX, 4, 3);
   DspinSignals<dspin_int_rsp_width>**** signal_dspin_false_int_rsp_in =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_false_int_rsp_in", XMAX, YMAX, 4, 2);
   DspinSignals<dspin_int_rsp_width>**** signal_dspin_false_int_rsp_out =
      alloc_elems<DspinSignals<dspin_int_rsp_width> >("signal_dspin_false_int_rsp_out", XMAX, YMAX, 4, 2);


   // Horizontal inter-clusters RAM network DSPIN 
   DspinSignals<dspin_ram_cmd_width>** signal_dspin_ram_cmd_h_inc =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_ram_cmd_h_inc", XMAX-1, YMAX);
   DspinSignals<dspin_ram_cmd_width>** signal_dspin_ram_cmd_h_dec =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_ram_cmd_h_dec", XMAX-1, YMAX);
   DspinSignals<dspin_ram_rsp_width>** signal_dspin_ram_rsp_h_inc =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_ram_rsp_h_inc", XMAX-1, YMAX);
   DspinSignals<dspin_ram_rsp_width>** signal_dspin_ram_rsp_h_dec =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_ram_rsp_h_dec", XMAX-1, YMAX);

   // Vertical inter-clusters RAM network DSPIN
   DspinSignals<dspin_ram_cmd_width>** signal_dspin_ram_cmd_v_inc =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_ram_cmd_v_inc", XMAX, YMAX-1);
   DspinSignals<dspin_ram_cmd_width>** signal_dspin_ram_cmd_v_dec =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_ram_cmd_v_dec", XMAX, YMAX-1);
   DspinSignals<dspin_ram_rsp_width>** signal_dspin_ram_rsp_v_inc =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_ram_rsp_v_inc", XMAX, YMAX-1);
   DspinSignals<dspin_ram_rsp_width>** signal_dspin_ram_rsp_v_dec =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_ram_rsp_v_dec", XMAX, YMAX-1);

   // Mesh boundaries RAM network DSPIN 
   DspinSignals<dspin_ram_cmd_width>*** signal_dspin_false_ram_cmd_in =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_false_ram_cmd_in", XMAX, YMAX, 4);
   DspinSignals<dspin_ram_cmd_width>*** signal_dspin_false_ram_cmd_out =
      alloc_elems<DspinSignals<dspin_ram_cmd_width> >("signal_dspin_false_ram_cmd_out", XMAX, YMAX, 4);
   DspinSignals<dspin_ram_rsp_width>*** signal_dspin_false_ram_rsp_in =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_false_ram_rsp_in", XMAX, YMAX, 4);
   DspinSignals<dspin_ram_rsp_width>*** signal_dspin_false_ram_rsp_out =
      alloc_elems<DspinSignals<dspin_ram_rsp_width> >("signal_dspin_false_ram_rsp_out", XMAX, YMAX, 4);

   ////////////////////////////
   //      Loader    
   ////////////////////////////

#if USE_ALMOS
   soclib::common::Loader loader(almos_bootloader_pathname,
                                 almos_archinfo_pathname,
                                 almos_kernel_pathname);
#else
   soclib::common::Loader loader(soft_name);
#endif

   typedef soclib::common::GdbServer<soclib::common::Mips32ElIss> proc_iss;
   proc_iss::set_loader(loader);

   ////////////////////////////////////////
   //  Instanciated Hardware Components
   ////////////////////////////////////////

   std::cout << std::endl << "External Bus and Peripherals" << std::endl << std::endl;

   // IOX network
   VciIoxNetwork<vci_param_ext>* iox_network;
   iox_network = new VciIoxNetwork<vci_param_ext>( "iox_network",  
                                                   maptab_iox,
                                                   0,        // cluster_id
                                                   8,        // number of targets
                                                   4 );      // number of initiators
   // boot ROM
   VciSimpleRom<vci_param_ext>*  iox_brom;
   iox_brom = new VciSimpleRom<vci_param_ext>( "iox_brom",
                                               IntTab(0, IOX_BROM_TGT_ID),
                                               maptab_iox,
                                               loader );
   // Network Controller
   VciMultiNic<vci_param_ext>*  iox_mnic;
   iox_mnic = new VciMultiNic<vci_param_ext>( "iox_mnic",
                                              IntTab(0, IOX_MNIC_TGT_ID),
                                              maptab_iox,
                                              NB_NIC_CHANNELS,
                                              nic_rx_name,
                                              nic_tx_name,
                                              0,           // mac_4 address
                                              0 );         // mac_2 address

   // Frame Buffer
   VciFrameBuffer<vci_param_ext>*  iox_fbuf;
   iox_fbuf = new VciFrameBuffer<vci_param_ext>( "iox_fbuf",
                                                 IntTab(0, IOX_FBUF_TGT_ID),
                                                 maptab_iox,
                                                 FBUF_X_SIZE, FBUF_Y_SIZE );

   // Block Device
   VciBlockDeviceTsar<vci_param_ext>*  iox_bdev;
   iox_bdev = new VciBlockDeviceTsar<vci_param_ext>( "iox_bdev",
                                                     maptab_iox,
                                                     IntTab(0, BDEV_LOCAL_SRCID),
                                                     IntTab(0, IOX_BDEV_TGT_ID),
                                                     disk_name,
                                                     block_size,
                                                     64);        // burst size (bytes)

   // Chained Buffer DMA controller
   VciChbufDma<vci_param_ext>*  iox_cdma;
   iox_cdma = new VciChbufDma<vci_param_ext>( "iox_cdma",
                                              maptab_iox,
                                              IntTab(0, CDMA_LOCAL_SRCID),
                                              IntTab(0, IOX_CDMA_TGT_ID),
                                              64,          // burst size (bytes)
                                              2*NB_NIC_CHANNELS );
   // Multi-TTY controller
   std::vector<std::string> vect_names;
   for( size_t tid = 0 ; tid < NB_TTY_CHANNELS ; tid++ )
   {
      std::ostringstream term_name;
      term_name <<  "term" << tid;
      vect_names.push_back(term_name.str().c_str());
   }
   VciMultiTty<vci_param_ext>*  iox_mtty;
   iox_mtty = new VciMultiTty<vci_param_ext>( "iox_mtty",
                                              IntTab(0, IOX_MTTY_TGT_ID),
                                              maptab_iox, 
                                              vect_names);
   // Clusters 
   TsarIobCluster<vci_param_int,
                  vci_param_ext,
                  dspin_int_cmd_width,
                  dspin_int_rsp_width,
                  dspin_ram_cmd_width,
                  dspin_ram_rsp_width>* clusters[XMAX][YMAX];

#if USE_OPENMP
#pragma omp parallel
    {
#pragma omp for
#endif
        for(size_t i = 0; i  < (XMAX * YMAX); i++)
        {
            size_t x = i / YMAX;
            size_t y = i % YMAX;

#if USE_OPENMP
#pragma omp critical
            {
#endif
            std::cout << std::endl;
            std::cout << "Cluster_" << std::dec << x << "_" << y << std::endl;
            std::cout << std::endl;

            std::ostringstream sc;
            sc << "cluster_" << x << "_" << y;
            clusters[x][y] = new TsarIobCluster<vci_param_int,
                                                vci_param_ext,
                                                dspin_int_cmd_width,
                                                dspin_int_rsp_width,
                                                dspin_ram_cmd_width,
                                                dspin_ram_rsp_width>
            (
                sc.str().c_str(),
                NB_PROCS_MAX,
                NB_DMA_CHANNELS,
                x,
                y,
                XMAX,
                YMAX,

                maptab_int,
                maptab_ram,
                maptab_iox,

                x_width,
                y_width,
                vci_srcid_width - x_width - y_width,            // l_id width,

                INT_MEMC_TGT_ID,
                INT_XICU_TGT_ID,
                INT_MDMA_TGT_ID,
                INT_IOBX_TGT_ID,

                INT_PROC_INI_ID,
                INT_MDMA_INI_ID,
                INT_IOBX_INI_ID,

                RAM_XRAM_TGT_ID,

                RAM_MEMC_INI_ID,
                RAM_MEMC_INI_ID,

                MEMC_WAYS,
                MEMC_SETS,
                L1_IWAYS,
                L1_ISETS,
                L1_DWAYS,
                L1_DSETS,
                XRAM_LATENCY,

                loader,

                frozen_cycles,
                debug_from,
                debug_ok and (cluster(x,y) == debug_memc_id),
                debug_ok and (cluster(x,y) == debug_proc_id),
                debug_ok and debug_iob
            );

#if USE_OPENMP
            } // end critical
#endif
        } // end for
#if USE_OPENMP
    }
#endif

    std::cout << std::endl;

    ///////////////////////////////////////////////////////////////////////////////
    //     Net-list 
    ///////////////////////////////////////////////////////////////////////////////

    // IOX network connexion
    iox_network->p_clk                                       (signal_clk);
    iox_network->p_resetn                                    (signal_resetn);
    iox_network->p_to_ini[IOX_IOB0_INI_ID]                   (signal_vci_ini_iob0);
    iox_network->p_to_ini[IOX_IOB1_INI_ID]                   (signal_vci_ini_iob1);
    iox_network->p_to_ini[IOX_BDEV_INI_ID]                   (signal_vci_ini_bdev);
    iox_network->p_to_ini[IOX_CDMA_INI_ID]                   (signal_vci_ini_cdma);
    iox_network->p_to_tgt[IOX_IOB0_TGT_ID]                   (signal_vci_tgt_iob0);
    iox_network->p_to_tgt[IOX_IOB1_TGT_ID]                   (signal_vci_tgt_iob1);
    iox_network->p_to_tgt[IOX_MTTY_TGT_ID]                   (signal_vci_tgt_mtty);
    iox_network->p_to_tgt[IOX_FBUF_TGT_ID]                   (signal_vci_tgt_fbuf);
    iox_network->p_to_tgt[IOX_MNIC_TGT_ID]                   (signal_vci_tgt_mnic);
    iox_network->p_to_tgt[IOX_BROM_TGT_ID]                   (signal_vci_tgt_brom);
    iox_network->p_to_tgt[IOX_BDEV_TGT_ID]                   (signal_vci_tgt_bdev);
    iox_network->p_to_tgt[IOX_CDMA_TGT_ID]                   (signal_vci_tgt_cdma);

    // BDEV connexion
	 iox_bdev->p_clk                                          (signal_clk);
    iox_bdev->p_resetn                                       (signal_resetn);
    iox_bdev->p_irq                                          (signal_irq_bdev);
    iox_bdev->p_vci_target                                   (signal_vci_tgt_bdev);
    iox_bdev->p_vci_initiator                                (signal_vci_ini_bdev);

    std::cout << "  - BDEV connected" << std::endl;

    // FBUF connexion
    iox_fbuf->p_clk                                          (signal_clk);
    iox_fbuf->p_resetn                                       (signal_resetn);
    iox_fbuf->p_vci                                          (signal_vci_tgt_fbuf);

    std::cout << "  - FBUF connected" << std::endl;

    // MNIC connexion
    iox_mnic->p_clk                                          (signal_clk);
    iox_mnic->p_resetn                                       (signal_resetn);
    iox_mnic->p_vci                                          (signal_vci_tgt_mnic);
    for ( size_t i=0 ; i<NB_NIC_CHANNELS ; i++ )
    {
         iox_mnic->p_rx_irq[i]                               (signal_irq_mnic_rx[i]);
         iox_mnic->p_tx_irq[i]                               (signal_irq_mnic_tx[i]);
    }

    std::cout << "  - MNIC connected" << std::endl;

    // BROM connexion
    iox_brom->p_clk                                          (signal_clk);
    iox_brom->p_resetn                                       (signal_resetn);
    iox_brom->p_vci                                          (signal_vci_tgt_brom);

    std::cout << "  - BROM connected" << std::endl;

    // MTTY connexion
    iox_mtty->p_clk                                          (signal_clk);
    iox_mtty->p_resetn                                       (signal_resetn);
    iox_mtty->p_vci                                          (signal_vci_tgt_mtty);
    for ( size_t i=0 ; i<NB_TTY_CHANNELS ; i++ )
    {
        iox_mtty->p_irq[i]              	                   (signal_irq_mtty[i]);
    }

    std::cout << "  - MTTY connected" << std::endl;

    // CDMA connexion
    iox_cdma->p_clk                       	                (signal_clk);
    iox_cdma->p_resetn                    	                (signal_resetn);
    iox_cdma->p_vci_target                	                (signal_vci_tgt_cdma);
    iox_cdma->p_vci_initiator             	                (signal_vci_ini_cdma);
    for ( size_t i=0 ; i<(NB_NIC_CHANNELS*2) ; i++)
    {
        iox_cdma->p_irq[i]                                   (signal_irq_cdma[i]);
    }

    std::cout << "  - CDMA connected" << std::endl;

    // IRQ connexions (for cluster_iob0 only)
    // IRQ_MNIC_RX  -> IRQ[08] to IRQ[09]
    // IRQ_MNIC_TX  -> IRQ[10] to IRQ[11]
    // IRQ_CDMA     -> IRQ[12] to IRQ[15]
    // IRQ_MTTY     -> IRQ[16] to IRQ[30]
    // IRQ_BDEV     -> IRQ[31]

    size_t mx = 16 + NB_TTY_CHANNELS;
    for ( size_t n=0 ; n<32 ; n++ )
    {
        if      ( n < 8  ) clusters[0][0]->p_irq[n]->bind    (signal_unused_irq[n]);
        else if ( n < 10 ) clusters[0][0]->p_irq[n]->bind    (signal_irq_mnic_rx[n-8]);
        else if ( n < 12 ) clusters[0][0]->p_irq[n]->bind    (signal_irq_mnic_tx[n-10]);
        else if ( n < 16 ) clusters[0][0]->p_irq[n]->bind    (signal_irq_cdma[n-12]);
        else if ( n < mx ) clusters[0][0]->p_irq[n]->bind    (signal_irq_mtty[n-16]);
        else if ( n < 31 ) clusters[0][0]->p_irq[n]->bind    (signal_unused_irq[n]);
        else               clusters[0][0]->p_irq[n]->bind    (signal_irq_bdev);
    }

    // IOB0 cluster connexion to IOX network
    clusters[0][0]->p_vci_iox_ini->bind                      (signal_vci_ini_iob0);
    clusters[0][0]->p_vci_iox_tgt->bind                      (signal_vci_tgt_iob0);

    // IOB1 cluster connexion to IOX network
    clusters[XMAX-1][YMAX-1]->p_vci_iox_ini->bind            (signal_vci_ini_iob1);
    clusters[XMAX-1][YMAX-1]->p_vci_iox_tgt->bind            (signal_vci_tgt_iob1);

    // All clusters Clock & RESET connexions
    for ( size_t x = 0; x < (XMAX); x++ )
    {
        for (size_t y = 0; y < YMAX; y++)
        {
            clusters[x][y]->p_clk     (signal_clk);
            clusters[x][y]->p_resetn  (signal_resetn);
        }
    }

   // Inter Clusters horizontal connections
   if (XMAX > 1)
   {
      for (size_t x = 0; x < (XMAX-1); x++)
      {
         for (size_t y = 0; y < YMAX; y++)
         {
            for (size_t k = 0; k < 3; k++)
            {
               clusters[x][y]->p_dspin_int_cmd_out[EAST][k]      (signal_dspin_int_cmd_h_inc[x][y][k]);
               clusters[x+1][y]->p_dspin_int_cmd_in[WEST][k]     (signal_dspin_int_cmd_h_inc[x][y][k]);
               clusters[x][y]->p_dspin_int_cmd_in[EAST][k]       (signal_dspin_int_cmd_h_dec[x][y][k]);
               clusters[x+1][y]->p_dspin_int_cmd_out[WEST][k]    (signal_dspin_int_cmd_h_dec[x][y][k]);
            }

            for (size_t k = 0; k < 2; k++)
            {
               clusters[x][y]->p_dspin_int_rsp_out[EAST][k]      (signal_dspin_int_rsp_h_inc[x][y][k]);
               clusters[x+1][y]->p_dspin_int_rsp_in[WEST][k]     (signal_dspin_int_rsp_h_inc[x][y][k]);
               clusters[x][y]->p_dspin_int_rsp_in[EAST][k]       (signal_dspin_int_rsp_h_dec[x][y][k]);
               clusters[x+1][y]->p_dspin_int_rsp_out[WEST][k]    (signal_dspin_int_rsp_h_dec[x][y][k]);
            }

            clusters[x][y]->p_dspin_ram_cmd_out[EAST]      (signal_dspin_ram_cmd_h_inc[x][y]);
            clusters[x+1][y]->p_dspin_ram_cmd_in[WEST]     (signal_dspin_ram_cmd_h_inc[x][y]);
            clusters[x][y]->p_dspin_ram_cmd_in[EAST]       (signal_dspin_ram_cmd_h_dec[x][y]);
            clusters[x+1][y]->p_dspin_ram_cmd_out[WEST]    (signal_dspin_ram_cmd_h_dec[x][y]);
            clusters[x][y]->p_dspin_ram_rsp_out[EAST]      (signal_dspin_ram_rsp_h_inc[x][y]);
            clusters[x+1][y]->p_dspin_ram_rsp_in[WEST]     (signal_dspin_ram_rsp_h_inc[x][y]);
            clusters[x][y]->p_dspin_ram_rsp_in[EAST]       (signal_dspin_ram_rsp_h_dec[x][y]);
            clusters[x+1][y]->p_dspin_ram_rsp_out[WEST]    (signal_dspin_ram_rsp_h_dec[x][y]);
         }
      }
   }

   std::cout << std::endl << "Horizontal connections established" << std::endl;   

   // Inter Clusters vertical connections
   if (YMAX > 1) 
   {
      for (size_t y = 0; y < (YMAX-1); y++)
      {
         for (size_t x = 0; x < XMAX; x++)
         {
            for (size_t k = 0; k < 3; k++)
            {
               clusters[x][y]->p_dspin_int_cmd_out[NORTH][k]     (signal_dspin_int_cmd_v_inc[x][y][k]);
               clusters[x][y+1]->p_dspin_int_cmd_in[SOUTH][k]    (signal_dspin_int_cmd_v_inc[x][y][k]);
               clusters[x][y]->p_dspin_int_cmd_in[NORTH][k]      (signal_dspin_int_cmd_v_dec[x][y][k]);
               clusters[x][y+1]->p_dspin_int_cmd_out[SOUTH][k]   (signal_dspin_int_cmd_v_dec[x][y][k]);
            }

            for (size_t k = 0; k < 2; k++)
            {
               clusters[x][y]->p_dspin_int_rsp_out[NORTH][k]     (signal_dspin_int_rsp_v_inc[x][y][k]);
               clusters[x][y+1]->p_dspin_int_rsp_in[SOUTH][k]    (signal_dspin_int_rsp_v_inc[x][y][k]);
               clusters[x][y]->p_dspin_int_rsp_in[NORTH][k]      (signal_dspin_int_rsp_v_dec[x][y][k]);
               clusters[x][y+1]->p_dspin_int_rsp_out[SOUTH][k]   (signal_dspin_int_rsp_v_dec[x][y][k]);
            }

            clusters[x][y]->p_dspin_ram_cmd_out[NORTH]     (signal_dspin_ram_cmd_v_inc[x][y]);
            clusters[x][y+1]->p_dspin_ram_cmd_in[SOUTH]    (signal_dspin_ram_cmd_v_inc[x][y]);
            clusters[x][y]->p_dspin_ram_cmd_in[NORTH]      (signal_dspin_ram_cmd_v_dec[x][y]);
            clusters[x][y+1]->p_dspin_ram_cmd_out[SOUTH]   (signal_dspin_ram_cmd_v_dec[x][y]);
            clusters[x][y]->p_dspin_ram_rsp_out[NORTH]     (signal_dspin_ram_rsp_v_inc[x][y]);
            clusters[x][y+1]->p_dspin_ram_rsp_in[SOUTH]    (signal_dspin_ram_rsp_v_inc[x][y]);
            clusters[x][y]->p_dspin_ram_rsp_in[NORTH]      (signal_dspin_ram_rsp_v_dec[x][y]);
            clusters[x][y+1]->p_dspin_ram_rsp_out[SOUTH]   (signal_dspin_ram_rsp_v_dec[x][y]);
         }
      }
   }

   std::cout << "Vertical connections established" << std::endl;

   // East & West boundary cluster connections
   for (size_t y = 0; y < YMAX; y++)
   {
      for (size_t k = 0; k < 3; k++)
      {
         clusters[0][y]->p_dspin_int_cmd_in[WEST][k]          (signal_dspin_false_int_cmd_in[0][y][WEST][k]);
         clusters[0][y]->p_dspin_int_cmd_out[WEST][k]         (signal_dspin_false_int_cmd_out[0][y][WEST][k]);
         clusters[XMAX-1][y]->p_dspin_int_cmd_in[EAST][k]     (signal_dspin_false_int_cmd_in[XMAX-1][y][EAST][k]);
         clusters[XMAX-1][y]->p_dspin_int_cmd_out[EAST][k]    (signal_dspin_false_int_cmd_out[XMAX-1][y][EAST][k]);
      }

      for (size_t k = 0; k < 2; k++)
      {
         clusters[0][y]->p_dspin_int_rsp_in[WEST][k]          (signal_dspin_false_int_rsp_in[0][y][WEST][k]);
         clusters[0][y]->p_dspin_int_rsp_out[WEST][k]         (signal_dspin_false_int_rsp_out[0][y][WEST][k]);
         clusters[XMAX-1][y]->p_dspin_int_rsp_in[EAST][k]     (signal_dspin_false_int_rsp_in[XMAX-1][y][EAST][k]);
         clusters[XMAX-1][y]->p_dspin_int_rsp_out[EAST][k]    (signal_dspin_false_int_rsp_out[XMAX-1][y][EAST][k]);
      }

      clusters[0][y]->p_dspin_ram_cmd_in[WEST]                (signal_dspin_false_ram_cmd_in[0][y][WEST]);
      clusters[0][y]->p_dspin_ram_cmd_out[WEST]               (signal_dspin_false_ram_cmd_out[0][y][WEST]);
      clusters[0][y]->p_dspin_ram_rsp_in[WEST]                (signal_dspin_false_ram_rsp_in[0][y][WEST]);
      clusters[0][y]->p_dspin_ram_rsp_out[WEST]               (signal_dspin_false_ram_rsp_out[0][y][WEST]);

      clusters[XMAX-1][y]->p_dspin_ram_cmd_in[EAST]           (signal_dspin_false_ram_cmd_in[XMAX-1][y][EAST]);
      clusters[XMAX-1][y]->p_dspin_ram_cmd_out[EAST]          (signal_dspin_false_ram_cmd_out[XMAX-1][y][EAST]);
      clusters[XMAX-1][y]->p_dspin_ram_rsp_in[EAST]           (signal_dspin_false_ram_rsp_in[XMAX-1][y][EAST]);
      clusters[XMAX-1][y]->p_dspin_ram_rsp_out[EAST]          (signal_dspin_false_ram_rsp_out[XMAX-1][y][EAST]);
   }

   std::cout << "East & West boundaries established" << std::endl;

   // North & South boundary clusters connections
   for (size_t x = 0; x < XMAX; x++)
   {
      for (size_t k = 0; k < 3; k++)
      {
         clusters[x][0]->p_dspin_int_cmd_in[SOUTH][k]         (signal_dspin_false_int_cmd_in[x][0][SOUTH][k]);
         clusters[x][0]->p_dspin_int_cmd_out[SOUTH][k]        (signal_dspin_false_int_cmd_out[x][0][SOUTH][k]);
         clusters[x][YMAX-1]->p_dspin_int_cmd_in[NORTH][k]    (signal_dspin_false_int_cmd_in[x][YMAX-1][NORTH][k]);
         clusters[x][YMAX-1]->p_dspin_int_cmd_out[NORTH][k]   (signal_dspin_false_int_cmd_out[x][YMAX-1][NORTH][k]);
      }

      for (size_t k = 0; k < 2; k++)
      {
         clusters[x][0]->p_dspin_int_rsp_in[SOUTH][k]         (signal_dspin_false_int_rsp_in[x][0][SOUTH][k]);
         clusters[x][0]->p_dspin_int_rsp_out[SOUTH][k]        (signal_dspin_false_int_rsp_out[x][0][SOUTH][k]);
         clusters[x][YMAX-1]->p_dspin_int_rsp_in[NORTH][k]    (signal_dspin_false_int_rsp_in[x][YMAX-1][NORTH][k]);
         clusters[x][YMAX-1]->p_dspin_int_rsp_out[NORTH][k]   (signal_dspin_false_int_rsp_out[x][YMAX-1][NORTH][k]);
      }

      clusters[x][0]->p_dspin_ram_cmd_in[SOUTH]               (signal_dspin_false_ram_cmd_in[x][0][SOUTH]);
      clusters[x][0]->p_dspin_ram_cmd_out[SOUTH]              (signal_dspin_false_ram_cmd_out[x][0][SOUTH]);
      clusters[x][0]->p_dspin_ram_rsp_in[SOUTH]               (signal_dspin_false_ram_rsp_in[x][0][SOUTH]);
      clusters[x][0]->p_dspin_ram_rsp_out[SOUTH]              (signal_dspin_false_ram_rsp_out[x][0][SOUTH]);

      clusters[x][YMAX-1]->p_dspin_ram_cmd_in[NORTH]          (signal_dspin_false_ram_cmd_in[x][YMAX-1][NORTH]);
      clusters[x][YMAX-1]->p_dspin_ram_cmd_out[NORTH]         (signal_dspin_false_ram_cmd_out[x][YMAX-1][NORTH]);
      clusters[x][YMAX-1]->p_dspin_ram_rsp_in[NORTH]          (signal_dspin_false_ram_rsp_in[x][YMAX-1][NORTH]);
      clusters[x][YMAX-1]->p_dspin_ram_rsp_out[NORTH]         (signal_dspin_false_ram_rsp_out[x][YMAX-1][NORTH]);
   }

   std::cout << "North & South boundaries established" << std::endl;

   ////////////////////////////////////////////////////////
   //   Simulation
   ///////////////////////////////////////////////////////

   sc_start(sc_core::sc_time(0, SC_NS));
   signal_resetn = false;

   // network boundaries signals
   for (size_t x = 0; x < XMAX ; x++)
   {
      for (size_t y = 0; y < YMAX ; y++)
      {
         for (size_t a = 0; a < 4; a++)
         {
            for (size_t k = 0; k < 3; k++)
            {
               signal_dspin_false_int_cmd_in[x][y][a][k].write = false;
               signal_dspin_false_int_cmd_in[x][y][a][k].read = true;
               signal_dspin_false_int_cmd_out[x][y][a][k].write = false;
               signal_dspin_false_int_cmd_out[x][y][a][k].read = true;
            }

            for (size_t k = 0; k < 2; k++)
            {
               signal_dspin_false_int_rsp_in[x][y][a][k].write = false;
               signal_dspin_false_int_rsp_in[x][y][a][k].read = true;
               signal_dspin_false_int_rsp_out[x][y][a][k].write = false;
               signal_dspin_false_int_rsp_out[x][y][a][k].read = true;
            }

            signal_dspin_false_ram_cmd_in[x][y][a].write = false;
            signal_dspin_false_ram_cmd_in[x][y][a].read = true;
            signal_dspin_false_ram_cmd_out[x][y][a].write = false;
            signal_dspin_false_ram_cmd_out[x][y][a].read = true;

            signal_dspin_false_ram_rsp_in[x][y][a].write = false;
            signal_dspin_false_ram_rsp_in[x][y][a].read = true;
            signal_dspin_false_ram_rsp_out[x][y][a].write = false;
            signal_dspin_false_ram_rsp_out[x][y][a].read = true;
         }
      }
   }

   sc_start(sc_core::sc_time(1, SC_NS));
   signal_resetn = true;

   for (size_t n = 1; n < ncycles; n++)
   {
      // Monitor a specific address for L1 & L2 caches
      // clusters[1][1]->proc[0]->cache_monitor(0x8ba4ULL);
      // clusters[0][0]->memc->cache_monitor(   0x12180ULL);

      if (debug_ok and (n > debug_from) and (n % debug_period == 0))
      {
         std::cout << "****************** cycle " << std::dec << n ;
         std::cout << " ************************************************" << std::endl;

        // trace proc[debug_proc_id] 
        if ( debug_proc_id < XMAX*YMAX*NB_PROCS_MAX )
        {

        size_t l = debug_proc_id % NB_PROCS_MAX ;
        size_t y = (debug_proc_id / NB_PROCS_MAX) % YMAX ;
        size_t x = debug_proc_id / (YMAX * NB_PROCS_MAX) ;

        clusters[x][y]->proc[l]->print_trace(1);

        std::ostringstream proc_signame;
        proc_signame << "[SIG]PROC_" << x << "_" << y << "_" << l ;
        clusters[x][y]->signal_int_vci_ini_proc[l].print_trace(proc_signame.str());

        std::ostringstream p2m_signame;
        p2m_signame << "[SIG]PROC_" << x << "_" << y << "_" << l << " P2M" ;
        clusters[x][y]->signal_int_dspin_p2m_proc[l].print_trace(p2m_signame.str());

        std::ostringstream m2p_signame;
        m2p_signame << "[SIG]PROC_" << x << "_" << y << "_" << l << " M2P" ;
        clusters[x][y]->signal_int_dspin_m2p_proc[l].print_trace(m2p_signame.str());

//      std::ostringstream p_cmd_signame;
//      p_cmd_signame << "[SIG]PROC_" << x << "_" << y << "_" << l << " CMD" ;
//      clusters[x][y]->signal_int_dspin_cmd_proc_i[l].print_trace(p_cmd_signame.str());

//      std::ostringstream p_rsp_signame;
//      p_rsp_signame << "[SIG]PROC_" << x << "_" << y << "_" << l << " RSP" ;
//      clusters[x][y]->signal_int_dspin_rsp_proc_i[l].print_trace(p_rsp_signame.str());

        // trace INT routers and xbar in a given cluster
//      clusters[x][y]->int_xbar_m2p_c->print_trace();
//      clusters[x][y]->int_router_cmd->print_trace(1);
//      clusters[x][y]->int_xbar_rsp_d->print_trace();
//      clusters[x][y]->int_xbar_cmd_d->print_trace();
//      clusters[x][y]->signal_int_dspin_cmd_l2g_d.print_trace("[SIG]L2G CMD");
//      clusters[x][y]->signal_int_dspin_cmd_g2l_d.print_trace("[SIG]G2L CMD");
//      clusters[x][y]->signal_int_dspin_rsp_l2g_d.print_trace("[SIG]L2G RSP");
//      clusters[x][y]->signal_int_dspin_rsp_g2l_d.print_trace("[SIG]G2L RSP");
        }

        // trace memc[debug_memc_id] 
        if ( debug_memc_id < XMAX*YMAX )
        {

        size_t x = debug_memc_id / YMAX;
        size_t y = debug_memc_id % YMAX;

        clusters[x][y]->memc->print_trace();

        std::ostringstream smemc_tgt;
        smemc_tgt << "[SIG]MEMC_TGT_" << x << "_" << y;
        clusters[x][y]->signal_int_vci_tgt_memc.print_trace(smemc_tgt.str());

        std::ostringstream smemc_ini;
        smemc_ini << "[SIG]MEMC_INI_" << x << "_" << y;
        clusters[x][y]->signal_ram_vci_ini_memc.print_trace(smemc_ini.str());

//      clusters[x][y]->ram_router_cmd->print_trace();
//      clusters[x][y]->ram_xbar_cmd->print_trace();

//      std::ostringstream sg2l;
//      sg2l << "[SIG]G2L_" << x << "_" << y;
//      clusters[x][y]->signal_ram_dspin_cmd_g2l.print_trace(sg2l.str());

        clusters[x][y]->xram->print_trace();

        std::ostringstream sxram_tgt;
        sxram_tgt << "[SIG]XRAM_TGT_" << x << "_" << y;
        clusters[x][y]->signal_ram_vci_tgt_xram.print_trace(sxram_tgt.str());

        std::ostringstream sm2p;
        sm2p << "[SIG]MEMC_" << x << "_" << y << " M2P" ;
        clusters[x][y]->signal_int_dspin_m2p_memc.print_trace(sm2p.str());

        std::ostringstream sp2m;
        sp2m << "[SIG]MEMC_" << x << "_" << y << " P2M" ;
        clusters[x][y]->signal_int_dspin_p2m_memc.print_trace(sp2m.str());

//      std::ostringstream m_cmd_signame;
//      m_cmd_signame << "[SIG]MEMC_" << x << "_" << y <<  " CMD" ;
//      clusters[x][y]->signal_int_dspin_cmd_memc_t.print_trace(m_cmd_signame.str());

//      std::ostringstream m_rsp_signame;
//      m_rsp_signame << "[SIG]MEMC_" << x << "_" << y <<  " RSP" ;
//      clusters[x][y]->signal_int_dspin_rsp_memc_t.print_trace(m_rsp_signame.str());

        std::ostringstream siob_ini;
        siob_ini << "[SIG]IOB_INI_" << x << "_" << y;
        clusters[x][y]->signal_ram_vci_ini_iobx.print_trace(siob_ini.str());
        }
        
        // trace components iob  
        if ( debug_iob )
        {
        clusters[0][0]->iob->print_trace();
        clusters[0][0]->signal_int_vci_tgt_iobx.print_trace( "[SIG]IOB0 INT TGT" );
//      clusters[0][0]->signal_int_dspin_cmd_iobx_t.print_trace("[SIG]IOB0 INT CMD");
//      clusters[0][0]->signal_int_dspin_rsp_iobx_t.print_trace("[SIG]IOB0 INT RSP");

        clusters[XMAX-1][YMAX-1]->iob->print_trace();
        clusters[XMAX-1][YMAX-1]->signal_int_vci_tgt_iobx.print_trace( "[SIG]IOB1 INT TGT" );
//      clusters[XMAX-1][YMAX-1]->signal_int_dspin_cmd_iobx_t.print_trace("[SIG]IOB1 INT CMD");
//      clusters[XMAX-1][YMAX-1]->signal_int_dspin_rsp_iobx_t.print_trace("[SIG]IOB1 INT RSP");
        } 

        // trace external peripherals
        iox_network->print_trace();

        signal_vci_ini_iob0.print_trace("[SIG]IOB0 IOX INI");
        signal_vci_tgt_iob0.print_trace("[SIG]IOB0 IOX TGT");
        signal_vci_ini_iob1.print_trace("[SIG]IOB1 IOX INI");
        signal_vci_tgt_iob1.print_trace("[SIG]IOB1 IOX TGT");

        iox_cdma->print_trace();
        signal_vci_tgt_cdma.print_trace("[SIG]CDMA_TGT");
        signal_vci_ini_cdma.print_trace("[SIG]CDMA_INI");

//      iox_brom->print_trace();
//      signal_vci_tgt_brom.print_trace("[SIG]BROM");

        iox_mtty->print_trace();
        signal_vci_tgt_mtty.print_trace("[SIG]MTTY");

        iox_bdev->print_trace();
        signal_vci_tgt_bdev.print_trace("[SIG]BDEV_TGT");
        signal_vci_ini_bdev.print_trace("[SIG]BDEV_INI");

//      iox_fbuf->print_trace();
//      signal_vci_tgt_fbuf.print_trace("[SIG]FBUF");

      }

      sc_start(sc_core::sc_time(1, SC_NS));
   }
   return EXIT_SUCCESS;
}

int sc_main (int argc, char *argv[])
{
   try {
      return _main(argc, argv);
   } catch (std::exception &e) {
      std::cout << e.what() << std::endl;
   } catch (...) {
      std::cout << "Unknown exception occured" << std::endl;
      throw;
   }
   return 1;
}


// Local Variables:
// tab-width: 3
// c-basic-offset: 3
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=3:tabstop=3:softtabstop=3



