/**
 * \file    : reset_utils.c
 * \date    : August 2012
 * \author  : Cesar Fuguet
 *
 * Definition of utilities functions used by the TSAR pre-loader
 */

#include <reset_utils.h>

/*****************************************
 * proctime()
 *
 * Returns processor local time.
 ****************************************/
inline unsigned int proctime() 
{
    unsigned int ret;
    asm volatile ("mfc0   %0,        $9":"=r" (ret));
    return ret;
}

/*****************************************
 * memcpy( _dst, _src, size )
 *
 * Transfer data between two memory buffers.
 *
 * \param _dst   : Destination buffer base address 
 * \param _src   : Source buffer base address
 * \param size   : Number of bytes to transfer 
 *
 ****************************************/
void * memcpy(void *_dst, const void *_src, unsigned int size)
{
    unsigned int *dst = _dst;
    const unsigned int *src = _src;
    if ( ! ((unsigned int)dst & 3) && ! ((unsigned int)src & 3) )
        while (size > 3) 
        {
            *dst++ = *src++;
            size -= 4;
        }

    unsigned char *cdst = (unsigned char*) dst;
    unsigned char *csrc = (unsigned char*) src;

    while (size--) 
    {
        *cdst++ = *csrc++;
    }
    return _dst;
}

/*****************************************
 * memset( _dst, value, size )
 *
 * Initialize memory buffers with predefined value.
 *
 * \param _dst   : Destination buffer base address 
 * \param value  : Initialization value 
 * \param size   : Number of bytes to initialize
 *
 ****************************************/
void * memset(void *_dst, const int value, unsigned int size)
{
    char * dst = (char *) _dst;

    while(size--) *dst++ = (char) value;

    return _dst;
}

/*****************************************
 * reset_print_elf_phdr( elf_phdr_ptr )
 *
 * Print some fields of a ELF program header
 *
 * \param elf_phdr_ptr : Pointer to the ELF program header to print
 *
 ****************************************/
void reset_print_elf_phdr(Elf32_Phdr * elf_phdr_ptr)
{
    reset_puts("- type   : ");
    reset_putx(elf_phdr_ptr->p_type);

    reset_puts("\n- offset : ");
    reset_putx(elf_phdr_ptr->p_offset);

    reset_puts("\n- vaddr  : ");
    reset_putx(elf_phdr_ptr->p_vaddr);

    reset_puts("\n- paddr  : ");
    reset_putx(elf_phdr_ptr->p_paddr);

    reset_puts("\n- filesz : ");
    reset_putx(elf_phdr_ptr->p_filesz);

    reset_puts("\n- memsz  : ");
    reset_putx(elf_phdr_ptr->p_memsz);

    reset_puts("\n- flags  : ");
    reset_putx(elf_phdr_ptr->p_flags);

    reset_puts("\n- align  : ");
    reset_putx(elf_phdr_ptr->p_align);
}

// vim: tabstop=4 : softtabstop=4 : shiftwidth=4 : expandtab
