//////////////////////////////////////////////////////////////////////////////
// File: tsar_iob_cluster.cpp
// Author: Alain Greiner 
// Copyright: UPMC/LIP6
// Date : april 2013
// This program is released under the GNU public license
//////////////////////////////////////////////////////////////////////////////
// Cluster(0,0) & Cluster(xmax-1,ymax-1) contains the IOB0 & IOB1 components.
// These two clusters contain 6 extra components:
// - 1 vci_io_bridge (connected to the 3 networks.
// - 3 vci_dspin_wrapper for the IOB.
// - 2 dspin_local_crossbar for commands and responses. 
//////////////////////////////////////////////////////////////////////////////

#include "../include/tsar_iob_cluster.h"

namespace soclib { namespace caba  {

//////////////////////////////////////////////////////////////////////////
//                 Constructor
//////////////////////////////////////////////////////////////////////////
template<typename vci_param_int, 
         typename vci_param_ext,
         size_t   dspin_int_cmd_width, 
         size_t   dspin_int_rsp_width,
         size_t   dspin_ram_cmd_width, 
         size_t   dspin_ram_rsp_width>
TsarIobCluster<vci_param_int, 
               vci_param_ext,
               dspin_int_cmd_width, 
               dspin_int_rsp_width,
               dspin_ram_cmd_width, 
               dspin_ram_rsp_width>::TsarIobCluster(
//////////////////////////////////////////////////////////////////////////
                    sc_module_name                     insname,
                    size_t                             nb_procs,
                    size_t                             nb_dmas,
                    size_t                             x_id,
                    size_t                             y_id,
                    size_t                             xmax,
                    size_t                             ymax,

                    const soclib::common::MappingTable &mt_int,
                    const soclib::common::MappingTable &mt_ram, 
                    const soclib::common::MappingTable &mt_iox, 

                    size_t                             x_width,
                    size_t                             y_width,
                    size_t                             l_width,

                    size_t                             memc_int_tgtid,  // local index
                    size_t                             xicu_int_tgtid,  // local index
                    size_t                             mdma_int_tgtid,  // local index
                    size_t                             iobx_int_tgtid,  // local index

                    size_t                             proc_int_srcid,  // local index
                    size_t                             mdma_int_srcid,  // local index
                    size_t                             iobx_int_srcid,  // local index

                    size_t                             xram_ram_tgtid,  // local index

                    size_t                             memc_ram_srcid,  // local index
                    size_t                             iobx_ram_srcid,  // local index

                    size_t                             memc_ways,
                    size_t                             memc_sets,
                    size_t                             l1_i_ways,
                    size_t                             l1_i_sets,
                    size_t                             l1_d_ways,
                    size_t                             l1_d_sets,
                    size_t                             xram_latency,

                    const Loader                      &loader,

                    uint32_t                           frozen_cycles,
                    uint32_t                           debug_start_cycle,
                    bool                               memc_debug_ok,
                    bool                               proc_debug_ok,
                    bool                               iob_debug_ok )
    : soclib::caba::BaseModule(insname),
      p_clk("clk"),
      p_resetn("resetn")
{
    assert( (x_id < xmax) and (y_id < ymax) and "Illegal cluster coordinates");

    size_t cluster_id = (x_id<<4) + y_id;

    size_t cluster_iob0 = 0;                            // South-West cluster
    size_t cluster_iob1 = ((xmax-1)<<4) + ymax-1;       // North-East cluster

    // Vectors of DSPIN ports for inter-cluster communications
    p_dspin_int_cmd_in  = alloc_elems<DspinInput<dspin_int_cmd_width> >("p_int_cmd_in", 4, 3);
    p_dspin_int_cmd_out = alloc_elems<DspinOutput<dspin_int_cmd_width> >("p_int_cmd_out", 4, 3);
    p_dspin_int_rsp_in  = alloc_elems<DspinInput<dspin_int_rsp_width> >("p_int_rsp_in", 4, 2);
    p_dspin_int_rsp_out = alloc_elems<DspinOutput<dspin_int_rsp_width> >("p_int_rsp_out", 4, 2);

    p_dspin_ram_cmd_in  = alloc_elems<DspinInput<dspin_ram_cmd_width> >("p_ext_cmd_in", 4);
    p_dspin_ram_cmd_out = alloc_elems<DspinOutput<dspin_ram_cmd_width> >("p_ext_cmd_out", 4);
    p_dspin_ram_rsp_in  = alloc_elems<DspinInput<dspin_ram_rsp_width> >("p_ext_rsp_in", 4);
    p_dspin_ram_rsp_out = alloc_elems<DspinOutput<dspin_ram_rsp_width> >("p_ext_rsp_out", 4);

    // ports in cluster_iob0 and cluster_iob1 only
    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
        // VCI ports from IOB to IOX network
        p_vci_iob_iox_ini = new soclib::caba::VciInitiator<vci_param_ext>;
        p_vci_iob_iox_tgt = new soclib::caba::VciTarget<vci_param_ext>; 

        // DSPIN ports from IOB to RAM network
        p_dspin_iob_cmd_out = new soclib::caba::DspinOutput<dspin_ram_cmd_width>;
        p_dspin_iob_rsp_in  = new soclib::caba::DspinInput<dspin_ram_rsp_width>;
    }

    // IRQ ports in cluster_iob0 only
    if ( cluster_id == cluster_iob0 )
    {
        for ( size_t n=0 ; n<32 ; n++ ) p_irq[n] = new sc_in<bool>;
    }

    /////////////////////////////////////////////////////////////////////////////
    //    Hardware components
    /////////////////////////////////////////////////////////////////////////////

    ////////////  PROCS
    for (size_t p = 0; p < nb_procs; p++)
    { 
        std::ostringstream s_proc;
        s_proc << "proc_" << x_id << "_" << y_id << "_" << p;
        proc[p] = new VciCcVCacheWrapper<vci_param_int,
                                         dspin_int_cmd_width,
                                         dspin_int_rsp_width,
                                         GdbServer<Mips32ElIss> >(
                      s_proc.str().c_str(),
                      cluster_id*nb_procs + p,        // GLOBAL PROC_ID
                      mt_int,                         // Mapping Table INT network
                      IntTab(cluster_id,p),           // SRCID
                      (cluster_id << l_width) + p,    // CC_GLOBAL_ID
                      8,                              // ITLB ways
                      8,                              // ITLB sets
                      8,                              // DTLB ways
                      8,                              // DTLB sets
                      l1_i_ways,l1_i_sets,16,         // ICACHE size
                      l1_d_ways,l1_d_sets,16,         // DCACHE size
                      4,                              // WBUF nlines
                      4,                              // WBUF nwords
                      x_width,
                      y_width,
                      frozen_cycles,                  // max frozen cycles
                      debug_start_cycle,
                      proc_debug_ok);

        std::ostringstream s_wi_proc;
        s_wi_proc << "proc_wi_" << x_id << "_" << y_id << "_" << p;
        proc_wi[p] = new VciDspinInitiatorWrapper<vci_param_int,
                                                      dspin_int_cmd_width,
                                                      dspin_int_rsp_width>(
                     s_wi_proc.str().c_str(),
                     x_width + y_width + l_width);
    }

    ///////////  MEMC    
    std::ostringstream s_memc;
    s_memc << "memc_" << x_id << "_" << y_id;
    memc = new VciMemCache<vci_param_int,
                           vci_param_ext,
                           dspin_int_rsp_width,
                           dspin_int_cmd_width>(
                     s_memc.str().c_str(),
                     mt_int,                             // Mapping Table INT network
                     mt_ram,                             // Mapping Table RAM network
                     IntTab(cluster_id, memc_ram_srcid), // SRCID RAM network
                     IntTab(cluster_id, memc_int_tgtid), // TGTID INT network
                     x_width,                            // number of bits for x coordinate
                     y_width,                            // number of bits for y coordinate
                     memc_ways, memc_sets, 16,           // CACHE SIZE
                     3,                                  // MAX NUMBER OF COPIES
                     4096,                               // HEAP SIZE
                     8,                                  // TRANSACTION TABLE DEPTH
                     8,                                  // UPDATE TABLE DEPTH
                     8,                                  // INVALIDATE TABLE DEPTH
                     debug_start_cycle,
                     memc_debug_ok );

    std::ostringstream s_wt_memc;
    s_wt_memc << "memc_wt_" << x_id << "_" << y_id;
    memc_int_wt = new VciDspinTargetWrapper<vci_param_int,
                                            dspin_int_cmd_width,
                                            dspin_int_rsp_width>(
                     s_wt_memc.str().c_str(),
                     x_width + y_width + l_width);

    std::ostringstream s_wi_memc;
    s_wi_memc << "memc_wi_" << x_id << "_" << y_id;
    memc_ram_wi = new VciDspinInitiatorWrapper<vci_param_ext,
                                               dspin_ram_cmd_width,
                                               dspin_ram_rsp_width>(
                     s_wi_memc.str().c_str(),
                     x_width + y_width + l_width);

    ///////////   XICU
    std::ostringstream s_xicu;
    s_xicu << "xicu_" << x_id << "_" << y_id;
    xicu = new VciXicu<vci_param_int>(
                     s_xicu.str().c_str(),
                     mt_int,                            // mapping table INT network
                     IntTab(cluster_id,xicu_int_tgtid), // TGTID direct space
                     32,                                // number of timer IRQs
                     32,                                // number of hard IRQs
                     32,                                // number of soft IRQs
                     nb_procs);                         // number of output IRQs

    std::ostringstream s_wt_xicu;
    s_wt_xicu << "xicu_wt_" << x_id << "_" << y_id;
    xicu_int_wt = new VciDspinTargetWrapper<vci_param_int,
                                        dspin_int_cmd_width,
                                        dspin_int_rsp_width>(
                     s_wt_xicu.str().c_str(),
                     x_width + y_width + l_width);

    ////////////  MDMA
    std::ostringstream s_mdma;
    s_mdma << "mdma_" << x_id << "_" << y_id;
    mdma = new VciMultiDma<vci_param_int>(
                     s_mdma.str().c_str(),
                     mt_int,
                     IntTab(cluster_id, nb_procs),        // SRCID
                     IntTab(cluster_id, mdma_int_tgtid),  // TGTID
                     64,                                  // burst size
                     nb_dmas);                            // number of IRQs

    std::ostringstream s_wt_mdma;
    s_wt_mdma << "mdma_wt_" << x_id << "_" << y_id;
    mdma_int_wt = new VciDspinTargetWrapper<vci_param_int,
                                            dspin_int_cmd_width,
                                            dspin_int_rsp_width>(
                     s_wt_mdma.str().c_str(),
                     x_width + y_width + l_width);

    std::ostringstream s_wi_mdma;
    s_wi_mdma << "mdma_wi_" << x_id << "_" << y_id;
    mdma_int_wi = new VciDspinInitiatorWrapper<vci_param_int,
                                               dspin_int_cmd_width,
                                               dspin_int_rsp_width>(
                     s_wi_mdma.str().c_str(),
                     x_width + y_width + l_width);

    ///////////  Direct LOCAL_XBAR(S)
    size_t nb_direct_initiators      = nb_procs + 1;
    size_t nb_direct_targets         = 3;
    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
        nb_direct_initiators         = nb_procs + 2;
        nb_direct_targets            = 4;
    }

    std::ostringstream s_int_xbar_cmd_d;
    s_int_xbar_cmd_d << "int_xbar_cmd_d_" << x_id << "_" << y_id;
    int_xbar_cmd_d = new DspinLocalCrossbar<dspin_int_cmd_width>(
                     s_int_xbar_cmd_d.str().c_str(),
                     mt_int,                       // mapping table
                     x_id, y_id,                   // cluster coordinates
                     x_width, y_width, l_width,
                     nb_direct_initiators,         // number of local of sources
                     nb_direct_targets,            // number of local dests 
                     2, 2,                         // fifo depths  
                     true,                         // CMD crossbar
                     true,                         // use routing table 
                     false );                      // no broacast

    std::ostringstream s_int_xbar_rsp_d;
    s_int_xbar_rsp_d << "int_xbar_rsp_d_" << x_id << "_" << y_id;
    int_xbar_rsp_d = new DspinLocalCrossbar<dspin_int_rsp_width>(
                     s_int_xbar_rsp_d.str().c_str(),
                     mt_int,                       // mapping table
                     x_id, y_id,                   // cluster coordinates
                     x_width, y_width, l_width,
                     nb_direct_targets,            // number of local sources      
                     nb_direct_initiators,         // number of local dests
                     2, 2,                         // fifo depths
                     false,                        // RSP crossbar  
                     false,                        // don't use routing table 
                     false );                      // no broacast

    ////////////  Coherence LOCAL_XBAR(S)
    std::ostringstream s_int_xbar_m2p_c;
    s_int_xbar_m2p_c << "int_xbar_m2p_c_" << x_id << "_" << y_id;
    int_xbar_m2p_c = new DspinLocalCrossbar<dspin_int_cmd_width>(
                     s_int_xbar_m2p_c.str().c_str(),
                     mt_int,                       // mapping table
                     x_id, y_id,                   // cluster coordinates
                     x_width, y_width, l_width,    // several dests
                     1,                            // number of local sources
                     nb_procs,                     // number of local dests 
                     2, 2,                         // fifo depths  
                     true,                         // pseudo CMD
                     false,                        // no routing table
                     true );                       // broacast

    std::ostringstream s_int_xbar_p2m_c;
    s_int_xbar_p2m_c << "int_xbar_p2m_c_" << x_id << "_" << y_id;
    int_xbar_p2m_c = new DspinLocalCrossbar<dspin_int_rsp_width>(
                     s_int_xbar_p2m_c.str().c_str(),
                     mt_int,                       // mapping table
                     x_id, y_id,                   // cluster coordinates
                     x_width, y_width, 0,          // only one dest
                     nb_procs,                     // number of local sources
                     1,                            // number of local dests
                     2, 2,                         // fifo depths  
                     false,                        // pseudo RSP
                     false,                        // no routing table
                     false );                      // no broacast 

    std::ostringstream s_int_xbar_clack_c;
    s_int_xbar_clack_c << "int_xbar_clack_c_" << x_id << "_" << y_id;
    int_xbar_clack_c = new DspinLocalCrossbar<dspin_int_cmd_width>(
                     s_int_xbar_clack_c.str().c_str(),
                     mt_int,                       // mapping table
                     x_id, y_id,                   // cluster coordinates
                     x_width, y_width, l_width,
                     1,                            // number of local sources
                     nb_procs,                     // number of local targets 
                     1, 1,                         // fifo depths
                     true,                         // CMD
                     false,                        // don't use local routing table
                     false);                       // broadcast

    //////////////  INT ROUTER(S)
    std::ostringstream s_int_router_cmd;
    s_int_router_cmd << "router_cmd_" << x_id << "_" << y_id;
    int_router_cmd = new VirtualDspinRouter<dspin_int_cmd_width>(
                     s_int_router_cmd.str().c_str(),
                     x_id,y_id,                    // coordinate in the mesh
                     x_width, y_width,             // x & y fields width
                     3,                            // nb virtual channels
                     4,4);                         // input & output fifo depths

    std::ostringstream s_int_router_rsp;
    s_int_router_rsp << "router_rsp_" << x_id << "_" << y_id;
    int_router_rsp = new VirtualDspinRouter<dspin_int_rsp_width>(
                     s_int_router_rsp.str().c_str(),
                     x_id,y_id,                    // router coordinates in mesh
                     x_width, y_width,             // x & y fields width
                     2,                            // nb virtual channels
                     4,4);                         // input & output fifo depths

    //////////////  XRAM
    std::ostringstream s_xram;
    s_xram << "xram_" << x_id << "_" << y_id;
    xram = new VciSimpleRam<vci_param_ext>(
                     s_xram.str().c_str(),
                     IntTab(cluster_id, xram_ram_tgtid ),
                     mt_ram,
                     loader,
                     xram_latency);

    std::ostringstream s_wt_xram;
    s_wt_xram << "xram_wt_" << x_id << "_" << y_id;
    xram_ram_wt = new VciDspinTargetWrapper<vci_param_ext,
                                            dspin_ram_cmd_width,
                                            dspin_ram_rsp_width>(
                     s_wt_xram.str().c_str(),
                     x_width + y_width + l_width);

    /////////////  RAM ROUTER(S)
    std::ostringstream s_ram_router_cmd;
    s_ram_router_cmd << "ram_router_cmd_" << x_id << "_" << y_id;
    size_t is_iob0 = (x_id == 0) and (y_id == 0);
    size_t is_iob1 = (x_id == (xmax-1)) and (y_id == (ymax-1));
    ram_router_cmd = new DspinRouterTsar<dspin_ram_cmd_width>(
                     s_ram_router_cmd.str().c_str(),
                     x_id, y_id,                     // router coordinates in mesh
                     x_width,                        // x field width in first flit
                     y_width,                        // y field width in first flit
                     4, 4,                           // input & output fifo depths
                     is_iob0,                        // cluster contains IOB0
                     is_iob1,                        // cluster contains IOB1
                     false,                          // not a response router
                     l_width);                       // local field width in first flit

    std::ostringstream s_ram_router_rsp;
    s_ram_router_rsp << "ram_router_rsp_" << x_id << "_" << y_id;
    ram_router_rsp = new DspinRouterTsar<dspin_ram_rsp_width>(
                     s_ram_router_rsp.str().c_str(),
                     x_id, y_id,                     // coordinates in mesh
                     x_width,                        // x field width in first flit
                     y_width,                        // y field width in first flit
                     4, 4,                           // input & output fifo depths
                     is_iob0,                        // cluster contains IOB0
                     is_iob1,                        // cluster contains IOB1
                     true,                           // response router
                     l_width);                       // local field width in first flit


    ////////////////////// I/O  CLUSTER ONLY    ///////////////////////
    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
        ///////////  IO_BRIDGE
        size_t iox_local_id;
        size_t global_id;
        bool   has_irqs;
        if ( cluster_id == cluster_iob0 ) 
        {
            iox_local_id = 0;
            global_id    = cluster_iob0;
            has_irqs     = true;
        }
        else
        {
            iox_local_id = 1;
            global_id    = cluster_iob1;
            has_irqs     = false;
        }

        std::ostringstream s_iob;
        s_iob << "iob_" << x_id << "_" << y_id;   
        iob = new VciIoBridge<vci_param_int,
                              vci_param_ext>(  
                     s_iob.str().c_str(),
                     mt_ram,                               // EXT network maptab
                     mt_int,                               // INT network maptab
                     mt_iox,                               // IOX network maptab
                     IntTab( global_id, iobx_int_tgtid ),  // INT TGTID
                     IntTab( global_id, iobx_int_srcid ),  // INT SRCID
                     IntTab( global_id, iox_local_id   ),  // IOX TGTID
                     has_irqs, 
                     16,                                   // cache line words
                     8,                                    // IOTLB ways
                     8,                                    // IOTLB sets
                     debug_start_cycle,
                     iob_debug_ok );
        
        std::ostringstream s_iob_int_wi;
        s_iob_int_wi << "iob_int_wi_" << x_id << "_" << y_id;   
        iob_int_wi = new VciDspinInitiatorWrapper<vci_param_int,
                                                  dspin_int_cmd_width,
                                                  dspin_int_rsp_width>(
                     s_iob_int_wi.str().c_str(),
                     x_width + y_width + l_width);

        std::ostringstream s_iob_int_wt;
        s_iob_int_wt << "iob_int_wt_" << x_id << "_" << y_id;   
        iob_int_wt = new VciDspinTargetWrapper<vci_param_int,
                                               dspin_int_cmd_width,
                                               dspin_int_rsp_width>(
                     s_iob_int_wt.str().c_str(),
                     x_width + y_width + l_width);

        std::ostringstream s_iob_ram_wi;
        s_iob_ram_wi << "iob_ram_wi_" << x_id << "_" << y_id;   
        iob_ram_wi = new VciDspinInitiatorWrapper<vci_param_ext,
                                                  dspin_ram_cmd_width,
                                                  dspin_ram_rsp_width>(
                     s_iob_ram_wi.str().c_str(),
                     x_width + y_width + l_width);
    } // end if IO

    ////////////////////////////////////
    // Connections are defined here
    ////////////////////////////////////

    // on coherence network : local srcid[proc] in [0...nb_procs-1]
    //                      : local srcid[memc] = nb_procs
    // In cluster_iob0, 32 HWI interrupts from external peripherals
    // are connected to the XICU ports p_hwi[0:31] 
    // In other clusters, no HWI interrupts are connected to XICU
 
    //////////////////////// internal CMD & RSP routers
    int_router_cmd->p_clk                        (this->p_clk);
    int_router_cmd->p_resetn                     (this->p_resetn);
    int_router_rsp->p_clk                        (this->p_clk);
    int_router_rsp->p_resetn                     (this->p_resetn);

    for (int i = 0; i < 4; i++)
    {
        for(int k = 0; k < 3; k++)
        {
            int_router_cmd->p_out[i][k]          (this->p_dspin_int_cmd_out[i][k]);
            int_router_cmd->p_in[i][k]           (this->p_dspin_int_cmd_in[i][k]);
        }

        for(int k = 0; k < 2; k++)
        {
            int_router_rsp->p_out[i][k]          (this->p_dspin_int_rsp_out[i][k]);
            int_router_rsp->p_in[i][k]           (this->p_dspin_int_rsp_in[i][k]);
        }
    }

    // local ports
    int_router_cmd->p_out[4][0]                  (signal_int_dspin_cmd_g2l_d);
    int_router_cmd->p_out[4][1]                  (signal_int_dspin_m2p_g2l_c);
    int_router_cmd->p_out[4][2]                  (signal_int_dspin_clack_g2l_c);
    int_router_cmd->p_in[4][0]                   (signal_int_dspin_cmd_l2g_d);
    int_router_cmd->p_in[4][1]                   (signal_int_dspin_m2p_l2g_c);
    int_router_cmd->p_in[4][2]                   (signal_int_dspin_clack_l2g_c);
    
    int_router_rsp->p_out[4][0]                  (signal_int_dspin_rsp_g2l_d);
    int_router_rsp->p_out[4][1]                  (signal_int_dspin_p2m_g2l_c);
    int_router_rsp->p_in[4][0]                   (signal_int_dspin_rsp_l2g_d);
    int_router_rsp->p_in[4][1]                   (signal_int_dspin_p2m_l2g_c);

    ///////////////////// CMD DSPIN  local crossbar direct
    int_xbar_cmd_d->p_clk                        (this->p_clk);
    int_xbar_cmd_d->p_resetn                     (this->p_resetn);
    int_xbar_cmd_d->p_global_out                 (signal_int_dspin_cmd_l2g_d);
    int_xbar_cmd_d->p_global_in                  (signal_int_dspin_cmd_g2l_d);

    int_xbar_cmd_d->p_local_out[memc_int_tgtid]  (signal_int_dspin_cmd_memc_t);
    int_xbar_cmd_d->p_local_out[xicu_int_tgtid]  (signal_int_dspin_cmd_xicu_t);
    int_xbar_cmd_d->p_local_out[mdma_int_tgtid]  (signal_int_dspin_cmd_mdma_t);

    int_xbar_cmd_d->p_local_in[mdma_int_srcid]   (signal_int_dspin_cmd_mdma_i);

    for (size_t p = 0; p < nb_procs; p++)
    int_xbar_cmd_d->p_local_in[proc_int_srcid+p] (signal_int_dspin_cmd_proc_i[p]);

    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
    int_xbar_cmd_d->p_local_out[iobx_int_tgtid]  (signal_int_dspin_cmd_iobx_t);
    int_xbar_cmd_d->p_local_in[iobx_int_srcid]   (signal_int_dspin_cmd_iobx_i);
    }

    //////////////////////// RSP DSPIN  local crossbar direct
    int_xbar_rsp_d->p_clk                        (this->p_clk);
    int_xbar_rsp_d->p_resetn                     (this->p_resetn);
    int_xbar_rsp_d->p_global_out                 (signal_int_dspin_rsp_l2g_d);
    int_xbar_rsp_d->p_global_in                  (signal_int_dspin_rsp_g2l_d);

    int_xbar_rsp_d->p_local_in[memc_int_tgtid]   (signal_int_dspin_rsp_memc_t);
    int_xbar_rsp_d->p_local_in[xicu_int_tgtid]   (signal_int_dspin_rsp_xicu_t);
    int_xbar_rsp_d->p_local_in[mdma_int_tgtid]   (signal_int_dspin_rsp_mdma_t);

    int_xbar_rsp_d->p_local_out[mdma_int_srcid]  (signal_int_dspin_rsp_mdma_i);

    for (size_t p = 0; p < nb_procs; p++)
    int_xbar_rsp_d->p_local_out[proc_int_srcid+p] (signal_int_dspin_rsp_proc_i[p]);

    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
    int_xbar_rsp_d->p_local_in[iobx_int_tgtid]   (signal_int_dspin_rsp_iobx_t);
    int_xbar_rsp_d->p_local_out[iobx_int_srcid]  (signal_int_dspin_rsp_iobx_i);
    }

    ////////////////////// M2P DSPIN local crossbar coherence
    int_xbar_m2p_c->p_clk                        (this->p_clk);
    int_xbar_m2p_c->p_resetn                     (this->p_resetn);
    int_xbar_m2p_c->p_global_out                 (signal_int_dspin_m2p_l2g_c);
    int_xbar_m2p_c->p_global_in                  (signal_int_dspin_m2p_g2l_c);
    int_xbar_m2p_c->p_local_in[0]                (signal_int_dspin_m2p_memc);
    for (size_t p = 0; p < nb_procs; p++) 
        int_xbar_m2p_c->p_local_out[p]           (signal_int_dspin_m2p_proc[p]);

    ////////////////////////// P2M DSPIN local crossbar coherence
    int_xbar_p2m_c->p_clk                        (this->p_clk);
    int_xbar_p2m_c->p_resetn                     (this->p_resetn);
    int_xbar_p2m_c->p_global_out                 (signal_int_dspin_p2m_l2g_c);
    int_xbar_p2m_c->p_global_in                  (signal_int_dspin_p2m_g2l_c);
    int_xbar_p2m_c->p_local_out[0]               (signal_int_dspin_p2m_memc);
    for (size_t p = 0; p < nb_procs; p++) 
        int_xbar_p2m_c->p_local_in[p]            (signal_int_dspin_p2m_proc[p]);

    ////////////////////// CLACK DSPIN local crossbar coherence
    int_xbar_clack_c->p_clk                      (this->p_clk);
    int_xbar_clack_c->p_resetn                   (this->p_resetn);
    int_xbar_clack_c->p_global_out               (signal_int_dspin_clack_l2g_c);
    int_xbar_clack_c->p_global_in                (signal_int_dspin_clack_g2l_c);
    int_xbar_clack_c->p_local_in[0]              (signal_int_dspin_clack_memc);
    for (size_t p = 0; p < nb_procs; p++)
        int_xbar_clack_c->p_local_out[p]         (signal_int_dspin_clack_proc[p]);

    //////////////////////////////////// Processors
    for (size_t p = 0; p < nb_procs; p++)
    {
        proc[p]->p_clk                           (this->p_clk);
        proc[p]->p_resetn                        (this->p_resetn);
        proc[p]->p_vci                           (signal_int_vci_ini_proc[p]);
        proc[p]->p_dspin_m2p                     (signal_int_dspin_m2p_proc[p]);
        proc[p]->p_dspin_p2m                     (signal_int_dspin_p2m_proc[p]);
        proc[p]->p_dspin_clack                   (signal_int_dspin_clack_proc[p]);
        proc[p]->p_irq[0]                        (signal_proc_it[p]);
        for ( size_t j = 1 ; j < 6 ; j++)
        {
            proc[p]->p_irq[j]                    (signal_false);
        }

        proc_wi[p]->p_clk                        (this->p_clk);
        proc_wi[p]->p_resetn                     (this->p_resetn);
        proc_wi[p]->p_dspin_cmd                  (signal_int_dspin_cmd_proc_i[p]);
        proc_wi[p]->p_dspin_rsp                  (signal_int_dspin_rsp_proc_i[p]);
        proc_wi[p]->p_vci                        (signal_int_vci_ini_proc[p]);
    }

    ///////////////////////////////////// XICU
    xicu->p_clk                                  (this->p_clk);
    xicu->p_resetn                               (this->p_resetn);
    xicu->p_vci                                  (signal_int_vci_tgt_xicu);
    for ( size_t p=0 ; p<nb_procs ; p++)
    {
        xicu->p_irq[p]                           (signal_proc_it[p]);
    }
    for ( size_t i=0 ; i<32 ; i++)
    {
        if (cluster_id == cluster_iob0) 
            xicu->p_hwi[i]                       (*(this->p_irq[i]));
        else  
            xicu->p_hwi[i]                       (signal_false);
    }                     

    // wrapper XICU
    xicu_int_wt->p_clk                           (this->p_clk);
    xicu_int_wt->p_resetn                        (this->p_resetn);
    xicu_int_wt->p_dspin_cmd                     (signal_int_dspin_cmd_xicu_t);
    xicu_int_wt->p_dspin_rsp                     (signal_int_dspin_rsp_xicu_t);
    xicu_int_wt->p_vci                           (signal_int_vci_tgt_xicu);

    ///////////////////////////////////// MEMC
    memc->p_clk                                  (this->p_clk);
    memc->p_resetn                               (this->p_resetn);
    memc->p_vci_ixr                              (signal_ram_vci_ini_memc);
    memc->p_vci_tgt                              (signal_int_vci_tgt_memc);
    memc->p_dspin_p2m                            (signal_int_dspin_p2m_memc);
    memc->p_dspin_m2p                            (signal_int_dspin_m2p_memc);
    memc->p_dspin_clack                          (signal_int_dspin_clack_memc);
    memc->p_irq                                  (signal_irq_memc);

    // wrapper to INT network
    memc_int_wt->p_clk                           (this->p_clk);
    memc_int_wt->p_resetn                        (this->p_resetn);
    memc_int_wt->p_dspin_cmd                     (signal_int_dspin_cmd_memc_t);
    memc_int_wt->p_dspin_rsp                     (signal_int_dspin_rsp_memc_t);
    memc_int_wt->p_vci                           (signal_int_vci_tgt_memc);

    // wrapper to RAM network
    memc_ram_wi->p_clk                           (this->p_clk);
    memc_ram_wi->p_resetn                        (this->p_resetn);
    memc_ram_wi->p_dspin_cmd                     (signal_ram_dspin_cmd_memc_i);
    memc_ram_wi->p_dspin_rsp                     (signal_ram_dspin_rsp_memc_i);
    memc_ram_wi->p_vci                           (signal_ram_vci_ini_memc);

    //////////////////////////////////// XRAM
    xram->p_clk                                  (this->p_clk);
    xram->p_resetn                               (this->p_resetn);
    xram->p_vci                                  (signal_ram_vci_tgt_xram);

    // wrapper to RAM network
    xram_ram_wt->p_clk                           (this->p_clk);
    xram_ram_wt->p_resetn                        (this->p_resetn);
    xram_ram_wt->p_dspin_cmd                     (signal_ram_dspin_cmd_xram_t);
    xram_ram_wt->p_dspin_rsp                     (signal_ram_dspin_rsp_xram_t);
    xram_ram_wt->p_vci                           (signal_ram_vci_tgt_xram);

    /////////////////////////////////// MDMA
    mdma->p_clk                                  (this->p_clk);
    mdma->p_resetn                               (this->p_resetn);
    mdma->p_vci_target                           (signal_int_vci_tgt_mdma);
    mdma->p_vci_initiator                        (signal_int_vci_ini_mdma);
    for (size_t i=0 ; i<nb_dmas ; i++)
        mdma->p_irq[i]                           (signal_irq_mdma[i]);

    // target wrapper
    mdma_int_wt->p_clk                           (this->p_clk);
    mdma_int_wt->p_resetn                        (this->p_resetn);
    mdma_int_wt->p_dspin_cmd                     (signal_int_dspin_cmd_mdma_t);
    mdma_int_wt->p_dspin_rsp                     (signal_int_dspin_rsp_mdma_t);
    mdma_int_wt->p_vci                           (signal_int_vci_tgt_mdma);

    // initiator wrapper 
    mdma_int_wi->p_clk                           (this->p_clk);
    mdma_int_wi->p_resetn                        (this->p_resetn);
    mdma_int_wi->p_dspin_cmd                     (signal_int_dspin_cmd_mdma_i);
    mdma_int_wi->p_dspin_rsp                     (signal_int_dspin_rsp_mdma_i);
    mdma_int_wi->p_vci                           (signal_int_vci_ini_mdma);

    //////////////////////////// RAM network CMD & RSP routers
    ram_router_cmd->p_clk                    (this->p_clk);
    ram_router_cmd->p_resetn                 (this->p_resetn);
    ram_router_rsp->p_clk                    (this->p_clk);
    ram_router_rsp->p_resetn                 (this->p_resetn);
    for( size_t n=0 ; n<4 ; n++)
    {
        ram_router_cmd->p_out[n]             (this->p_dspin_ram_cmd_out[n]);
        ram_router_cmd->p_in[n]              (this->p_dspin_ram_cmd_in[n]);
        ram_router_rsp->p_out[n]             (this->p_dspin_ram_rsp_out[n]);
        ram_router_rsp->p_in[n]              (this->p_dspin_ram_rsp_in[n]);
    }
    ram_router_cmd->p_out[4]                 (signal_ram_dspin_cmd_xram_t);
    ram_router_cmd->p_in[4]                  (signal_ram_dspin_cmd_memc_i);
    ram_router_rsp->p_out[4]                 (signal_ram_dspin_rsp_memc_i);
    ram_router_rsp->p_in[4]                  (signal_ram_dspin_rsp_xram_t);
   
    ///////////////////////// IOB exists only in cluster_iob0 & cluster_iob1. 
    if ( (cluster_id == cluster_iob0) or (cluster_id == cluster_iob1) )
    {
        // IO bridge
        iob->p_clk                               (this->p_clk);
        iob->p_resetn                            (this->p_resetn);
        iob->p_vci_ini_iox                       (*(this->p_vci_iob_iox_ini));
        iob->p_vci_tgt_iox                       (*(this->p_vci_iob_iox_tgt));
        iob->p_vci_tgt_int                       (signal_int_vci_tgt_iobx);
        iob->p_vci_ini_int                       (signal_int_vci_ini_iobx);
        iob->p_vci_ini_ram                       (signal_ram_vci_ini_iobx);

        if ( cluster_id == cluster_iob0 )
               for ( size_t n=0 ; n<32 ; n++ )
                   (*iob->p_irq[n])                 (*(this->p_irq[n]));

        // initiator wrapper to RAM network
        iob_ram_wi->p_clk                        (this->p_clk);
        iob_ram_wi->p_resetn                     (this->p_resetn);
        iob_ram_wi->p_dspin_cmd                  (*(this->p_dspin_iob_cmd_out));
        iob_ram_wi->p_dspin_rsp                  (*(this->p_dspin_iob_rsp_in));
        iob_ram_wi->p_vci                        (signal_ram_vci_ini_iobx);

        // initiator wrapper to INT network
        iob_int_wi->p_clk                        (this->p_clk);
        iob_int_wi->p_resetn                     (this->p_resetn);
        iob_int_wi->p_dspin_cmd                  (signal_int_dspin_cmd_iobx_i);
        iob_int_wi->p_dspin_rsp                  (signal_int_dspin_rsp_iobx_i);
        iob_int_wi->p_vci                        (signal_int_vci_ini_iobx);

        // target wrapper to INT network
        iob_int_wt->p_clk                        (this->p_clk);
        iob_int_wt->p_resetn                     (this->p_resetn);
        iob_int_wt->p_dspin_cmd                  (signal_int_dspin_cmd_iobx_t);
        iob_int_wt->p_dspin_rsp                  (signal_int_dspin_rsp_iobx_t);
        iob_int_wt->p_vci                        (signal_int_vci_tgt_iobx);
    }

} // end constructor

}}


// Local Variables:
// tab-width: 3
// c-basic-offset: 3
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=3:tabstop=3:softtabstop=3



