/**
 * \file   reset_ioc.c
 * \date   December 2013
 * \author Cesar Fuguet
 *
 * \brief  API for accessing the disk controller
 *
 * \note   These functions call the specific disk controller driver depending
 *         on the USE_IOC_BDV, USE_IOC_SPI or USE_IOC_RDK constants
 */

#include <reset_ioc.h>
#include <defs.h>

#if !defined(USE_IOC_BDV) && !defined(USE_IOC_SPI) && !defined(USE_IOC_RDK)
#   error "One of the USE_IOC_* constants must be defined in the hard_config.h"
#endif

#if (USE_IOC_BDV + USE_IOC_SPI + USE_IOC_RDK) != 1
#   error "Only one disk controller must be used"
#endif

#if USE_IOC_SPI
#include <reset_sdc.h>
#endif

#if USE_IOC_BDV
#include <reset_bdv.h>
#endif

#if USE_IOC_RDK
#include <reset_rdk.h>
#endif

/**
 * \brief Initialize the disk controller
 */
int reset_ioc_init()
{
#if USE_IOC_BDV
    return reset_bdv_init();
#elif USE_IOC_SPI
    return reset_sdc_init();
#elif USE_IOC_RDK
    return reset_rdk_init();
#else
#   error "reset_ioc_init() : Not supported disk controller chosen"
#endif
}

/**
 * \param lba   : first block index on the disk
 * \param buffer: base address of the memory buffer
 * \param count : number of blocks to be transfered
 *
 * \brief Transfer data from disk to a memory buffer
 *
 * \note  This is a blocking function. The function returns once the transfer
 *        is completed.
 */
int reset_ioc_read( unsigned int lba, void* buffer, unsigned int count )
{
#if USE_IOC_BDV
    return reset_bdv_read(lba, buffer, count);
#elif USE_IOC_SPI
    return reset_sdc_read(lba, buffer, count);
#elif USE_IOC_RDK
    return reset_rdk_read(lba, buffer, count);
#else
#   error "reset_ioc_read() : Not supported disk controller chosen"
#endif
}

/*
 * vim: tabstop=4 : softtabstop=4 : shiftwidth=4 : expandtab
 */
