#!/usr/bin/env python

from math import log, ceil
from genarch import *

#########################################################################################
#   file   : arch_info.py  (for the tsar_generic_iob architecture)
#   date   : may 2014
#   author : Alain Greiner
#########################################################################################
#  This python script defines a specific instance of "tsar_generic_iob" architecture
#  for the ALMOS-MK operating system. It is used to generate the "hard_config.h" 
#  and the "arch_info.bin files, used by bthe ALMOS-MK bootloader.
#
#  The "tsar_generic_iob" architecture includes 7 external peripherals, accessed
#  through an IOB components located in cluster [0,0] or in cluster [x_size-1, y_size-1].
#  Available peripherals are: TTY, IOC, FBF, ROM, NIC, CMA, PIC.
#  All clusters contain (nb_cores) processors, one L2 cache, one XCU, and
#  one optional hardware coprocessor connected to a MWMR controller.
#
#  The following parameters are constructor arguments:
#  - x_size         : number of clusters in a row (from 1 to 16)
#  - y_size         : number of clusters in a column (from & to 16)
#  - nb_cores       : number of processors per cluster (from 1 to 4)
#  - nb_ttys        : number of TTY channels (can be from 1 to 8)
#  - fbf_width      : frame_buffer width = frame_buffer heigth
#  - ioc_type       : can be 'IOC_BDV','IOC_HBA','IOC_SDC', 'IOC_SPI','NONE'
#  - mwr_type       : coprocessor type (can be 'MWR_GCD','MWR_DCT','MWR_CPY','NONE')
#  - nb_nics        : number of NIC channels (can be from 1 to 2)
#  - nb_cmas        : number of CMA channels (can be from 1 to 4)
#  - io_cxy         : cluster_io identifier
#  - boot_cxy       : boot cluster identifier
#
#  The following parameters are imposed by the "tsar_generic_iob" architecture:
#  - devices_max    : max number of devices per cluster
#  - x_width        : number of bits for x coordinate
#  - y_width        : number of bits for y coordinate
#  - paddr_width    : number of bits for physical address
#  - irqs_per_core  : number of input IRQs per processor
########################################################################################

############################
def arch( x_size        = 2,
          y_size        = 2,
          nb_cores      = 2,
          nb_ttys       = 1,
          fbf_width     = 128,
          ioc_type      = 'IOC_BDV',
          mwr_type      = 'MWR_CPY',
          nb_nics       = 1, 
          nb_cmas       = 2,
          io_cxy        = 0,
          boot_cxy      = 0 ):

    ### architecture constants

    p_width       = 2
    x_width       = 4
    y_width       = 4
    paddr_width   = 40
    irqs_per_core = 4           
    devices_max   = 16  

    ### constructor parameters checking

    assert( (x_size == 1) or (x_size == 2) or (x_size == 4)
             or (x_size == 8) or (x_size == 16) )

    assert( (y_size == 1) or (y_size == 2) or (y_size == 4)
             or (y_size == 8) or (y_size == 16) )

    assert( nb_cores <= 4 )

    assert( (nb_ttys >= 1) and (nb_ttys <= 8) )

    assert( (nb_nics >= 1) and (nb_nics <= 2) )

    assert( (nb_cmas >= 1) and (nb_cmas <= 4) )

    assert( ioc_type in ['IOC_BDV','IOC_HBA','IOC_SDC','IOC_SPI','IOC_RDK'] )

    assert( mwr_type in ['MWR_GCD','MWR_DCT','MWR_CPY'] )

    assert( (io_cxy == 0) or (io_cxy == ((x_size-1)<<y_width) + (y_size-1)) ) 

    assert( ((boot_cxy >> y_width) < x_size) and ((boot_cxy & ((1<<y_width)-1)) < y_size) )
 
    ### define platform name

    platform_name = 'tsar_iob_%d_%d_%d' % ( x_size, y_size , nb_cores )
    platform_name += '_%d_%d_%s_%s' % ( fbf_width , nb_ttys , ioc_type , mwr_type )

    ### define physical segments replicated in all clusters

    ram_base = 0x0000000000
    ram_size = 0x4000000                   # 64 Mbytes

    xcu_base = 0x00B0000000
    xcu_size = 0x1000                      # 4 Kbytes

    mwr_base = 0x00B1000000
    mwr_size = 0x1000                      # 4 Kbytes 

    mmc_base = 0x00B2000000
    mmc_size = 0x1000                      # 4 Kbytes

    ### define physical segments for external peripherals
    ## These segments are only defined in cluster_io

    ioc_base  = 0x00B3000000
    ioc_size  = 0x1000                     # 4 Kbytes

    tty_base  = 0x00B4000000
    tty_size  = 0x4000                     # 16 Kbytes

    nic_base  = 0x00B5000000
    nic_size  = 0x80000                    # 512 kbytes

    cma_base  = 0x00B6000000
    cma_size  = 0x1000 * nb_cmas           # 4 kbytes * nb_cmas

    fbf_base  = 0x00B7000000
    fbf_size  = fbf_width * fbf_width      # fbf_width * fbf_width bytes

    pic_base  = 0x00B8000000
    pic_size  = 0x1000                     # 4 Kbytes

    iob_base  = 0x00BE000000
    iob_size  = 0x1000                     # 4 bytes

    rom_base  = 0x00BFC00000
    rom_size  = 0x4000                     # 16 Kbytes

    ### define  bootloader vsegs base addresses and sizes
    ### We want to pack these 4 vsegs in 2 big pages
    ### => boot cost two BIG pages in cluster[0][0]

    boot_archi_vbase   = 0x00000000           # ident
    boot_archi_size    = 0x00100000           # 1 Mbytes

    boot_code_vbase      = 0x00100000           # ident
    boot_code_size       = 0x00080000           # 512 Kbytes

    boot_stack_vbase     = 0x00180000           # ident
    boot_stack_size      = 0x00080000           # 512 Kbytes

    boot_data_vbase      = 0x00200000           # ident
    boot_data_size       = 0x00200000           # 2 Mbytes

    ### define kernel vsegs base addresses and sizes
    ### code, init, ptab, heap & sched vsegs are replicated in all clusters.
    ### data & uncdata vsegs are only mapped in cluster[0][0].

    kernel_code_vbase    = 0x80000000
    kernel_code_size     = 0x00200000           # 2 Mbytes per cluster

    kernel_data_vbase    = 0x90000000
    kernel_data_size     = 0x00200000           # 2 Mbytes in cluster[0,0]

    kernel_ptab_vbase    = 0xE0000000
    kernel_ptab_size     = 0x00200000           # 2 Mbytes per cluster

    kernel_heap_vbase    = 0xD0000000
    kernel_heap_size     = 0x00400000           # 4 Mbytes per cluster

    kernel_sched_vbase   = 0xA0000000   
    kernel_sched_size    = 0x00002000*nb_cores  # 8 Kbytes per proc per cluster

    ############################
    ### call Header constructor
    ############################

    archi = Root( name           = platform_name,
                  x_size         = x_size,
                  y_size         = y_size,
                  cores_max      = nb_cores,
                  devices_max    = devices_max,
                  paddr_width    = paddr_width,
                  x_width        = x_width,
                  y_width        = y_width,
                  irqs_per_core  = irqs_per_core,
                  use_ramdisk    = (ioc_type == 'RDK'),
                  io_cxy         = io_cxy,          
                  boot_cxy       = boot_cxy,
                  reset_address  = rom_base )

    ##############################################
    ### construct replicated hardware components 
    ##############################################

    for x in xrange( x_size ):
        for y in xrange( y_size ):
            cxy = (x << y_width) + y;
            offset     = cxy << (paddr_width - x_width - y_width)

            ram = archi.addDevice( ptype    = 'RAM' , 
                                   base     = ram_base + offset, 
                                   size     = ram_size )

            xcu = archi.addDevice( ptype    = 'XCU', 
                                   base     = xcu_base + offset, 
                                   size     = xcu_size,
                                   channels = nb_cores * irqs_per_core, 
                                   arg0     = 16, 
                                   arg1     = 16,
                                   arg2     = 16 )

            mmc = archi.addDevice( ptype    = 'MMC',
                                   base     = mmc_base + offset,
                                   size     = mmc_size )
            archi.addIrq( dstdev = xcu, port = 0, srcdev = mmc, isrtype = 'ISR_MMC' )

            if ( mwr_type == 'MWR_GCD' ):
                mwr = archi.addDevice( ptype = 'MWR_GCD',
                                       base  = mwr_base + offset,
                                       size  = mwr_size,
                                       arg0  = 2, 
                                       arg1  = 1,
                                       arg2  = 1, 
                                       arg3  = 0 )
                archi.addIrq( dstdev = xcu, port = 1, srcdev = mwr, isrtype = 'ISR_MWR' )

            if ( mwr_type == 'MWR_DCT' ):
                mwr = archi.addDevice( ptype = 'MWR_DCT',
                                       base  = mwr_base + offset,
                                       size  = mwr_size,
                                       arg0  = 1, 
                                       arg1  = 1,
                                       arg2  = 1, 
                                       arg3  = 0 )
                archi.addIrq( dstdev = xcu, port = 1, srcdev = mwr, isrtype = 'ISR_MWR' )

            if ( mwr_type == 'MWR_CPY' ):
                mwr = archi.addDevice( ptype = 'MWR_CPY',
                                       base  = mwr_base + offset,
                                       size  = mwr_size,
                                       arg0  = 1, 
                                       arg1  = 1,
                                       arg2  = 1, 
                                       arg3  = 0 )
                archi.addIrq( dstdev = xcu, port = 1, srcdev = mwr, isrtype = 'ISR_MWR' )

            for p in xrange ( nb_cores ):
                archi.addCore( (x<<(y_width+p_width)) + (y<<p_width) + p,  # hardware identifier
                               (x<<y_width) + y,                           # cluster identifier
                               p )                                         # local index

    #################################################
    ### construct hardware components in IO cluster  
    #################################################

    offset = io_cxy << (paddr_width - x_width - y_width)

    iob = archi.addDevice( ptype    = 'IOB', 
                           base     = iob_base + offset,
                           size     = iob_size )

    ioc = archi.addDevice( ptype    = ioc_type,
                           base     = ioc_base + offset,
                           size     = ioc_size )

    tty = archi.addDevice( ptype    = 'TTY',
                           base     = tty_base + offset,
                           size     = tty_size, 
                           channels = nb_ttys )

    nic = archi.addDevice( ptype    = 'NIC',
                           base     = nic_base + offset, 
                           size     = nic_size, 
                           channels = nb_nics )

    cma = archi.addDevice( ptype    = 'CMA',
                           base     = cma_base + offset,
                           size     = cma_size, 
                           channels = nb_cmas )

    fbf = archi.addDevice( ptype    = 'FBF',
                           base     = fbf_base + offset,
                           size     = fbf_size, 
                           arg0     = fbf_width,
                           arg1     = fbf_width )

    rom = archi.addDevice( ptype    = 'ROM',
                           base     = rom_base + offset, 
                           size     = rom_size )

    pic = archi.addDevice( ptype    ='PIC',
                           base     = pic_base + offset,
                           size     = pic_size, 
                           arg0     = 32 )

    if   ( ioc_type == 'IOC_BDV' ): isr_ioc = 'ISR_BDV'
    elif ( ioc_type == 'IOC_HBA' ): isr_ioc = 'ISR_HBA'
    elif ( ioc_type == 'IOC_SDC' ): isr_ioc = 'ISR_SDC'
    elif ( ioc_type == 'IOC_SPI' ): isr_ioc = 'ISR_SPI'
    else                          : isr_ioc = 'ISR_DEFAULT'

    archi.addIrq( dstdev = pic, port = 0 , srcdev = nic, isrtype = 'ISR_NIC_RX', channel = 0 )
    archi.addIrq( dstdev = pic, port = 1 , srcdev = nic, isrtype = 'ISR_NIC_RX', channel = 1 )
    archi.addIrq( dstdev = pic, port = 2 , srcdev = nic, isrtype = 'ISR_NIC_TX', channel = 0 )
    archi.addIrq( dstdev = pic, port = 3 , srcdev = nic, isrtype = 'ISR_NIC_TX', channel = 1 )
    archi.addIrq( dstdev = pic, port = 4 , srcdev = cma, isrtype = 'ISR_CMA'   , channel = 0 )
    archi.addIrq( dstdev = pic, port = 5 , srcdev = cma, isrtype = 'ISR_CMA'   , channel = 1 )
    archi.addIrq( dstdev = pic, port = 6 , srcdev = cma, isrtype = 'ISR_CMA'   , channel = 2 )
    archi.addIrq( dstdev = pic, port = 7 , srcdev = cma, isrtype = 'ISR_CMA'   , channel = 3 )
    archi.addIrq( dstdev = pic, port = 8 , srcdev = ioc, isrtype = isr_ioc     , channel = 0 )
    archi.addIrq( dstdev = pic, port = 16, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 0 )
    archi.addIrq( dstdev = pic, port = 17, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 1 )
    archi.addIrq( dstdev = pic, port = 18, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 2 )
    archi.addIrq( dstdev = pic, port = 19, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 3 )
    archi.addIrq( dstdev = pic, port = 20, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 4 )
    archi.addIrq( dstdev = pic, port = 21, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 5 )
    archi.addIrq( dstdev = pic, port = 22, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 6 )
    archi.addIrq( dstdev = pic, port = 23, srcdev = tty, isrtype = 'ISR_TTY_RX', channel = 7 )

    return archi

################################# platform test ####################################

if __name__ == '__main__':

    archi = arch()

    print archi.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

