
 /* SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Author   : Abdelmalek SI MERABET
 * Date     : Februrary 2011
 * Copyright: UPMC - LIP6
 */
#ifndef DSPIN_HALF_GATEWAY_INITIATOR_FAST_H
#define DSPIN_HALF_GATEWAY_INITIATOR_FAST_H

#include "caba_base_module.h"
#include "generic_fifo.h"
#include "mapping_table.h"
#include "ring_signals_fast.h"
#include "dspin_interface.h"

#define  BRDCST_TGT_MASK  0xFFFFFFFFE1ULL
#define  HI_DEBUG

namespace soclib { namespace caba {

namespace {

const char *ring_cmd_fsm_state_str_hi[] = {
        "CMD_IDLE",
        "OWNER",
	"BDC_FIRST",
	"BDC_SECOND",
        "SENDING",
};

#ifdef HI_DEBUG

const char *ring_rsp_fsm_state_str_hi[] = {
        "RSP_IDLE",
        "ALLOC",
        "NALLOC",
};
#endif

}
template<typename vci_param, int ring_cmd_data_size, int ring_rsp_data_size>
class DspinHalfGatewayInitiatorFastC
{

typedef LocalRingSignals ring_signal_t;
typedef DspinInput<ring_cmd_data_size>   cmd_in_t;
typedef DspinOutput<ring_rsp_data_size>  rsp_out_t;

private:
        
        enum ring_rsp_fsm_state_e {
            	RSP_IDLE,    // waiting for first flit of a response packet
            	ALLOC,      // next flit of a local rsp packet
            	NALLOC,     // next flit of a ring rsp packet
            };
        
        enum ring_cmd_fsm_state_e {
            	CMD_IDLE,	    
             	OWNER,  
		BDC_FIRST,
		BDC_SECOND,	
            	SENDING,
                PREEMPT,          	    
            };
        
        // structural parameters
 	std::string         m_name;
        bool                m_alloc_init;

        
        // internal fifos 
        GenericFifo<uint64_t > m_cmd_fifo;     // fifo for the local command packet
        GenericFifo<uint64_t > m_rsp_fifo;     // fifo for the local response packet
        
        // routing table
        soclib::common::AddressDecodingTable<uint32_t, bool> m_lt;

	uint64_t r_brdcst_save; 
        uint32_t m_srcid;
        uint32_t m_nb_target;
        uint32_t m_x_width;
        uint32_t m_y_width;

        // internal registers
        sc_core::sc_signal<bool>     r_preempt;         // ring preempted, priority to gate init.
        sc_core::sc_signal<uint32_t> r_cpt_tgt;         // target id for sequential broadcast
        sc_core::sc_signal<uint32_t> r_ring_cmd_fsm;    // ring command packet FSM (distributed)
        sc_core::sc_signal<uint32_t> r_ring_rsp_fsm;    // ring response packet FSM

public :

#define __renRegGateInit(x) x((((std::string) name)+"_" #x).c_str())

DspinHalfGatewayInitiatorFastC(
	const char     *name,
        bool            alloc_init,
        const int       &wrapper_fifo_depth,
        const soclib::common::MappingTable &mt,
        const soclib::common::IntTab &ringid,
	const uint32_t &nb_target,
	const uint32_t &x_width,
        const uint32_t &y_width)
      : m_name(name),
        m_alloc_init(alloc_init),
        m_cmd_fifo(((std::string) name)+"m_cmd_fifo", wrapper_fifo_depth),
        m_rsp_fifo(((std::string) name)+"m_rsp_fifo", wrapper_fifo_depth),
        m_lt(mt.getIdLocalityTable(ringid)),
	m_nb_target(nb_target),
	m_x_width(x_width),
	m_y_width(y_width),
        __renRegGateInit(r_cpt_tgt),
        __renRegGateInit(r_ring_cmd_fsm),
        __renRegGateInit(r_ring_rsp_fsm)
 { } //  end constructor


void reset()
{
	if(m_alloc_init)
		r_ring_cmd_fsm = OWNER;
	else
		r_ring_cmd_fsm = CMD_IDLE;

	r_ring_rsp_fsm = RSP_IDLE;
	m_cmd_fifo.init();
	m_rsp_fifo.init();

}

void transition(const cmd_in_t &p_gate_cmd_in, const rsp_out_t &p_gate_rsp_out, const ring_signal_t p_ring_in, cmd_str &init_cmd, bool &init_rsp_val, const bool tga)      
{

	bool      cmd_fifo_get = false;
	bool      cmd_fifo_put = false;
	uint64_t  cmd_fifo_data = 0;

	bool      rsp_fifo_put = false;
	uint64_t  rsp_fifo_data = 0;


//////////// VCI CMD FSM /////////////////////////

	if (p_gate_cmd_in.write.read()) {
		cmd_fifo_data = (uint64_t) p_gate_cmd_in.data.read();
		cmd_fifo_put =  m_cmd_fifo.wok();
	}

	bool rsp_fifo_get = p_gate_rsp_out.read.read();

//////////// RING CMD FSM /////////////////////////
	switch( r_ring_cmd_fsm ) 
	{
		case CMD_IDLE:  
                { 
			bool eop = ( (int) ((m_cmd_fifo.read() >> (ring_cmd_data_size - 1) ) & 0x1) == 1);
                        // tga : target gate allocated
			if(m_cmd_fifo.rok())
			{
#ifdef HI_DEBUG
   std::cout << std::dec << sc_time_stamp() << " - " << m_name 
                         << " - ring_cmd_fsm  = " << ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm] 
                         << " - fifo ROK : " << m_cmd_fifo.rok()
                         << " - fifo _data : " << std::hex << m_cmd_fifo.read()
                         << " - in grant : " << p_ring_in.cmd_grant
                         << " - in tga : " << tga
                         << " - in wok : " << p_ring_in.cmd_r
                         << " - in data : " << p_ring_in.cmd_data

                         << std::endl;
#endif
				r_preempt    = tga;

				if (m_cmd_fifo.read() & 0x1 == 0x1) // broadcast
				{
					if (tga || p_ring_in.cmd_grant)
					{
				                r_cpt_tgt      = 0;                   
				                r_brdcst_save  = m_cmd_fifo.read() & BRDCST_TGT_MASK; // save first flit of brdcst
				                r_ring_cmd_fsm = BDC_FIRST;
					}
				}
				else    if (tga)
					{

						cmd_fifo_get   = p_ring_in.cmd_r;
						if (eop)
				        		r_ring_cmd_fsm = CMD_IDLE;
						else
							r_ring_cmd_fsm = SENDING;
						break;
					}

					if(p_ring_in.cmd_grant)
					{
				        	r_ring_cmd_fsm = SENDING;
					}

			}
		}
		break;

		case OWNER:
			if ( m_cmd_fifo.rok()) // && p_ring_in.cmd_wok.read() ) 
			{
#ifdef HI_DEBUG
   std::cout << std::dec << sc_time_stamp() << " - " << m_name 
                         << " - ring_cmd_fsm  = " << ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm] 
                         << " - fifo ROK : " << m_cmd_fifo.rok()
                         << " - in grant : " << p_ring_in.cmd_grant
                         << " - in wok : " << p_ring_in.cmd_r
                         << " - fifo data : " << std::hex << m_cmd_fifo.read()
                         << std::endl;
#endif       

                        	bool eop = ( (int) ((m_cmd_fifo.read() >> (ring_cmd_data_size - 1) ) & 0x1) == 1);

                                if (m_cmd_fifo.read() & 0x1 == 0x1) // broadcast
                                {
                                        r_cpt_tgt      = 0;                   
                                        r_brdcst_save  = m_cmd_fifo.read() & BRDCST_TGT_MASK; // save first flit of brdcst
                                        r_ring_cmd_fsm = BDC_FIRST;                                        
                                } 
//
				else
				{
					if ( eop && p_ring_in.cmd_r )  
					 {
					 	cmd_fifo_get = true;
					        if ( p_ring_in.cmd_grant )
					 	        r_ring_cmd_fsm = OWNER;
					      	else
					 	        r_ring_cmd_fsm = CMD_IDLE;
					 }  
					
					 if (!eop || !p_ring_in.cmd_r) 
					 {
					 	cmd_fifo_get = p_ring_in.cmd_r;
					 	r_ring_cmd_fsm = SENDING;
					 }

				}
             
			}   
			else if ( !p_ring_in.cmd_grant)
				r_ring_cmd_fsm = CMD_IDLE; 
		break;

                case BDC_FIRST:
                       
#ifdef HI_DEBUG
        std::cout << std::dec << sc_time_stamp() << " - " << m_name
                  << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm] 
                  << " - fifo ROK : " << m_cmd_fifo.rok()
                  << " - in grant : " << p_ring_in.cmd_grant  
                  << " - preempt : " << r_preempt
                  << " - in wok : " << p_ring_in.cmd_r
		  << " - cpt_tgt : " << r_cpt_tgt.read()
                  << " - fifo data : " << std::hex << m_cmd_fifo.read()
                  << std::endl;

#endif

                        if (p_ring_in.cmd_r)                                 
			{
                        	if (r_cpt_tgt.read() == 0)
				{
                                	cmd_fifo_get = true;
				}
                                r_ring_cmd_fsm = BDC_SECOND;
			}
                break;

                case BDC_SECOND:
#ifdef HI_DEBUG
        std::cout << std::dec << sc_time_stamp() << " - " << m_name
                  << " - ring_cmd_fsm = " << ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm] 
                  << " - fifo ROK : " << m_cmd_fifo.rok()
                  << " - in grant : " << p_ring_in.cmd_grant  
                  << " - preempt : " << r_preempt
                  << " - in wok : " << p_ring_in.cmd_r
                  << " - fifo data : " << std::hex << m_cmd_fifo.read()
                  << std::endl;

#endif

          		if(m_cmd_fifo.rok() && p_ring_in.cmd_r)
			{ 
				if(r_cpt_tgt.read() == m_nb_target - 1)
				{
					cmd_fifo_get = true;  

					if ( !r_preempt && p_ring_in.cmd_grant )
					{
						r_ring_cmd_fsm = OWNER;  
					}
					else 	//r_preempt || !p_ring_in.cmd_grant   
					{
						r_ring_cmd_fsm = CMD_IDLE;
					}
				}
				else
				{
					r_cpt_tgt      = r_cpt_tgt + 1;
                                	r_ring_cmd_fsm = BDC_FIRST;
				}

			}
                break;

		case SENDING:   
#ifdef HI_DEBUG
if(m_cmd_fifo.rok())
   std::cout << std::dec << sc_time_stamp() << " - " << m_name 
                         << " - ring_cmd_fsm  = " << ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm] 
                         << " - fifo ROK : " << m_cmd_fifo.rok()
                         << " - in grant : " << p_ring_in.cmd_grant
                         << " - in wok : " << p_ring_in.cmd_r
                         << " - fifo data : " << std::hex << m_cmd_fifo.read()
                         << std::endl;
#endif                 


			if(m_cmd_fifo.rok() && p_ring_in.cmd_r ) 
			{
				cmd_fifo_get = true; 
                                bool eop = ((int) (m_cmd_fifo.read() >> (ring_cmd_data_size - 1) ) & 0x1) == 1;
				if (eop) 
				{  
					if (!r_preempt && p_ring_in.cmd_grant )
						r_ring_cmd_fsm = OWNER;  
					else   
						r_ring_cmd_fsm = CMD_IDLE; 
				}        
			}      
		break;

	} // end switch ring cmd fsm
 
/////////// RING RSP FSM ////////////////////////
    
	switch( r_ring_rsp_fsm ) 
	{
		case RSP_IDLE:  

		{
			uint32_t  rsrcid  = (uint32_t)  ((sc_dt::sc_uint<vci_param::S>) ((p_ring_in.rsp_data >> (ring_rsp_data_size-m_x_width-m_y_width - 1)) << (vci_param::S-m_x_width-m_y_width)));
			bool islocal      = !m_lt[rsrcid];
			bool reop         = ((p_ring_in.rsp_data >> (ring_rsp_data_size - 1)) & 0x1) == 1;



			if (p_ring_in.rsp_w)
                        {
#ifdef HI_DEBUG
   std::cout << std::dec << sc_time_stamp() << " - " << m_name 
              << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_hi[r_ring_rsp_fsm]
              << " - in preempt : " << p_ring_in.rsp_preempt
              << " - in rok : " << p_ring_in.rsp_w
              << " - in data : " << std::hex << p_ring_in.rsp_data
              << " - rsrcid : " << rsrcid
              << " - isloc : " << islocal 
              << " - in wok : " << p_ring_in.rsp_r
              << " - fifo wok : " << m_rsp_fifo.wok()
              << " - reop : " << reop
              << std::endl;
#endif
                                if(islocal) 
			        {   
				        rsp_fifo_put  = m_rsp_fifo.wok();
				        rsp_fifo_data = p_ring_in.rsp_data;

                                        if (reop && m_rsp_fifo.wok())
                                                r_ring_rsp_fsm = RSP_IDLE;
                                        else
                                                r_ring_rsp_fsm = ALLOC;
			        }

			        else  // !islocal 
			        {

                                        if (reop && p_ring_in.rsp_r)
                                                r_ring_rsp_fsm = RSP_IDLE;
                                        else
                                                r_ring_rsp_fsm = NALLOC;
			        }
                        }
			else // !p_ring_in.rsp_w 
		        	r_ring_rsp_fsm = RSP_IDLE;
		}
		break;

		case ALLOC:
		{
			bool reop     = ((p_ring_in.rsp_data >> (ring_rsp_data_size - 1)) & 0x1) == 1;
#ifdef HI_DEBUG
if(p_ring_in.rsp_w || p_ring_in.rsp_preempt)
   std::cout << std::dec << sc_time_stamp() << " - " << m_name 
              << " - ring_rsp_fsm = " << ring_rsp_fsm_state_str_hi[r_ring_rsp_fsm] 
              << " - in preempt : " << p_ring_in.rsp_preempt
              << " - in rok : " << p_ring_in.rsp_w
              << " - in wok : " << p_ring_in.rsp_r 
              << " - in data : " << std::hex << p_ring_in.rsp_data
              << " - fifo wok : " << m_rsp_fifo.wok()	
	      << " - reop : " << reop
              << std::endl;
#endif


                        if(p_ring_in.rsp_preempt) break;

			if (p_ring_in.rsp_w && m_rsp_fifo.wok() && reop)         
			{

				rsp_fifo_put  = true;
				rsp_fifo_data = p_ring_in.rsp_data;

				if(p_ring_in.rsp_palloc)
			     		r_ring_rsp_fsm = NALLOC;
				else
			     		r_ring_rsp_fsm = RSP_IDLE;             
			}
			else //  !p_ring_in.rsp_w || !m_rsp_fifo.wok() || !reop
			{

				rsp_fifo_put  = p_ring_in.rsp_w && m_rsp_fifo.wok();
				rsp_fifo_data = p_ring_in.rsp_data;
				r_ring_rsp_fsm = ALLOC;             
			}
		} 
		break;

		case NALLOC:     
		{

#ifdef HI_DEBUG
if(p_ring_in.rsp_w)
   std::cout << std::dec << sc_time_stamp() << " - " << m_name
                         << " - ring_rsp_fsm  = " << ring_rsp_fsm_state_str_hi[r_ring_rsp_fsm] 
                         << " - in rok : " << p_ring_in.rsp_w
                         << " - fifo wok : " <<  m_rsp_fifo.wok()   
                         << " - in wok : " << p_ring_in.rsp_r 
                         << " - in data : " << std::hex << p_ring_in.rsp_data
                         << std::endl;
#endif

			bool reop  = ((p_ring_in.rsp_data >> (ring_rsp_data_size - 1)) & 0x1) == 1;


			if (p_ring_in.rsp_w && reop && p_ring_in.rsp_r)
			{
				r_ring_rsp_fsm = RSP_IDLE; 
			}
			else
			{
				r_ring_rsp_fsm = NALLOC;
			}
		}
		break;

	} // end switch rsp fsm


    ////////////////////////
    //  fifos update      //
   ////////////////////////

//-- keep trace on ring traffic
        init_cmd.cmdval  = cmd_fifo_get;
        init_cmd.flit    = m_cmd_fifo.read();
        init_cmd.state   = ring_cmd_fsm_state_str_hi[r_ring_cmd_fsm];

	//init_cmd_val = cmd_fifo_get;
	init_rsp_val = rsp_fifo_put;
// local cmd fifo update
	if (  cmd_fifo_put &&  cmd_fifo_get ) m_cmd_fifo.put_and_get(cmd_fifo_data);
	else if (  cmd_fifo_put && !cmd_fifo_get ) m_cmd_fifo.simple_put(cmd_fifo_data);
	else if ( !cmd_fifo_put &&  cmd_fifo_get ) m_cmd_fifo.simple_get();
	
// local rsp fifo update
	if (  rsp_fifo_put &&  rsp_fifo_get ) m_rsp_fifo.put_and_get(rsp_fifo_data);
	else if (  rsp_fifo_put && !rsp_fifo_get ) m_rsp_fifo.simple_put(rsp_fifo_data);
	else if ( !rsp_fifo_put &&  rsp_fifo_get ) m_rsp_fifo.simple_get();
     
}  // end Transition()

///////////////////////////////////////////////////////////////////
void genMoore(cmd_in_t &p_gate_cmd_in, rsp_out_t &p_gate_rsp_out)
///////////////////////////////////////////////////////////////////
{
	p_gate_rsp_out.write = m_rsp_fifo.rok();
	p_gate_rsp_out.data  = (sc_dt::sc_uint<ring_rsp_data_size>) m_rsp_fifo.read();

	p_gate_cmd_in.read= m_cmd_fifo.wok();

} // end genMoore

/////////////////////////////////////////////////////////////////////////////////////////////////
void update_ring_signals(ring_signal_t p_ring_in, ring_signal_t &p_ring_out, bool tga, bool &iga)
/////////////////////////////////////////////////////////////////////////////////////////////////
// palloc may have 2 values :
// palloc = 2 : means Target Gate still allocated
// global state for targets :  TG  TL0  TL1   TL2  ... TLn
//                              A   N    N     P2       N
//                              A   N    N     N        N
// palloc = 1 : means Target Gate free (not allocated)
// global state for targets :  TG  TL0  TL1   TL2  ... TLn
//                              A   N    N     P2       N
//                              N   N    N     P1       N                       
//                              I   I    I     I        I                                                                
// TLi not allocated, in case of preempt and last flit, needs to test value of palloc
// if palloc=1      => next state : IDLE (TG not allocated)
// else (palloc2=2) => next state : NALLOC (TG still Allocated)
{    
	switch( r_ring_cmd_fsm ) 
	{
		case CMD_IDLE:
                {
			bool brdcst = (m_cmd_fifo.read() & 0x1 == 0x1);

			p_ring_out.cmd_grant     = !m_cmd_fifo.rok() && p_ring_in.cmd_grant;

                        if (m_cmd_fifo.rok() && tga && !brdcst) 
                        {
                        	p_ring_out.cmd_preempt = 1;  
                        	p_ring_out.cmd_palloc  = 1;  
                        	p_ring_out.cmd_header  = 1; 
                                p_ring_out.cmd_w       = 1;
		                p_ring_out.cmd_data    = m_cmd_fifo.read();
                        }
                        else
                        {
                       		p_ring_out.cmd_preempt = 0;  
                        	p_ring_out.cmd_palloc  = 0;  
                        	p_ring_out.cmd_header  = 0; 
        		        p_ring_out.cmd_w       = p_ring_in.cmd_w;
	        	        p_ring_out.cmd_data    = p_ring_in.cmd_data;
                        }

                }
		break;
	
		case OWNER:   
		{     
                        bool eop = ((int) ((m_cmd_fifo.read() >> (ring_cmd_data_size - 1) ) & 0x1) == 1);
                        bool brdcst = (m_cmd_fifo.read() & 0x1 == 0x1);     
 
			p_ring_out.cmd_grant = (!m_cmd_fifo.rok() || (eop && p_ring_in.cmd_r)) ;  

                        p_ring_out.cmd_preempt = 0;
                        p_ring_out.cmd_header  = 0;
                        p_ring_out.cmd_palloc  = 0;
               	        p_ring_out.cmd_w       = m_cmd_fifo.rok() && !brdcst;
		        p_ring_out.cmd_data    = m_cmd_fifo.read();
		}
		break;

	       	case BDC_FIRST:
			if(r_preempt)
			{
                        	p_ring_out.cmd_header  = 1;
                        	p_ring_out.cmd_palloc  = 1+(tga ? 1:0);
                        	p_ring_out.cmd_preempt = 1;
				p_ring_out.cmd_grant   = p_ring_in.cmd_grant;
			}
			else
			{
                       		p_ring_out.cmd_header  = 0;
                        	p_ring_out.cmd_palloc  = 0;
                        	p_ring_out.cmd_preempt = 0;
				p_ring_out.cmd_grant   = false;
			}

			p_ring_out.cmd_w     = true;
			p_ring_out.cmd_data  = r_brdcst_save | ((((uint64_t) r_cpt_tgt.read()) & 0xF) << 0x1);    
                break;
		
		case BDC_SECOND:

			if (r_preempt)
			{
				p_ring_out.cmd_grant  = p_ring_in.cmd_grant;
				p_ring_out.cmd_palloc = 1+(tga ? 1:0);
				
				p_ring_out.cmd_preempt = m_cmd_fifo.rok(); //&& tga; 
				p_ring_out.cmd_header  = 0;
				
				if ( m_cmd_fifo.rok() )
				{
				        p_ring_out.cmd_w     = 1; //m_cmd_fifo.rok();
				        p_ring_out.cmd_data  = m_cmd_fifo.read();
				}
				else
				{
					// if init local has finished, tga = 0
					// init gate remains the only initiator, then w = 0
				        p_ring_out.cmd_w     = p_ring_in.cmd_w && tga; 
				        p_ring_out.cmd_data  = p_ring_in.cmd_data;
				}
			}
			else
			{
                		// si on est dans cet etat, c'est qu'on n'utilise pas le cmd_preempt mais le gnt.
                		p_ring_out.cmd_preempt= 0;
                		p_ring_out.cmd_header = 0;
                		p_ring_out.cmd_palloc = 0;
        			p_ring_out.cmd_w      = m_cmd_fifo.rok();
	        		p_ring_out.cmd_data   = m_cmd_fifo.read();
				p_ring_out.cmd_grant  = m_cmd_fifo.rok() && p_ring_in.cmd_r && (r_cpt_tgt.read() == m_nb_target - 1);
			}
                break;
	
		case SENDING: 
                { 

			if (r_preempt)
			{
				p_ring_out.cmd_grant  = p_ring_in.cmd_grant;
				p_ring_out.cmd_palloc = 1+(tga ? 1:0);
				
				p_ring_out.cmd_preempt = m_cmd_fifo.rok(); //&& tga; 
				p_ring_out.cmd_header  = 0;
				
				if ( m_cmd_fifo.rok() )
				{
				        p_ring_out.cmd_w     = 1; //m_cmd_fifo.rok();
				        p_ring_out.cmd_data  = m_cmd_fifo.read();
				}
				else
				{
					// if init local has finished, tga = 0
					// init gate remains the only initiator, then w = 0
				        p_ring_out.cmd_w     = p_ring_in.cmd_w && tga; 
				        p_ring_out.cmd_data  = p_ring_in.cmd_data;
				}
			}
			else
			{
				bool eop = ((int) (m_cmd_fifo.read() >> (ring_cmd_data_size - 1) ) & 0x1) == 1;
                		// si on est dans cet etat, c'est qu'on n'utilise pas le cmd_preempt mais le gnt.
                		p_ring_out.cmd_preempt= 0;
                		p_ring_out.cmd_header = 0;
                		p_ring_out.cmd_palloc = 0;
        			p_ring_out.cmd_w      = m_cmd_fifo.rok();
	        		p_ring_out.cmd_data   = m_cmd_fifo.read();
				p_ring_out.cmd_grant  = m_cmd_fifo.rok() && p_ring_in.cmd_r && eop;
			}
                }
                break;
	
	} // end switch

	p_ring_out.cmd_r       = p_ring_in.cmd_r;

	p_ring_out.rsp_w       = p_ring_in.rsp_w;
	p_ring_out.rsp_data    = p_ring_in.rsp_data;

	p_ring_out.rsp_grant   = p_ring_in.rsp_grant;

	p_ring_out.rsp_palloc  = p_ring_in.rsp_palloc;
        p_ring_out.rsp_preempt = p_ring_in.rsp_preempt;
        p_ring_out.rsp_header  = p_ring_in.rsp_header;


	switch( r_ring_rsp_fsm ) 
	{
		case RSP_IDLE:	
		{
			uint32_t  rsrcid  = (uint32_t)  ((sc_dt::sc_uint<vci_param::S>) ((p_ring_in.rsp_data >> (ring_rsp_data_size-m_x_width-m_y_width - 1)) << (vci_param::S-m_x_width-m_y_width)));
			bool islocal      = !m_lt[rsrcid];

                        iga = false;

			if(p_ring_in.rsp_w && islocal) 
				p_ring_out.rsp_r = m_rsp_fifo.wok();
                        else
				p_ring_out.rsp_r = p_ring_in.rsp_r;
		}
		break;
	
		case ALLOC:
                        iga = true;

			if (!p_ring_in.rsp_preempt)
                                p_ring_out.rsp_r =  m_rsp_fifo.wok(); 	
                        else
        			p_ring_out.rsp_r = p_ring_in.rsp_r;
		break;
	
		case NALLOC:
                        iga = false;
			p_ring_out.rsp_r = p_ring_in.rsp_r;
		break;    
	} // end switch


} // end update_ring_signals

};

}} // end namespace

#endif // DSPIN_HALF_GATEWAY_INITIATOR_FAST_H

