/**********************************************************************
 * \file	io.h
 * \date	5 September 2012
 * \author	Cesar Fuguet / Alain Greiner
 *
 * Utility functions to write or read memory mapped hardware registers
 *********************************************************************/

#ifndef IO_H
#define IO_H

#include <defs.h>

/**********************************************************************
 * Read an 32 bits memory mapped hardware register
 * with physical address extension to access cluster_io
 *********************************************************************/
static inline unsigned int ioread32(void * addr)
{
    unsigned int value;
    unsigned int ext = CLUSTER_IO;

    asm volatile(
        "mtc2  %2,    $24            \n"  /* PADDR_EXT <= cluster_io */
        "lw    %0,    0(%1)          \n"  /* value <= *(ext\addr)    */
        "mtc2  $0,    $24            \n"  /* PADDR_EXT <= 0          */
        : "=r" (value)
        : "r" (addr), "r" (ext) );

    return value;
}

/**********************************************************************
 * Read an 16 bits memory mapped hardware register
 *********************************************************************/
static inline unsigned short ioread16(void * addr)
{
	return *(volatile unsigned short *) addr;
}

/**********************************************************************
 * Read an 8 bits memory mapped hardware register
 *********************************************************************/
static inline unsigned char ioread8(void * addr)
{
	return *(volatile unsigned char *) addr;
}

/**********************************************************************
 * Write an 32 bits memory mapped hardware register
 * with physical address extension to access cluster_io
 *********************************************************************/
static inline void iowrite32(void * addr, unsigned int value)
{
    unsigned int ext = CLUSTER_IO;

    asm volatile(
        "mtc2  %2,    $24            \n"  /* PADDR_EXT <= cluster_io */
        "sw    %0,    0(%1)          \n"  /* *(ext\addr) <= value    */
        "mtc2  $0,    $24            \n"  /* PADDR_EXT <= 0          */
	    "sync                        \n"  /* sync barrier            */
        :
        : "r" (value), "r" (addr), "r" (ext) );
}

/**********************************************************************
 * Write an 16 bits memory mapped hardware register
 *********************************************************************/
static inline void iowrite16(void * addr, unsigned short value)
{
	*(volatile unsigned short *) addr = value;
	asm volatile("sync" ::: "memory");
}

/**********************************************************************
 * Write an 8 bits memory mapped hardware register
 *********************************************************************/
static inline void iowrite8(void * addr, unsigned char value)
{
	*(volatile unsigned char *) addr = value;
	asm volatile("sync" ::: "memory");
}

#endif
