/**CFile***********************************************************************

  FileName    [bmcCmd.c]

  PackageName [bmc]

  Synopsis    [Command interface for bounded model checking (bmc)]

  Author      [Mohammad Awedh]

  Copyright   [This file was created at the University of Colorado at
  Boulder.  The University of Colorado at Boulder makes no warranty
  about the suitability of this software for any purpose.  It is
  presented on an AS IS basis.]
******************************************************************************/

#include "bmcInt.h"

static char rcsid[] UNUSED = "$Id: bmcCmd.c,v 1.83 2010/04/09 23:50:57 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
static jmp_buf timeOutEnv;
static int bmcTimeOut;		/* timeout value */
static long alarmLapTime;	/* starting CPU time for timeout */

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/*static*/ BmcOption_t * ParseBmcOptions(int argc, char **argv);
static int CommandBmc(Hrc_Manager_t ** hmgr, int argc, char ** argv);
static int CommandWriteCnf(Hrc_Manager_t ** hmgr, int argc, char ** argv);
static void TimeOutHandle(void);
static void DispatchBmcCommand(Ntk_Network_t *network, Ctlsp_Formula_t *ltlFormula, array_t *constraintArray, BmcOption_t *options);
static int CommandCnfSat(Hrc_Manager_t ** hmgr, int argc, char ** argv);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Initializes the BMC package.]

  SideEffects []

  SeeAlso     [Bmc_End]

******************************************************************************/
void
Bmc_Init(void)
{
  Cmd_CommandAdd("bounded_model_check", CommandBmc, /* doesn't change network */  0);
  
  Cmd_CommandAdd("cnf_sat", CommandCnfSat, /* doesn't change network */  0); 
  Cmd_CommandAdd("write_cnf", CommandWriteCnf, /* doesn't change network */  0); 
}

/**Function********************************************************************

  Synopsis    [Ends BMC package.]

  SideEffects []

  SeeAlso     [Bmc_Init]

******************************************************************************/
void
Bmc_End(void)
{
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis [Parse the user input for command "bmc".]

  Description []

  SideEffects []

******************************************************************************/
/*static*/ BmcOption_t *
ParseBmcOptions(
  int argc,
  char **argv)
{
  BmcOption_t  *options = BmcOptionAlloc();
  char *ltlFileName     = NIL(char);
  unsigned int i;
  int          c;
  
  if (!options){
    return NIL(BmcOption_t);
  }
  
  options->dbgOut = 0;
  /*
   * Parse command line options.
   */
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "E:C:S:F:O::I:hiv:m:k:s:o:t:d:r:e")) != EOF) {
    switch(c) {
    case 'F':
      options->fairFile = Cmd_FileOpen(util_optarg, "r", NIL(char *), 0);
      if (options->fairFile == NIL(FILE)) {
	(void) fprintf(vis_stdout,
		       "** bmc error: Cannot open the file %s\n",
		       util_optarg);
	BmcOptionFree(options);
	return NIL(BmcOption_t);
      }
      break;
    case 'O':
      options->dbgOut = Cmd_FileOpen(util_optarg, "w", NIL(char *), 0);
      if(options->dbgOut == NIL(FILE)) {
        (void) fprintf(vis_stdout, "**bmc error: Cannot open %s for debug information.\n", util_optarg);
         BmcOptionFree(options);
         return NIL(BmcOption_t);
      } 
      break;
    case 'm':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->minK = atoi(util_optarg);
      break;
    case 'k':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->maxK = atoi(util_optarg);
      break;
    case 's':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->step = atoi(util_optarg);
      break;
    case 'r':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->inductiveStep = atoi(util_optarg);
      break;
    case 'e':
      options->earlyTermination = TRUE;
      break;	
    case 'd':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->dbgLevel = atoi(util_optarg);
      break;	
    case 't' :
      options->timeOutPeriod = atoi(util_optarg);
	break;
    case 'i':
      options->printInputs = TRUE;
      break;	
    case 'h':
      goto usage;
    case 'v':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->verbosityLevel = (Bmc_VerbosityLevel) atoi(util_optarg);
      break;
    case 'o':
      options->cnfFileName = util_strsav(util_optarg);
      break;	
    case 'I' :
      options->incremental = atoi(util_optarg);
      break;
    case 'E':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->encoding = atoi(util_optarg);
      if((options->encoding < 0) || (options->encoding > 2)){
	goto usage;
      }
      break;
    case 'S':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->satSolver = atoi(util_optarg);
      if((options->satSolver < 0) || (options->satSolver > 1)){
	goto usage;
      }
      break;
    case 'C':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      options->clauses = atoi(util_optarg);
      if((options->clauses < 0) || (options->clauses > 2)){
	goto usage;
      }
      break;
    default:
      goto usage;
    }
  }
  if (options->minK > options->maxK){
    (void) fprintf(vis_stderr, "** bmc error: value for -m option must not be greater than vlaue for -k option\n");    
    goto usage;
  }
  /*
   * Open the ltl file.
   */
  if (argc - util_optind == 0) {
    (void) fprintf(vis_stderr, "** bmc error: file containing ltl formulae not provided\n");
    goto usage;
  }
  else if (argc - util_optind > 1) {
    (void) fprintf(vis_stderr, "** bmc error: too many arguments\n");
    goto usage;
  }
  ltlFileName = util_strsav(argv[util_optind]);
  
  /* Read LTL Formulae */
  if (!ltlFileName) 
      goto usage;

  options->ltlFile = Cmd_FileOpen(ltlFileName, "r", NIL(char *), 0);
  if (options->ltlFile == NIL(FILE)) {
    (void) fprintf(vis_stdout,"** bmc error: Cannot open the file %s\n", ltlFileName);
    FREE(ltlFileName);
    BmcOptionFree(options);
    return NIL(BmcOption_t);
  }
  FREE(ltlFileName);
  /* create SAT Solver input file */
  if (options->cnfFileName == NIL(char)) {
    options->satInFile = BmcCreateTmpFile(); 
    if (options->satInFile == NIL(char)){
      BmcOptionFree(options);
      return NIL(BmcOption_t);
    }
  } else {
    options->satInFile = options->cnfFileName;
  }
  /* create SAT Solver output file */
  options->satOutFile = BmcCreateTmpFile();
  if (options->satOutFile == NIL(char)){
    BmcOptionFree(options);
    return NIL(BmcOption_t);
  }
  return options;
  
 usage:
  (void) fprintf(vis_stderr, "usage: bmc [-h][-m minimum_length][-k maximum_length][-s step][-r inductive_step][-e][-F <fairness_file>][-d debug_level][-E <encoding>][-C <clauses>][-S <sat_solver>][-O <debug_file>][-I <level>][-v verbosity_level][-t period][-o <cnf_file>] <ltl_file>\n");
  (void) fprintf(vis_stderr, "   -h \tprint the command usage\n");
  (void) fprintf(vis_stderr, "   -m \tminimum length of counterexample to be checked (default is 0)\n");  
  (void) fprintf(vis_stderr, "   -k \tmaximum length of counterexample to be checked (default is 1)\n");
  (void) fprintf(vis_stderr, "   -s \tincrementing value of counterexample length (default is 1)\n");
  (void) fprintf(vis_stderr, "   -r \tUse check for termination at each inductive_step to check if the property passes (default is 0).\n");
  (void) fprintf(vis_stderr, "   -e \tUse early termination to check if the property passes. Valid only with -r \n"); 
  (void) fprintf(vis_stderr, "   -F <fairness_file>\n");
  (void) fprintf(vis_stderr, "   -d <debug_level>\n");
  (void) fprintf(vis_stderr, "       debug_level = 0 => No debugging performed (Default)\n");
  (void) fprintf(vis_stderr, "       debug_level = 1 => Debugging with minimal output: generate counter-example if the LTL formula fails.\n");
  (void) fprintf(vis_stderr, "       debug_level = 2 => Debugging with full output in Aiger format: generate counter-example if the LTL formula fails.\n");
  (void) fprintf(vis_stderr, "   -O <debug_file>\n");
  (void) fprintf(vis_stderr, "   -i \tPrint input values in debug traces.\n");
  (void) fprintf(vis_stderr, "   -v <verbosity_level>\n");
  (void) fprintf(vis_stderr, "       verbosity_level = 0 => no feedback (Default)\n");
  (void) fprintf(vis_stderr, "       verbosity_level = 1 => code status\n");
  (void) fprintf(vis_stderr, "       verbosity_level = 2 => code status and CPU usage profile\n");
  (void) fprintf(vis_stderr, "   -t <period> time out period\n");
  (void) fprintf(vis_stderr, "   -E <encoding>\n");
  (void) fprintf(vis_stderr, "       encoding = 0 => Original BMC encoding (Default)\n");
  (void) fprintf(vis_stderr, "       encoding = 1 => SNF encoding\n");
  (void) fprintf(vis_stderr, "       encoding = 2 => Fixpoint encoding\n");
  /*
  (void) fprintf(vis_stderr, "       encoding = 3 => FNF encoding\n");
  */
  (void) fprintf(vis_stderr, "   -S <sat_solver>\n");
  (void) fprintf(vis_stderr, "       sat_solver = 0 => CirCUs (Default)\n");
  (void) fprintf(vis_stderr, "       sat_solver = 1 => cusp\n");
  (void) fprintf(vis_stderr, "   -C <clauses>\n");
  (void) fprintf(vis_stderr, "       clauses = 0 => Apply CirCUs on circuit (Default)\n");
  (void) fprintf(vis_stderr, "       clauses = 1 => Apply SAT solver on CNF (new)\n");
  (void) fprintf(vis_stderr, "       clauses = 2 => Apply SAT solver on CNF (old)\n");
  (void) fprintf(vis_stderr, "   -I <level>\n");
  (void) fprintf(vis_stderr, "       level = 1 => Trace Objective\n");
  (void) fprintf(vis_stderr, "       level = 2 => Distill\n");
  (void) fprintf(vis_stderr, "       level = 3 => Trace Objective + Distill\n");
  (void) fprintf(vis_stderr, "   -o <cnf_file> contains CNF of the counterexample\n");
  (void) fprintf(vis_stderr, "   <ltl_file> The input file containing LTL formulae to be checked.\n");

  BmcOptionFree(options);
  return NIL(BmcOption_t);
}

/**Function********************************************************************

  Synopsis [Alloc Memory for BmcOption_t.]

  SideEffects []

  SeeAlso []
******************************************************************************/
BmcOption_t *
BmcOptionAlloc(void)
{
  BmcOption_t *result = ALLOC(BmcOption_t, 1);
  
  if (!result){
    return result;
  }
  result->ltlFile = NIL(FILE);
  result->fairFile = NIL(FILE);
  result->cnfFileName = NIL(char);
  result->minK = 0;
  result->maxK = 1;
  result->step = 1;
  result->timeOutPeriod  = 0;
  result->startTime = 0;
  result->verbosityLevel = BmcVerbosityNone_c;
  result->dbgLevel       = 0;
  result->inductiveStep  = 0;
  result->printInputs    = FALSE;
  result->satSolverError = FALSE;
  result->satInFile  = NIL(char);
  result->satOutFile = NIL(char);
  result->incremental = 0;
  result->earlyTermination = 0;
  result->satSolver = CirCUs;  /* default is 0 */
  result->clauses   = 0;  /* default is 0 */
  result->encoding  = 0;  /* default is 0 */
  return result;
}

/**Function********************************************************************

  Synopsis [Free BmcOption_t, and close files.]

  SideEffects []

  SeeAlso []
******************************************************************************/
void
BmcOptionFree(
  BmcOption_t *result)
{
  if (result->ltlFile != NIL(FILE)){
    fclose(result->ltlFile);
  }
  if (result->cnfFileName != NIL(char)){
    FREE(result->cnfFileName);
  } else if (result->satInFile != NIL(char)) {
    unlink(result->satInFile);
    FREE(result->satInFile);
  }
  if (result->fairFile != NIL(FILE)){
    fclose(result->fairFile);
  }
  if (result->satOutFile != NIL(char)) {
    unlink(result->satOutFile);
    FREE(result->satOutFile);
  }
  FREE(result);
}

/**Function********************************************************************

  Synopsis    [Create a temporary file]

  Description []

  SideEffects []

******************************************************************************/
char *
BmcCreateTmpFile(void)
{
#if HAVE_MKSTEMP && HAVE_CLOSE  
  char   cnfFileName[] = "/tmp/vis.XXXXXX";
  int    fd;
#else
  char   *cnfFileName;
  char   buffer[512];
#endif

#if HAVE_MKSTEMP && HAVE_CLOSE    
  fd = mkstemp(cnfFileName);
  if (fd == -1){
#else
  cnfFileName = util_strsav(tmpnam(buffer));
  if (cnfFileName == NIL(char)){
#endif       
    (void)fprintf(vis_stderr,"** bmc error: Can not create temporary file. ");
    (void)fprintf(vis_stderr,"Clean up /tmp and try again.\n");
    return NIL(char);
  }
#if HAVE_MKSTEMP && HAVE_CLOSE
  close(fd);
#endif   
  return util_strsav(cnfFileName);
} /* createTmpFile() */

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Implements the bounded_model_check command]

  CommandName [bounded_model_check]

  CommandSynopsis [Performs a SAT-based LTL model checking on a flattened network]

  CommandArguments [\[-h\] \[-v &lt;verbosity_level&gt;\]
                    \[-m &lt;minimum_length&gt;\] \[-k &lt;maximum_length&gt;\] \[-s &lt;step_value&gt;\]
		    \[-r &lt;termination_check_step&gt;\] \[-e\]
		    \[-F &lt;fairness_file&gt;\]
		    \[-S &lt;sat_solver&gt;\] \[-E &lt;encoding&gt;\]
		    \[-C &lt;clause_type&gt;\] \[-I &lt;inc_level&gt;\]
		    \[-d &lt;dbg_level&gt;\] \[-t &lt;timeOutPeriod&gt;\]
		    \[-o &lt;cnf_file&gt;\] \[-i\]
		    \[-O &lt;debug_file&gt;\] &lt;ltl_file&gt; ]
		    
  CommandDescription [Performs a SAT-based LTL bounded model checking (BMC) on
  a flattened network. This command looks for a counterexample of length <code>
  k </code> ( minimum_length &le; <code> k </code> &le; maximum_length). If
  <code> -r </code> is specified, this command performs check for termination
  after each <code> termination_check_step </code>.  If no counterexample is
  found, this command increases <code> k </code> by step_value (specifies by
  the <code> -s </code> option) until a counterexample is found, the search
  becomes intractable (timed out), or <code> k </code> reaches a bound
  (determine by the check for termination), and then we conclude that the LTL
  property passes.

  This command implements the basic encoding of Bounded Model Checking (BMC) as
  descibed in the paper "Symbolic Model Checking without BDDs".  However, the
  <code> -E </code> option can be used to select other encoding scheme.  We
  also applied some of the improvements in the BMC encoding described in the
  paper "Improving the Encoding of LTL Model Checking into SAT".
  
  To prove that an LTL property passes, we implement the termination methods
  that are descirebed in the papers "Checking Safety Properties Using Induction
  and a SAT-Solver" and "Proving More Properties with Bounded Model Checking".
  
  Before calling this command, the user should have initialized the design by
  calling the command <A HREF="flatten_hierarchyCmd.html"> <code>
  flatten_hierarchy</code></A>. If the user uses the -r option and the LTL
  property is a general LTL property, the command <A
  HREF="build_partition_maigsCmd.html"> <code> build_partition_maigs</code></A>
  must be called.  The aig graph must be built by calling the command <A
  HREF="build_partition_mddsCmd.html"> <code>build_partition_mdds</code></A>
  <p>
  
  Command options:
  <p>
  <dl>
  <dt> -h
  <dd> Print the command usage.
  
  <p>
  <dt>-m <code>&lt;minimum_length&gt;</code>
  <dd>Minimum length of counterexample to be checked (default is 0). 

  <dt>-k <code>&lt;maximum_length&gt;</code>
  <dd>Maximum length of counterexample to be checked (default is 1).

  <dt>-s <code>&lt;step_value&gt;</code>
  <dd>Incrementing value of counterexample length (default is 1).

  <dt>-r <code>&lt;termination_check_step&gt;</code>
  <dd>Check for termination for each termination_check_step (default is 0). 0 means don't check for termination.

  <dt>-e
  <dd> Activate early termination check.  Check if there is no simple path starts from an initial state.
       Valid only for general LTL and safety properties. Must be used with -r option.
  
  <dt>-S <code>&lt;sat_solver&gt;</code>
  <dd> Specify SAT solver
  <br><code>sat_solver</code>  must be one of the following:<p>
  <code>0</code>: CirCUs (Default) <p>
  <code>1</code>: zChaff <p>

  <dt>-E <code>&lt;encoding&gt;</code>
  <dd> Specify encoding scheme
  <br><code>encoding</code>  must be one of the following:<p>
  <code>0</code>: The original BMC encoding (Default) <p>
  <code>1</code>: SNF encoding <p>
  <code>2</code>: Fixpoint encoding <p>

  <dt>-C <code>&lt;clause_type&gt;</code>
  <dd> Specify clause type
  <br><code>encoding</code>  must be one of the following:<p>
  <code>0</code>: Apply CirCUs SAT solver on circuit (Default) <p>
  <code>1</code>: Apply SAT solver on CNF generated form circuit<p>
  <code>2</code>: Apply SAT solver on CNF <p>

  <dt>-I <code>&lt;inc_level&gt;</code>
  <dd> Specify increment sat solver type
  <br><code>encoding</code>  must be one of the following:<p>
  <code>1</code>: Trace Objective (Default) <p>
  <code>2</code>: Distill <p>
  <code>3</code>: Trace Objective + Distill <p>

  <dt>-F <code>&lt;fairness_file&gt;</code>
  <dd> Read fairness constraints from <code>&lt;fairness_file&gt;</code>.  Each
  formula in the file is a condition that must hold infinitely often on a fair
  path.
  
  <dt>-o <code>&lt;cnf_file&gt;</code>
  Save CNF formula in <code>&lt;cnf_file&gt;</code>

  <dt>-O <code>&lt;debug_file&gt;</code>
  Save counterexample to <code>&lt;debug_file&gt;</code>
  
  <dt> -t  <code>&lt;timeOutPeriod&gt;</code>
  <dd> Specify the time out period (in seconds) after which the command aborts. By default this option is set to infinity.
  <p>
  <dt> -v  <code>&lt;verbosity_level&gt;</code>
  <dd> Specify verbosity level. This sets the amount of feedback  on CPU usage and code status.

  <br><code>verbosity_level</code>  must be one of the following:<p>
  
  <code>0</code>: No feedback provided. This is the default.<p>

  <code>1</code>: Feedback on code location.<p>

  <code>2</code>: Feedback on code location and CPU usage.<p>

  <dt> -d <code> &lt;dbg_level&gt; </code> <dd> Specify the amount of
  debugging performed when the BMC models check the LTL formula. 
  <p> <dd> <code> dbg_level</code> must be one
  of the following:

  <p><code>0</code>: No debugging performed.
  <code>dbg_level</code>=<code>0</code> is the default.

  <p><code>1</code>: Debugging with minimal output: generate counter-example
  if the LTL formula fails and the counterexample length.

  <p>
  <dt> -i
  <dd> Print input values causing transitions between states during debugging.
  <p>
  
  <dt> <code> &lt;ltl_file&gt; </code>
  <dd> File containing LTL formulae to be checked.
  
  </dl>
  ]

  SideEffects []

  SeeAlso     [Ntk_NetworkAddApplInfo]

******************************************************************************/
static int
CommandBmc(
  Hrc_Manager_t ** hmgr,
  int              argc,
  char          ** argv)
{
  Ntk_Network_t     *network;
  BmcOption_t       *options;
  int               i;
  array_t           *formulaArray;
  array_t           *LTLformulaArray;
  bAig_Manager_t    *manager;
  array_t           *constraintArray = NIL(array_t);

  /*
   * Parse command line options.
   */
  if ((options = ParseBmcOptions(argc, argv)) == NIL(BmcOption_t)) {
      return 1;
  }
  /*
   * Read the network
   */
  network = Ntk_HrcManagerReadCurrentNetwork(*hmgr);
  if (network == NIL(Ntk_Network_t)) {
    (void) fprintf(vis_stdout, "** bmc error: No network\n");
    BmcOptionFree(options);
    return 1;
  }
  manager = Ntk_NetworkReadMAigManager(network);
  if (manager == NIL(mAig_Manager_t)) {
    (void) fprintf(vis_stdout, "** bmc error: run build_partition_maigs command first\n");
    BmcOptionFree(options);
    return 1;
  }
  /*
    We need the bdd when building the transition relation of the automaton
  */
  if(options->inductiveStep !=0){
    Fsm_Fsm_t *designFsm = NIL(Fsm_Fsm_t);
 
    designFsm = Fsm_HrcManagerReadCurrentFsm(*hmgr);
    if (designFsm == NIL(Fsm_Fsm_t)) {
      return 1;
    }
  }
  /* time out */
  if (options->timeOutPeriod > 0) {
    /* Set the static variables used by the signal handler. */
    bmcTimeOut = options->timeOutPeriod;
    alarmLapTime = options->startTime = util_cpu_ctime();
    (void) signal(SIGALRM, (void(*)(int))TimeOutHandle);
    (void) alarm(options->timeOutPeriod);
    if (setjmp(timeOutEnv) > 0) {
      (void) fprintf(vis_stdout,
		     "\n# BMC: timeout occurred after %d seconds.\n",
		     options->timeOutPeriod);
      (void) fprintf(vis_stdout, "# BMC: data may be corrupted.\n");
      BmcOptionFree(options);
      alarm(0);
      return 1;
    }
  }
  formulaArray  = Ctlsp_FileParseFormulaArray(options->ltlFile);
  if (formulaArray == NIL(array_t)) {
    (void) fprintf(vis_stderr,
		   "** bmc error: error in parsing CTL* Fromula from file\n");
    BmcOptionFree(options);
    return 1;
  }
  if (array_n(formulaArray) == 0) {
    (void) fprintf(vis_stderr, "** bmc error: No formula in file\n");
    BmcOptionFree(options);
    Ctlsp_FormulaArrayFree(formulaArray);
    return 1;
  }
  LTLformulaArray = Ctlsp_FormulaArrayConvertToLTL(formulaArray);
  Ctlsp_FormulaArrayFree(formulaArray);
  if (LTLformulaArray ==  NIL(array_t)){
    (void) fprintf(vis_stdout, "** bmc error: Invalid LTL formula\n");
    BmcOptionFree(options);
    return 1;
  }
  if (options->fairFile != NIL(FILE)) {
    constraintArray = BmcReadFairnessConstraints(options->fairFile);
    if(constraintArray == NIL(array_t)){
      Ctlsp_FormulaArrayFree(LTLformulaArray);
      BmcOptionFree(options);
      return 1;
    }
    if(!Ctlsp_LtlFormulaArrayIsPropositional(constraintArray)){
      Ctlsp_FormulaArrayAddLtlFairnessConstraints(LTLformulaArray,
						  constraintArray);
      Ctlsp_FormulaArrayFree(constraintArray);
      constraintArray = NIL(array_t);
    }
  }
  /*
    Call a proper BMC function based on LTL formula.
  */
  for (i = 0; i < array_n(LTLformulaArray); i++) { 
    Ctlsp_Formula_t *ltlFormula     = array_fetch(Ctlsp_Formula_t *,
						  LTLformulaArray, i);
    
    DispatchBmcCommand(network, ltlFormula, constraintArray, options);
  }
  /*
    Free used memeory
  */
  if (constraintArray != NIL(array_t)){
    Ctlsp_FormulaArrayFree(constraintArray);
  }
  Ctlsp_FormulaArrayFree(LTLformulaArray);
  BmcOptionFree(options);
  fflush(vis_stdout);
  fflush(vis_stderr);
  alarm(0);
  return 0;

}/* CommandBmc() */

/**Function********************************************************************

  Synopsis    [Handle function for timeout.]

  Description [This function is called when the time out occurs.
  Since alarm is set with an elapsed time, this function checks if the
  CPU time corresponds to the timeout of the command.  If not, it
  reprograms the alarm to fire again later.]

  SideEffects []

******************************************************************************/
static void
TimeOutHandle(void)
{
  int seconds = (int) ((util_cpu_ctime() - alarmLapTime) / 1000);

  if (seconds < bmcTimeOut) {
    unsigned slack = (unsigned) (bmcTimeOut - seconds);
    (void) signal(SIGALRM, (void(*)(int)) TimeOutHandle);
    (void) alarm(slack);
  } else {
    longjmp(timeOutEnv, 1);
  }
} /* TimeOutHandle */


/**Function********************************************************************

  Synopsis    [Dispatch BMC]

  Description [Call a proper BMC routine.

  The -E for BMC encoding scheme
      -C clause generation 
      -S SAT solver   0  CirCUs
                      1 zChaff

  ]

  SideEffects []

******************************************************************************/
static void
DispatchBmcCommand(
  Ntk_Network_t   *network,
  Ctlsp_Formula_t *ltlFormula,
  array_t         *constraintArray,
  BmcOption_t     *options)
{
  Ctlsp_Formula_t *negLtlFormula = Ctlsp_LtllFormulaNegate(ltlFormula);
  st_table        *CoiTable      = st_init_table(st_ptrcmp, st_ptrhash);

  assert(ltlFormula != NIL(Ctlsp_Formula_t));
  assert(network != NIL(Ntk_Network_t));
 
  /*
    print out the given LTL formula
  */
  fprintf(vis_stdout, "Formula: ");
  Ctlsp_FormulaPrint(vis_stdout, ltlFormula);
  fprintf(vis_stdout, "\n");
  if (options->verbosityLevel >= BmcVerbosityMax_c){
    fprintf(vis_stdout, "Negated formula: ");
    Ctlsp_FormulaPrint(vis_stdout, negLtlFormula);
    fprintf(vis_stdout, "\n");
  }
  /*
    Compute the cone of influence of the LTL formula
  */
  BmcGetCoiForLtlFormula(network, negLtlFormula, CoiTable);

  if(options->clauses == 2){
    /*
      Generate clauses for each time frame.  This is the old way of generating
      clauses in BMC.
    */
    if (constraintArray != NIL(array_t)){
      BmcLtlVerifyGeneralLtl(network, negLtlFormula, CoiTable,
			     constraintArray, options);
    } else /*
	     If the LTL formula is propositional.
	   */
      if(Ctlsp_isPropositionalFormula(negLtlFormula)){
	BmcLtlVerifyProp(network, negLtlFormula, CoiTable, options);
      } else /*
	       If the LTL formula is of type G(p) (its negation is
	       of type F(q)), where p and q are propositional.
	     */
	if(Ctlsp_LtlFormulaIsFp(negLtlFormula)){
	  BmcLtlVerifyGp(network, negLtlFormula, CoiTable, options);
	} else /*
		 If the LTL formula is of type F(p) (its negation is
		 of type G(q)), where p and q are propositional.
	       */
	  if (Ctlsp_LtlFormulaIsGp(negLtlFormula)){
	    BmcLtlVerifyFp(network, negLtlFormula, CoiTable, options);
	  } else /*
		   If the depth of the LTL formula (the maximum level
		   of nesting temporal operators) = 1
		 */
	    /*
	    if (Ctlsp_LtlFormulaDepth(negLtlFormula) == 1){
	      BmcLtlVerifyUnitDepth(network, negLtlFormula, CoiTable, options);
	    } else
	    */
	      /* 
		 If the LTL formula is of type FG(p) (its negation is
		 of type GF(q)), where p and q are propositional.
	      */
	      if(Ctlsp_LtlFormulaIsGFp(negLtlFormula)){
		BmcLtlVerifyFGp(network, negLtlFormula, CoiTable, options);
	      } else
		/*
		  All other LTL formulae
		*/
		BmcLtlVerifyGeneralLtl(network, negLtlFormula, CoiTable,
				       NIL(array_t), options);
  } else {
    /*
     Build AIG for each time frame.
     */
    if (constraintArray != NIL(array_t)){
      if(options->encoding == 0){
       BmcCirCUsLtlVerifyGeneralLtl(network, negLtlFormula,
				    CoiTable,
				   constraintArray, options, 0);
      } else 
	if(options->encoding == 1){
	  BmcCirCUsLtlVerifyGeneralLtl(network, negLtlFormula,
					  CoiTable,
					  constraintArray, options, 1);
	} else 
	  if(options->encoding == 2){
	    BmcCirCUsLtlVerifyGeneralLtlFixPoint(network, negLtlFormula,
						 CoiTable,
						 constraintArray, options);
	  }
    } else
      /*
	If the LTL formula is propositional.
      */
      if(Ctlsp_isPropositionalFormula(negLtlFormula)){
	BmcCirCUsLtlVerifyProp(network, negLtlFormula, CoiTable, options);
      } else
	/*
	  If the LTL formula is of type G(p) (its negation is
	  of type F(q)), where p and q are propositional.
	*/
	if(Ctlsp_LtlFormulaIsFp(negLtlFormula)){
	  BmcCirCUsLtlVerifyGp(network, negLtlFormula, CoiTable, options);
	} else
	  /*
	    If the LTL formula is of type F(p) (its negation is
	    of type G(q)), where p and q are propositional.
	  */
	  if (Ctlsp_LtlFormulaIsGp(negLtlFormula)){
	    BmcCirCUsLtlVerifyFp(network, negLtlFormula, CoiTable, options);
	  } else
	    /*
	      If the depth of the LTL formula (the maximum level
	      of nesting temporal operators) = 1
	      
	      if (Ctlsp_LtlFormulaDepth(negLtlFormula) == 1){
	      BmcLtlVerifyUnitDepth(network, negLtlFormula, CoiTable, options);
	      } else
	       
	      If the LTL formula is of type FG(p) (its negation is
	      of type GF(q)), where p and q are propositional.
	    */
	    if(Ctlsp_LtlFormulaIsGFp(negLtlFormula)){
	      BmcCirCUsLtlVerifyFGp(network, negLtlFormula, CoiTable, options);
	    } else
	      if(Ctlsp_LtlFormulaTestIsSyntacticallyCoSafe(negLtlFormula) ||
	      	 (Ltl_AutomatonGetStrength(BmcAutLtlToAutomaton(network,ltlFormula)) == 0))
		{
		  BmcCirCUsLtlCheckSafety(network, negLtlFormula, options, CoiTable);
		}
	      else {
		/*
		  All other LTL formulae
		*/
		if(options->encoding == 0){
		  BmcCirCUsLtlVerifyGeneralLtl(network, negLtlFormula,
					       CoiTable,
					       NIL(array_t), options, 0);
		} else 
		  if(options->encoding == 1){
		    BmcCirCUsLtlVerifyGeneralLtl(network, negLtlFormula,
						    CoiTable,
						    NIL(array_t), options, 1);
		  } else 
		    if(options->encoding == 2){
		      BmcCirCUsLtlVerifyGeneralLtlFixPoint(network, negLtlFormula,
						 CoiTable,
							   NIL(array_t), options);
		    }
	      } 
  }
  
  st_free_table(CoiTable);
  Ctlsp_FormulaFree(negLtlFormula);
} /* DispatchBmcCommand() */


/**Function********************************************************************

  Synopsis    [cnf_sat]

  Description []
  
  CommandName [cnf_sat]

  CommandSynopsis [Perform SAT solving with CNF input]

  CommandArguments [\[-h\] \[-a &lt;assgined_filename&gt;\]
                    \[-f &lt;output_filename&gt;\] 
		    \[-t &lt;timeout&gt;\] 
		    \[-v &lt;verbose&gt;\] 
		    \[-b\] &lt;cnf_filename&gt; ]
		    

  CommandDescription [Perform SAT solving with CirCUs after reading CNF file
  <p>
  <dt> -b

  <dd> If the given CNF has small number of variables and clause then
  the BDD is built from the CNF clauses. If the monolithic BDD is built
  then we can conclude SAT or UNSAT, otherwise the normal SAT algorithm
  is invoked.

  <p>
  <p>
  <dt> -t  <code>&lt;timeOutPeriod&gt;</code>

  <dd> Specify the time out period (in seconds) after which the command
  aborts. By default this option is set to infinity.

  <p>

  <p>
  <dt> -f  <code>&lt;output_filename&gt;</code>
  <dd> Specify the output filename to save the satisfying assignments and 
  the statistics of SAT solving.
  <p>

  ]

  SideEffects []

  SeeAlso     []

******************************************************************************/

static int
CommandCnfSat(
  Hrc_Manager_t ** hmgr,
  int  argc,
  char ** argv)
{
  satOption_t *option;
  char *filename, c;
  char *forcedAssignFilename;
  int timeOutPeriod, i;

  option = sat_InitOption();
  timeOutPeriod = -1;
  forcedAssignFilename = 0;
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "a:bf:ht:v:c:")) != EOF) {
    switch(c) {
      case 'h':
	goto usage;
      case 'b' :
        option->BDDMonolithic = 1;
	break;
      case 'f' :
	option->outFilename = strdup(util_optarg);
	break;
      case 'a' :
	forcedAssignFilename = strdup(util_optarg);
	break;
      case 't' :
	timeOutPeriod = atoi(util_optarg);
	break;
      case 'v':
	for (i = 0; i < (int)(strlen(util_optarg)); i++) {
	  if (!isdigit((int)util_optarg[i])) {
	    goto usage;
	  }
	}
	option->verbose = (Bmc_VerbosityLevel) atoi(util_optarg);
	break;

	//Bing: for unsat core
      case 'c':
	option->coreGeneration = 1;
	option->unsatCoreFileName = strdup(util_optarg);
	option->minimizeConflictClause = 0;
	option->clauseDeletionHeuristic = 0;
	break;
      default:
	goto usage;
    }
  }

  if (argc - util_optind == 0) {
    (void) fprintf(vis_stderr, "** ERROR : file containing cnf file not provided\n");
    goto usage;
  }
  else if (argc - util_optind > 1) {
    (void) fprintf(vis_stderr, "** ERROR : too many arguments\n");
    goto usage;
  }
  filename = util_strsav(argv[util_optind]);

  if(forcedAssignFilename)
    option->forcedAssignArr = sat_ReadForcedAssignment(forcedAssignFilename);

  if (timeOutPeriod > 0) {
    bmcTimeOut = timeOutPeriod;
    alarmLapTime = util_cpu_ctime();
    (void) signal(SIGALRM, (void(*)(int))TimeOutHandle);
    (void) alarm(timeOutPeriod);
    if (setjmp(timeOutEnv) > 0) {
      (void) fprintf(vis_stderr,
		"** ERROR : timeout occurred after %d seconds.\n", timeOutPeriod);
      alarm(0);
      return 1;
    }
  }

  sat_CNFMain(option, filename);

  return 0;

  usage:
  (void) fprintf(vis_stderr, "usage: cnf_sat [-h] [-v verboseLevel] [-t timeout] <cnf_file>\n");
  (void) fprintf(vis_stderr, "   -h \tprint the command usage\n");
  (void) fprintf(vis_stderr, "   -a <filename> \tto make forced assignment\n");
  (void) fprintf(vis_stderr, "   -f <filename> \twrite log to the file\n");
  (void) fprintf(vis_stderr, "   -b \tuse BDD based method\n");
  (void) fprintf(vis_stderr, "   -v <verbosity_level>\n");
  (void) fprintf(vis_stderr, "   -t <period> time out period\n");
  (void) fprintf(vis_stderr, "   -c <filename> UNSAT core generation\n");
  (void) fprintf(vis_stderr, "   <cnf_file> CNF file to be checked.\n");
  return 1;
}


#if 0
/**Function********************************************************************

  Synopsis    [Implements the bdd_sat command.]

  Description [This command integrates BDD-based methods with SAT-based method
  to model safety property of the form AG(p). Where p is either a propositional
  formula or a path formula consists of only the temporal operator X.]
  
  CommandName [bdd_sat_bounded_model_check]

  CommandSynopsis [performs an LTL bounded model checking on a flattened network]

  CommandArguments [\[-h\] \[-v &lt;verbosity_level&gt;\]  \[-d &lt;dbg_level&gt;\] \[-i\] 
                    -m &lt;minimum_length&gt; -k &lt;maximum_length&gt; -s &lt;step_value&gt;
		    -o &lt;cnf_file&gt; &lt;ltl_file&gt; ]

		    -F &lt;fairness_file&gt; -o &lt;cnf_file&gt; &lt;ltl_file&gt; ]

  CommandDescription [Performs an LTL bounded model checking on a flattened
  network. This command looks for a counterexample of length &ge; minimum_length
  and &le; maximum_length. It increments the length by step_value. 
  
  Before calling this command, the user should have initialized the design
  by calling the command <A HREF="flatten_hierarchyCmd.html"> <code>
  flatten_hierarchy</code></A>, and
  then calling the command <A HREF="build_partition_maigsCmd.html"> <code>
  build_partition_maigs</code></A>.<p>
  The value of maximum length must be >= minimum length.<p>

  Command options:
  <p>

  <dl>

  <dt> -h
  <dd> Print the command usage.
  <p>

  <dt>-m <code>&lt;minimum_length&gt;</code>
  <dd>Minimum length of counterexample to be checked (default is 0). 

  <dt>-k <code>&lt;maximum_length&gt;</code>
  <dd>Maximum length of counterexample to be checked (default is 1).

  <dt>-s <code>&lt;step_value&gt;</code>
  <dd>Incrementing value of counterexample length (default is 1).

  <dt>-r <code>&lt;inductive_Step&gt;</code>
  <dd>Use inductive proof at each step_value to check if the property passes (default is 0). 0 means
  don't use the inductive proof. BMC will check using the indcution if the property passes if the current
  length of the counterexample is a multiple of inductive_Step.
  
  <dt>-F <code>&lt;fairness_file&gt;</code>
  <dd> Read fairness constraints from <code>&lt;fairness_file&gt;</code>.  Each
  formula in the file is a condition that must hold infinitely often on a fair
  path.

  <dt>-o <code>&lt;cnf_file&gt;</code>
  Save the CNF formula in <code>&lt;cnf_file&gt;</code>
  
  <dt> -t  <code>&lt;timeOutPeriod&gt;</code>
  <dd> Specify the time out period (in seconds) after which the command aborts. By default this option is set to infinity.
  <p>
  <dt> -v  <code>&lt;verbosity_level&gt;</code>
  <dd> Specify verbosity level. This sets the amount of feedback  on CPU usage and code status.

  <br><code>verbosity_level</code>  must be one of the following:<p>
  
  <code>0</code>: No feedback provided. This is the default.<p>

  <code>1</code>: Feedback on code location.<p>

  <code>2</code>: Feedback on code location and CPU usage.<p>

  <dt> -d <code> &lt;dbg_level&gt; </code> <dd> Specify the amount of
  debugging performed when the BMC models check the LTL formula. 
  <p> <dd> <code> dbg_level</code> must be one
  of the following:

  <p><code>0</code>: No debugging performed.
  <code>dbg_level</code>=<code>0</code> is the default.

  <p><code>1</code>: Debugging with minimal output: generate counter-example
  if the LTL formula fails and the counterexample length.

  <p>
  
  <dt> -h
  <dd> Print the command usage.
  <p>

  <dt> -i
  <dd> Print input values causing transitions between states during debugging.
  <p>

  <dt> -o <code>&lt;cnf_file&gt</code>;
  <dd> File containing CNF of the counterexample if exist. If specifies, the
  user can exam this file looking for information about the generated path.
  This file will be the input to the SAT solver.  The contents of this file
  is in dimacs format.
  
  <dt> <code> &lt;ltl_file&gt; </code>
  <dd> File containing LTL formulas to be model checked.
  
  </dl>
  ]

  SideEffects []

  SeeAlso     [Ntk_NetworkAddApplInfo]

******************************************************************************/ 
int
CommandBddSat(
  Hrc_Manager_t ** hmgr,
  int              argc,
  char          ** argv)
{
  Ntk_Network_t     *network;
  BmcOption_t       *options;
  array_t           *formulaArray;
  bAig_Manager_t    *mAigManager;
  Fsm_Fsm_t         *fsm;
  mdd_manager       *mddManager;
  int               startTime;


  /*
   * Parse command line options.
   */
  if ((options = ParseBmcOptions(argc, argv)) == NIL(BmcOption_t)) {
      return 1;
  }
  /* Read the network */
  network = Ntk_HrcManagerReadCurrentNetwork(*hmgr);
  if (network == NIL(Ntk_Network_t)) {
    (void) fprintf(vis_stdout, "** bdd_sat error: No network\n");
    BmcOptionFree(options);
    return 1;
  }
  /* read fsm */
  fsm = Fsm_NetworkReadOrCreateFsm(network);
  if (fsm == NIL(Fsm_Fsm_t)) {
    (void) fprintf(vis_stdout, "** bdd_sat error: Can't read or create fsm\n");
    BmcOptionFree(options);
    return 1;
  }
  mddManager = Fsm_FsmReadMddManager(fsm); /* mdd manager */
  
  mAigManager = Ntk_NetworkReadMAigManager(network);
  if (mAigManager == NIL(mAig_Manager_t)) {
    (void) fprintf(vis_stdout, "** bdd_sat error: run build_partition_maigs command first\n");
    BmcOptionFree(options);
    return 1;
  }
  formulaArray = Ctlsp_FileParseFormulaArray(options->ltlFile);
  if (formulaArray == NIL(array_t)) {
    (void) fprintf(vis_stderr,
		   "** bdd_sat error: Error in parsing CTL* Fromula from file\n");
    BmcOptionFree(options);
    return 1;
  }
  if (array_n(formulaArray) == 0) {
    (void) fprintf(vis_stderr, "** bdd_sat error: No formula in file\n");
    BmcOptionFree(options);
    array_free(formulaArray);
    return 1;
  }
  /* time out */
  if (options->timeOutPeriod > 0) {
    /* Set the static variables used by the signal handler. */
    bmcTimeOut = options->timeOutPeriod;
    alarmLapTime = options->startTime = util_cpu_ctime();
    (void) signal(SIGALRM, (void(*)(int))TimeOutHandle);
    (void) alarm(options->timeOutPeriod);
    if (setjmp(timeOutEnv) > 0) {
      (void) fprintf(vis_stdout,
		     "\n# bdd_sat: timeout occurred after %d seconds.\n",
		     options->timeOutPeriod);
      (void) fprintf(vis_stdout, "# bdd_sat: data may be corrupted.\n");
      BmcOptionFree(options);
      alarm(0);
      return 1;
    }
  }
  /* bdd and sat to model check the LTL property.*/
  BmcBddSat(network, formulaArray, options);

  Ctlsp_FormulaArrayFree(formulaArray);
  fflush(vis_stdout);
  fflush(vis_stderr);
  BmcOptionFree(options);
  alarm(0);
  return 0; /* normal exit */
}/* CommandBddSat() */
#endif

/**Function********************************************************************

  Synopsis    [write_cnf]

  Description []
  
  CommandName [write_cnf]

  CommandSynopsis [Generate the CNF form of a design]

  CommandArguments [\[-h\] \[-f &lt;ltl_filename&gt;\] 
		    \[-o &lt;out_filename;\] 
		    \[-k &lt;number of steps;\] 
		    

  CommandDescription [Generate the CNF form of the network and the ltl specification if mentionned.
  <p>
  <dt> -o  <code>&lt;output_filename&gt;</code>
  <dd> Specify the output filename to save the satisfying assignments and 
  the statistics of SAT solving.
  <p>
  <dt> -f  <code>&lt;ltl_filename&gt;</code>
  <dd> Specify the ltl filename that will be generate with.
  <p>
  <dt> -k  <code>&lt;value;</code>
  <dd> Specify the number of step unrolled (by default = 0)
  <p>

  ]

  SideEffects []

  SeeAlso     []

******************************************************************************/

int CommandWriteCnf(Hrc_Manager_t ** hmgr, int argc, char ** argv)
{
  Ntk_Network_t     *network;
  bAig_Manager_t    *manager;
  array_t           *formulaArray;
  array_t           *LTLformulaArray;
  char              * outFileName = NIL(char); 
  char              * ltlFileName = NIL(char); 
  FILE * ltlFile;
  FILE * outFile;
  int c,i;
  int maxK = 0;
  st_table          *allLatches;
  BmcCnfClauses_t   *cnfClauses = NIL(BmcCnfClauses_t);


  outFile = vis_stdout ; 
  // Parse option
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "f:o:k:h")) != EOF) {
    switch(c) {
      case 'h':
	goto usage;
      case 'o' :
	outFileName = strdup(util_optarg);
    outFile = Cmd_FileOpen(outFileName, "w", NIL(char *), 0); 
	break;
      case 'f' :
	ltlFileName = strdup(util_optarg);
    	break;
    case 'k':
      for (i = 0; i < strlen(util_optarg); i++) {
	if (!isdigit((int)util_optarg[i])) {
	  goto usage;
	}
      }
      maxK = atoi(util_optarg);
      break;

	break;
      default:
	goto usage;
    }
  }
  network = Ntk_HrcManagerReadCurrentNetwork(*hmgr);
  if (network == NIL(Ntk_Network_t)) {
    (void) fprintf(vis_stdout, "** bmc error: No network\n");
    return 1;
  }
  manager = Ntk_NetworkReadMAigManager(network);
  if (manager == NIL(mAig_Manager_t)) {
    (void) fprintf(vis_stdout, "** bmc error: run build_partition_maigs command first\n");
    return 1;
  }
   
    //    Create a clause database
     
    cnfClauses = BmcCnfClausesAlloc();
    // Gnerate clauses for an initialized path of length k
    // nodeToMvfAigTable Maps each node to its Multi-function And/Inv graph 
     st_table* nodeToMvfAigTable = (st_table *) Ntk_NetworkReadApplInfo(network, MVFAIG_NETWORK_APPL_KEY);
     assert(nodeToMvfAigTable != NIL(st_table));
      allLatches = st_init_table(st_ptrcmp, st_ptrhash);

     // Get all the latches of the network
	  lsGen           gen ;
	  Ntk_Node_t		*node;
	Ntk_NetworkForEachNode(network,gen, node){
		if (Ntk_NodeTestIsLatch(node)){
			st_insert(allLatches, (char *) node, (char *) 0);
		}
	}


    BmcCnfGenerateClausesForPath(network, 0, maxK, BMC_INITIAL_STATES,
				 cnfClauses, nodeToMvfAigTable, allLatches);

// If ltl file exists 
 if(ltlFileName != NIL(char))
  { 
    ltlFile =  Cmd_FileOpen(ltlFileName, "r", NIL(char *), 0);
  formulaArray  = Ctlsp_FileParseFormulaArray(ltlFile);
  if (formulaArray == NIL(array_t)) {
    (void) fprintf(vis_stderr,
		   "** bmc error: error in parsing ltl Fromula from file\n");
      return 1;
   }
    LTLformulaArray = Ctlsp_FormulaArrayConvertToLTL(formulaArray);
    for (i = 0; i < array_n(LTLformulaArray); i++) { 
    Ctlsp_Formula_t *ltlFormula     = array_fetch(Ctlsp_Formula_t *,
						  LTLformulaArray, i);
    BmcGenerateCnfForLtl(network, ltlFormula, 0, maxK, 0, cnfClauses);
}
  }


// Write Clauses

  st_generator *stGen;
  char         *name;
  int          cnfIndex,k;

    st_foreach_item_int(cnfClauses->cnfIndexTable, stGen, &name, &cnfIndex) {
    fprintf(outFile, "c %s %d\n",name, cnfIndex);
  }
  (void) fprintf(outFile, "p cnf %d %d\n", cnfClauses->cnfGlobalIndex-1,
		 cnfClauses->noOfClauses);
  if (cnfClauses->clauseArray != NIL(array_t)) {
    for (i = 0; i < cnfClauses->nextIndex; i++) {
      k = array_fetch(int, cnfClauses->clauseArray, i);
      (void) fprintf(outFile, "%d%c", k, (k == 0) ? '\n' : ' ');
    }
  }
 if(outFileName != NIL(char))
   fclose(outFile);
   
   return 1;

   usage:
  (void) fprintf(vis_stderr, "usage: write_cnf [-h] [-f ltl file] [-o outfile] [k]\n");
  (void) fprintf(vis_stderr, "   -h \tprint the command usage\n");
  (void) fprintf(vis_stderr, "   -f <filename> \tto add a ltlfile\n");
  (void) fprintf(vis_stderr, "   -o <filename> \twrite output in outputfile\n");
  (void) fprintf(vis_stderr, "   -k <value> \tnumber of unroll steps\n");
  return 1;
  
}
