/**CFile***********************************************************************

  FileName    [smtDl.c]

  PackageName [smt]

  Synopsis    [Routines for smt function.]

  Author      [Hyondeuk Kim]

  Copyright   [Copyright (c) 1995-2004, Regents of the University of Colorado

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

  Neither the name of the University of Colorado nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.]

******************************************************************************/

#ifdef HAVE_LIBGMP

#include "smt.h"

int
smt_idl_theory_solve(smtManager_t * sm)
{
  smt_bellman_ford_main(sm);

  if (!(sm->flag & SAT_MASK)) { 

    return 2; /* theory unsat */

  } else if (sm->litArr->size < sm->avarArr->size) {

    smt_dl_theory_propagation_main(sm); /* theory sat */

  }

  if (sm->litArr->size == sm->avarArr->size) {

    smt_check_solution(sm);
    
  }
  
  if (sm->tplits->size) { 

    return 1; /* theory prop exists */

  }

  return 0; /* sat without theory prop */
}

int
smt_rdl_theory_solve(smtManager_t * sm)
{
  smt_bellman_ford_main(sm);

  if (!(sm->flag & SAT_MASK)) { 

    return 2; /* theory unsat */

  } else if (sm->litArr->size < sm->avarArr->size) {

    smt_dl_theory_propagation_main(sm); /* theory sat */

  } 

  if (sm->litArr->size == sm->avarArr->size) {

    smt_check_solution(sm);
    
  }
  
  if (sm->tplits->size) { 

    return 1; /* theory prop exists */

  }

  return 0; /* sat without theory prop */
}

void
smt_bellman_ford_main(smtManager_t * sm)
{
  sm->stats->num_bf_call++;

  smt_generate_constraint_graph(sm);

  smt_bellman_ford_algorithm(sm);
}

void
smt_dl_theory_propagation_main(smtManager_t * sm)
{
  smt_dl_simple_theory_propagation(sm);

  if ((int) sm->stats->num_bf_call%300 == 0) {
    
    smt_init_dl_theory_propagation(sm->cG);
    
    if (sm->flag & IDL_MASK)
      smt_idl_theory_propagation(sm); 
    else if (sm->flag & RDL_MASK)
      smt_rdl_theory_propagation(sm); 
  }
}

void
smt_generate_constraint_graph(smtManager_t * sm)
{
  smtGraph_t * G;
  smtVertex_t * src, * dest;
  smtEdge_t * e;
  smtAvar_t * avar;
  smtNvar_t * lnvar, *rnvar;
  satArray_t *cur_edges; 
  double weight;
  int cur_index, lit, id;
  int vindex, eindex;
  int i;

  if (sm->cG == 0) smt_init_constraint_graph(sm);    

  G = sm->cG;
  eindex = G->esize;
  cur_edges = G->cur_edges;
  cur_edges->size = 0;

  /* generate edges */
  cur_index = sm->cur_index;

  for(i = cur_index; i < sm->litArr->size; i++) {
    lit = sm->litArr->space[i];
    id = (lit>0) ? lit : -lit;
    avar = (smtAvar_t *) sm->id2var->space[id];
    /* filter theory propagated avar */
    if (avar->flag & TPROP_MASK) continue;
    assert(avar->type != EQ_c);
      
    if (sm->avalues[id] == 1) {
      lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];
      weight = avar->constant;
    } else {
      assert(sm->avalues[id] == 0);
      lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];
      weight = -avar->constant - sm->epsilon;
    }

    vindex = lnvar->id - 1;
    dest = &(G->vHead[vindex]);
    vindex = rnvar->id - 1;
    src =  &(G->vHead[vindex]);

    e = smt_find_edge(src, dest);

    if (e) {
      if (weight < e->weight) {
	e->avar->flag |= IMPLIED_MASK;
	e->implied = sat_array_insert(e->implied, (long) e->avar);
	e->avar = avar;
	e->weight = weight;
	cur_edges = sat_array_insert(cur_edges, (long) e);

      } else {
	avar->flag |= IMPLIED_MASK;
	e->implied = sat_array_insert(e->implied, (long) avar);
      }
    } else {
      e = smt_add_edge(G, src, dest, avar, eindex++);
      e->weight = weight;
      cur_edges = sat_array_insert(cur_edges, (long) e);
      G->esize++;
    }
  }

  return;
}

void
smt_init_constraint_graph(smtManager_t * sm)
{
  smtGraph_t * G = sm->cG;
  int vindex, esize;
  int i;

  sm->cG = (smtGraph_t *) malloc(sizeof(smtGraph_t));
  G = sm->cG;
  memset(G, 0, sizeof(smtGraph_t));

  /* init vertex structure */
  G->vsize = sm->nvarArr->size;
  G->vHead = (smtVertex_t *) malloc(sizeof(smtVertex_t) * G->vsize);
  memset(G->vHead, 0, sizeof(smtVertex_t) * G->vsize);
  G->nvarArr = sm->nvarArr;
    
  for(i = 0; i < G->nvarArr->size; i++) {
    /*nvar = (smtNvar_t *) G->nvarArr->space[i];*/
    vindex = i;
    smt_add_vertex(G, vindex);
  }

  /* flag */
  G->flags = (int *) malloc(sizeof(int) * G->vsize);
  memset(G->flags, 0, sizeof(int) * (G->vsize));
  /* epsilonless */
  G->epsilonless = sm->epsilonless;
  /* path */
  G->paths = (smtVertex_t **) malloc(sizeof(smtVertex_t *) * G->vsize);
  memset(G->paths, 0, sizeof(smtVertex_t *) * G->vsize);
  G->bpaths = (smtVertex_t **) malloc(sizeof(smtVertex_t *) * G->vsize);
  memset(G->bpaths, 0, sizeof(smtVertex_t *) * G->vsize);
  G->fpaths = (smtVertex_t **) malloc(sizeof(smtVertex_t *) * G->vsize);
  memset(G->fpaths, 0, sizeof(smtVertex_t *) * G->vsize);
  /* dist */
  smt_init_dists_in_constraint_graph(sm);
  /* bvArr & fvArr */
  G->bvArr = sat_array_alloc(G->vsize);
  G->fvArr = sat_array_alloc(G->vsize);
  /* queue */
  G->queue = smt_create_queue(G->vsize);
  /* init edge structure */
  G->esize = 0;
  esize = sm->avarArr->size * 2;
  G->eHead = (smtEdge_t *) malloc(sizeof(smtEdge_t) * esize);
  memset(G->eHead, 0, sizeof(smtEdge_t) * esize);
  G->avarArr = sm->avarArr;
  G->cur_edges = sat_array_alloc(esize);
  G->neg_edges = sat_array_alloc(30);
  G->imp_edges = sat_array_alloc(esize);
  
  return;
}

void
smt_init_dists_in_constraint_graph(smtManager_t * sm)
{
  smtGraph_t * G = sm->cG;

  if (sm->flag & IDL_MASK) {
    G->idists = (int *) malloc(sizeof(int) * G->vsize);
    memset(G->idists, 0, sizeof(int) * (G->vsize));

    G->bidists = (int *) malloc(sizeof(int) * G->vsize);
    G->fidists = (int *) malloc(sizeof(int) * G->vsize);
    G->rdists = 0;
    G->brdists = 0;
    G->frdists = 0;
  } else if (sm->flag & RDL_MASK) {
    G->rdists = (double *) malloc(sizeof(double) * G->vsize);
    memset(G->rdists, 0, sizeof(double) * G->vsize); 

    G->brdists = (double *) malloc(sizeof(double) * G->vsize);
    G->frdists = (double *) malloc(sizeof(double) * G->vsize);
    G->idists = 0;
    G->bidists = 0;
    G->fidists = 0;
  } else {
    exit(0);
  }

  smt_init_theory_prop_dists(sm);
}

void
smt_init_theory_prop_dists(smtManager_t * sm)
{
  smtGraph_t * G = sm->cG;
  int i;

  if (sm->flag & IDL_MASK) {
    for(i = 0; i < G->vsize; i++) {
      G->bidists[i] = INT_MAX;
      G->fidists[i] = INT_MAX;
    }
  } else if (sm->flag & RDL_MASK) {
    for(i = 0; i < G->vsize; i++) {
      G->brdists[i] = HUGE_VAL;
      G->frdists[i] = HUGE_VAL;
    }
  }

  return;
}

void
smt_free_constraint_graph(smtGraph_t * G)
{
  if (G->idists) free(G->idists);
  if (G->rdists) free(G->rdists);
  if (G->bidists) free(G->bidists);
  if (G->fidists) free(G->fidists);
  if (G->brdists) free(G->brdists);
  if (G->frdists) free(G->frdists);

  smt_free_queue(G->queue);
  smt_free_vertex(G);
  smt_free_edge_implied(G->imp_edges);
  
  free(G->eHead);
  free(G->flags);
  free(G->paths);
  free(G->bpaths);
  free(G->fpaths);
  free(G->bvArr);
  free(G->fvArr);
  free(G->cur_edges);
  free(G->neg_edges);

  free(G);
}

void
smt_free_vertex(smtGraph_t * G)
{
  smtVertex_t * v;
  int i;
  
  for(i = 0; i < G->vsize; i++) {
    v = &(G->vHead[i]);
    free(v->ins);
    free(v->outs);
    free(v->targets);
  }

  free(G->vHead);

  return;
}

void
smt_free_edge_implied(satArray_t * edgeArr)
{
  smtEdge_t * e;
  satArray_t * implied;
  int i;
  
  for(i = 0; i < edgeArr->size; i++) {
    e = (smtEdge_t *) edgeArr->space[i];
    implied = e->implied;
    free(implied);
  }

  free(edgeArr);

  return;
}

void
smt_bellman_ford_algorithm(smtManager_t * sm)      
{
  smtGraph_t * G = sm->cG;
  smtVertex_t * v, * src = 0, * dest = 0, *parent = 0;
  smtEdge_t * e;
  smtQueue_t * Q;
  double new_dist = 0, cur_dist = 0, weight = 0;
  int i, qsize, cycleFound = 0;

  smt_init_bellman_ford_algorithm(sm);  

  Q = G->queue;

  qsize = Q->size;

  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {

    G->flags[v->index] &= RESET_FRINGE_MASK;

    if (G->flags[v->index] & VISITED_MASK)
      continue;

    for(i = 0; i < v->outs->size; i++) {
      e = (smtEdge_t *) v->outs->space[i];
      src = e->src;
      dest = e->dest;
      weight = e->weight;

      if (sm->flag & IDL_MASK) { 
	new_dist = (double) G->idists[src->index] + weight;
	cur_dist = (double) G->idists[dest->index];
      } else if (sm->flag & RDL_MASK) {
#if 1   /* 
	   for rounding error case 
	   e.g. cur_dist = -5.0000010000000001
 	        new_dist = -5.000001000000001
	*/
	weight = weight + sm->epsilonless;
#endif    
	new_dist = G->rdists[src->index] + weight;
	cur_dist = G->rdists[dest->index];
      } else 
	exit(0);

      if (new_dist < cur_dist) {
	/* check if src is in the subtree of dest.
	   if this is the case, the negative cycle is detected */
	parent = G->paths[v->index];
	while(1) {
	  if (parent == 0)
            break;
	  else if(parent == dest) {
	    G->paths[dest->index] = v;
	    G->flags[dest->index] &= RESET_VISITED_MASK;
	    cycleFound = 1;
	    break;
	  }
	  parent = G->paths[parent->index];
	}
	if (cycleFound)	break;

	smt_delete_subtree(G, dest);

	/* relaxation */
	if (sm->flag & IDL_MASK)  
	  G->idists[dest->index] = (int) new_dist;
	else if (sm->flag & RDL_MASK) {
	  new_dist = G->rdists[src->index] + e->weight;
	  G->rdists[dest->index] = new_dist;
	}

	G->paths[dest->index] = src;
	G->flags[dest->index] &= RESET_VISITED_MASK;
	
        if (!(G->flags[dest->index] & FRINGE_MASK)) {
          G->flags[dest->index] |= FRINGE_MASK; 
          smt_enqueue(Q, (long) dest);
        }
      }
    }

    if(cycleFound)	break;
  }

  if(cycleFound) {

    sm->stats->num_bf_conf++;

    sm->flag &= RESET_SAT_MASK;

    smt_collect_edges_in_neg_cycle(G, dest);

    smt_get_lemma_from_neg_cycle(sm, G->neg_edges);

    smt_retrieve_previous_distance(sm);

  } else if (qsize) {
    
    smt_update_value_with_current_distance(sm);

  }

  return;
}

void
smt_init_bellman_ford_algorithm(smtManager_t * sm)
{
  smtGraph_t * G = sm->cG;
  smtVertex_t * src, * dest;
  smtEdge_t * e;
  smtQueue_t * Q;
  double new_dist = 0, cur_dist = 0, weight = 0;
  int i;

  memset(G->flags, 0, sizeof(int) * G->vsize);
  memset(G->paths, 0, sizeof(smtVertex_t *) * G->vsize);

  Q = G->queue;
  smt_init_queue(Q);

  for(i = 0; i < G->cur_edges->size; i++) {
    e = (smtEdge_t *) G->cur_edges->space[i];
    src = e->src;
    dest = e->dest;
    weight = e->weight;

    if (sm->flag & IDL_MASK) { 
      new_dist = (double) G->idists[src->index] + weight;
      cur_dist = (double) G->idists[dest->index];
    } else if (sm->flag & RDL_MASK) {
      new_dist = G->rdists[src->index] + weight;
      cur_dist = G->rdists[dest->index];
    } else {
      exit(0);
    }

    if(new_dist < cur_dist) {
      if(!(G->flags[src->index] & FRINGE_MASK)) {
	G->flags[src->index] |= FRINGE_MASK; /* not in queue */
	smt_enqueue(Q, (long) src);
      }
    }
  }

  return;
}

void
smt_delete_subtree(smtGraph_t * G, smtVertex_t * v)
{
  smtEdge_t * e;
  int i;

  for(i = 0; i < v->outs->size; i++) {
    e = (smtEdge_t *) v->outs->space[i];

    if(G->paths[e->dest->index] == v) {
      G->paths[e->dest->index] = 0;
      G->flags[e->dest->index] |= VISITED_MASK;
      smt_delete_subtree(G, e->dest);
    }
  }

  return;
}

void
smt_collect_edges_in_neg_cycle(smtGraph_t * G, smtVertex_t * v)
{
  smtVertex_t * src, * dest;
  smtEdge_t * e;

  dest = v;
  G->neg_edges->size = 0;

  while(1) {
    if (G->flags[dest->index] & VISITED_MASK)
      break;
    src = G->paths[dest->index];
    e = smt_find_edge(src, dest);
    G->neg_edges = sat_array_insert(G->neg_edges, (long) e);
    G->flags[dest->index] |= VISITED_MASK;
    dest = src;    
  }

  return;
}

void
smt_get_lemma_from_neg_cycle(smtManager_t * sm, satArray_t * neg_edges)
{
  satManager_t *cm = sm->cm;
  smtAvar_t * avar;
  smtEdge_t * e;
  int i, id, sign, value, lit = 0;

  for(i = 0; i < neg_edges->size; i++) {
    e = (smtEdge_t *) neg_edges->space[i];
    avar = e->avar;
    value = sm->avalues[avar->id];

    if (value == 1)
      lit = avar->id;
    else if (value == 0)
      lit = -avar->id;
    else
      exit(0);
    
    sm->lemma = sat_array_insert(sm->lemma, lit);
    sign = (lit>0) ? 1 : 0;
    id = avar->id;
    cm->explanation = sat_array_insert(cm->explanation, ((id<<1) + sign));
  }
  
  return;
}

void
smt_retrieve_previous_distance(smtManager_t * sm)
{
  if (sm->flag & IDL_MASK) 
    memcpy(sm->cG->idists, sm->ivalues, sizeof(int) * (sm->cG->vsize));
  else if (sm->flag & RDL_MASK)
    memcpy(sm->cG->rdists, sm->rvalues, sizeof(double) * (sm->cG->vsize));

  return;
}

void
smt_update_value_with_current_distance(smtManager_t * sm)
{
  if (sm->flag & IDL_MASK) 
    memcpy(sm->ivalues, sm->cG->idists, sizeof(int) * (sm->cG->vsize));
  else if (sm->flag & RDL_MASK)
    memcpy(sm->rvalues, sm->cG->rdists, sizeof(double) * (sm->cG->vsize));

  return;
}

void
smt_init_dl_theory_propagation(smtGraph_t * G)
{
  memset(G->flags, 0, sizeof(int) * G->vsize);
}

void
smt_idl_theory_propagation(smtManager_t * sm)
{
  satManager_t * cm = sm->cm;
  smtGraph_t * G = sm->cG;
  smtAvar_t * avar;
  smtNvar_t * lnvar, * rnvar;
  smtVertex_t * src, * dest;
  int num_bavar, num_favar;
  int i, j, id, lit, level, vindex;

  level = cm->decisionStackSeperator->size;
#if 1
  if (level == 0) j = 1;
  else j = 3;
#else
  j = 1;
#endif

  for(i = sm->cur_index; i<sm->litArr->size; i+=j) { 

    lit = sm->litArr->space[i];
    id = (lit>0) ? lit : -lit;
    avar = (smtAvar_t *) sm->id2var->space[id];

    if (avar->flag & IMPLIED_MASK || avar->flag & TPROP_MASK) {
      if (i + j <= 0) {
	i = sm->cur_index;
	j = 3;
      }

      continue;
    }

    assert(avar->type != EQ_c);

    if (sm->avalues[id] == 1) {
      lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];
    } else {
      assert(sm->avalues[id] == 0);
      lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];
    }

    vindex = lnvar->id - 1;
    dest = &(G->vHead[vindex]);
    vindex = rnvar->id - 1;
    src =  &(G->vHead[vindex]);

    G->bvArr->size = 0;

    G->fvArr->size = 0;

    num_bavar = 0;

    num_favar = 0;

    smt_idl_gather_backward_reachable_vertex_with_dfs(G, src, &num_bavar);

    smt_idl_gather_forward_reachable_vertex_with_dfs(G, dest, &num_favar);
    
    if (num_bavar > num_favar) {
      
      smt_idl_theory_propagation_with_bv_arr(sm, avar);

    } else {

      smt_idl_theory_propagation_with_fv_arr(sm, avar);

    }

    smt_idl_init_reachable_vertex(G);
  }

  return;
}

void
smt_rdl_theory_propagation(smtManager_t * sm)
{
  satManager_t * cm = sm->cm;
  smtGraph_t * G = sm->cG;
  smtAvar_t * avar;
  smtNvar_t * lnvar, * rnvar;
  smtVertex_t * src, * dest;
  int num_bavar, num_favar;
  int i, j, id, lit, level, vindex;

  level = (int) cm->decisionStackSeperator->size;
#if 1
  if (level == 0) j = 1;
  else j = 3;
#else
  j = 1;
#endif

  for(i = sm->cur_index; i<sm->litArr->size; i+=j) { 

    lit = sm->litArr->space[i];
    id = (lit>0) ? lit : -lit;
    avar = (smtAvar_t *) sm->id2var->space[id];

    if (avar->flag & IMPLIED_MASK || avar->flag & TPROP_MASK) {
      if (i + j <= 0) {
	i = sm->cur_index;
	j = 3;
      }

      continue;
    }

    if (sm->avalues[id] == 1) {
      lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];
    } else {
      assert(sm->avalues[id] == 0);
      lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];
    }

    vindex = lnvar->id - 1;
    dest = &(G->vHead[vindex]);
    vindex = rnvar->id - 1;
    src =  &(G->vHead[vindex]);

    G->bvArr->size = 0;

    G->fvArr->size = 0;

    num_bavar = 0;

    num_favar = 0;

    smt_rdl_gather_backward_reachable_vertex_with_dfs(G, src, &num_bavar);

    smt_rdl_gather_forward_reachable_vertex_with_dfs(G, dest, &num_favar);
    
    if (num_bavar > num_favar) { 

      smt_rdl_theory_propagation_with_bv_arr(sm, avar);

    } else {

      smt_rdl_theory_propagation_with_fv_arr(sm, avar);
      
    }

    smt_rdl_init_reachable_vertex(G); 
  }
}

void
smt_idl_gather_backward_reachable_vertex_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v,
  int * num_avar)
{
  int depth; 

  G->bidists[v->index] = 0;
  G->bpaths[v->index] = 0;
  depth = 0;

  smt_idl_traverse_backward_with_dfs(G, v, num_avar, depth);
}

void
smt_idl_gather_forward_reachable_vertex_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v,
  int * num_avar)
{
  int depth; 

  G->fidists[v->index] = 0;
  G->fpaths[v->index] = 0;
  depth = 0;

  smt_idl_traverse_forward_with_dfs(G, v, num_avar, depth);
}

void
smt_rdl_gather_backward_reachable_vertex_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v,
  int * num_avar)
{
  int depth; 

  G->brdists[v->index] = 0;
  G->bpaths[v->index] = 0;
  depth = 0;

  smt_rdl_traverse_backward_with_dfs(G, v, num_avar, depth);
}

void
smt_rdl_gather_forward_reachable_vertex_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v,
  int * num_avar)
{
  int depth; 

  G->frdists[v->index] = 0;
  G->fpaths[v->index] = 0;
  depth = 0;

  smt_rdl_traverse_forward_with_dfs(G, v, num_avar, depth);
}

void
smt_idl_traverse_backward_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v, 
  int * num_avar, 
  int depth)
{
  smtNvar_t * nvar;
  smtVertex_t * w;
  smtEdge_t * e;
  double weight;
  int i;
#if 0
  depth++;  
#endif

  G->flags[v->index] |= BVISITED_MASK;

  if (!(G->flags[v->index] & BFRINGE_MASK)) {
    G->bvArr = sat_array_insert(G->bvArr, (long) v);
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    *num_avar += nvar->avarArr->size;
    G->flags[v->index] |= BFRINGE_MASK;
  }

  for(i = 0; i < v->ins->size; i++) {
    e = (smtEdge_t *) v->ins->space[i];
    w = e->src;
    weight = e->weight;

    if (!(G->flags[w->index] & BVISITED_MASK) ||
	G->bidists[w->index] > G->bidists[v->index] + (int) weight) {

      G->bidists[w->index] = G->bidists[v->index] + (int) weight;
      G->bpaths[w->index] = v;
      smt_idl_traverse_backward_with_dfs(G, w, num_avar, depth);      
    }
  }

  return;
}

void
smt_idl_traverse_forward_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v, 
  int * num_avar, 
  int depth)
{
  smtNvar_t * nvar;
  smtVertex_t * w;
  smtEdge_t * e;
  double weight;
  int i;
#if 0
  depth++;
#endif

  G->flags[v->index] |= FVISITED_MASK;

  if (!(G->flags[v->index] & FFRINGE_MASK)) {
    G->fvArr = sat_array_insert(G->fvArr, (long) v);
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    *num_avar += nvar->avarArr->size;
    G->flags[v->index] |= FFRINGE_MASK;
  }

  for(i = 0; i < v->outs->size; i++) {
    e = (smtEdge_t *) v->outs->space[i];
    w = e->dest;
    weight = e->weight;

    if (!(G->flags[w->index] & FVISITED_MASK) ||
	G->fidists[w->index] > G->fidists[v->index] + (int) weight) {

      G->fidists[w->index] = G->fidists[v->index] + (int) weight;
      G->fpaths[w->index] = v;
      smt_idl_traverse_forward_with_dfs(G, w, num_avar, depth);      
    }
  }

  return;
}

void
smt_rdl_traverse_backward_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v, 
  int * num_avar, 
  int depth)
{
  smtNvar_t * nvar;
  smtVertex_t * w;
  smtEdge_t * e;
  double weight;
  int i;
#if 0
  depth++;
#endif

  G->flags[v->index] |= BVISITED_MASK;

  if (!(G->flags[v->index] & BFRINGE_MASK)) {
    G->bvArr = sat_array_insert(G->bvArr, (long) v);
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    *num_avar += nvar->avarArr->size;
    G->flags[v->index] |= BFRINGE_MASK;
  }

  for(i = 0; i < v->ins->size; i++) {
    e = (smtEdge_t *) v->ins->space[i];
    w = e->src;
    weight = e->weight + G->epsilonless; 

    if (!(G->flags[w->index] & BVISITED_MASK) ||
    G->brdists[w->index] > G->brdists[v->index] + weight) {

      G->brdists[w->index] = G->brdists[v->index] + e->weight;
      G->bpaths[w->index] = v;
      smt_rdl_traverse_backward_with_dfs(G, w, num_avar, depth);
      
    }
  }

  return;
}

void
smt_rdl_traverse_forward_with_dfs(
  smtGraph_t * G, 
  smtVertex_t * v, 
  int * num_avar, 
  int depth)
{
  smtNvar_t * nvar;
  smtVertex_t * w;
  smtEdge_t * e;
  double weight;
  int i;
#if 0
  depth++;
#endif

  G->flags[v->index] |= FVISITED_MASK;

  if (!(G->flags[v->index] & FFRINGE_MASK)) {
    G->fvArr = sat_array_insert(G->fvArr, (long) v);
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    *num_avar += nvar->avarArr->size;
    G->flags[v->index] |= FFRINGE_MASK;
  }

  for(i = 0; i < v->outs->size; i++) {
    e = (smtEdge_t *) v->outs->space[i];
    w = e->dest;

    weight = e->weight + G->epsilonless; 

    if (!(G->flags[w->index] & FVISITED_MASK) ||
	G->frdists[w->index] > G->frdists[v->index] + weight) {

      G->frdists[w->index] = G->frdists[v->index] + e->weight;
      G->fpaths[w->index] = v;
      smt_rdl_traverse_forward_with_dfs(G, w, num_avar, depth);      
    }
  }

  return;
}

void
smt_idl_gather_backward_reachable_vertex_with_bfs(
  smtGraph_t * G, 
  smtVertex_t * src,
  int * num_avar)
{
  smtVertex_t * v, * w;
  smtEdge_t * e;
  smtNvar_t * nvar;
  smtQueue_t * Q = G->queue;
  double weight;
  int i;

  G->bidists[src->index] = 0;
  G->bpaths[src->index] = 0;

  smt_init_queue(Q);

  smt_enqueue(Q, (long) src);
    
  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {
    
    G->flags[v->index] &= RESET_BFRINGE_MASK;
    
    if (!(G->flags[v->index] & BARR_MASK)) {
      G->bvArr = sat_array_insert(G->bvArr, (long) v);
      nvar = (smtNvar_t *) G->nvarArr->space[v->index];
      *num_avar += nvar->avarArr->size;
      G->flags[v->index] |= BARR_MASK;
    }

    for(i = 0; i < v->ins->size; i++) {
      e = (smtEdge_t *) v->ins->space[i];
      w = e->src;
      weight = e->weight;

      if (G->bidists[w->index] > G->bidists[v->index] + (int) weight) {

	G->bidists[w->index] = G->bidists[v->index] + (int) weight;
	G->bpaths[w->index] = v;
	if (!(G->flags[w->index] & BFRINGE_MASK)) {
	  smt_enqueue(Q, (long) w);
	  G->flags[w->index] |= BFRINGE_MASK;
	}	
      }
    }
  }

  return;
}

void
smt_rdl_gather_backward_reachable_vertex_with_bfs(
  smtGraph_t * G, 
  smtVertex_t * src,
  int * num_avar)
{
  smtVertex_t * v, * w;
  smtEdge_t * e;
  smtNvar_t * nvar;
  smtQueue_t * Q = G->queue;
  double weight;
  int i;

  G->brdists[src->index] = 0;
  G->bpaths[src->index] = 0;

  smt_init_queue(Q);

  smt_enqueue(Q, (long) src);
    
  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {
    
    G->flags[v->index] &= RESET_BFRINGE_MASK;
    
    if (!(G->flags[v->index] & BARR_MASK)) {
      G->bvArr = sat_array_insert(G->bvArr, (long) v);
      nvar = (smtNvar_t *) G->nvarArr->space[v->index];
      *num_avar += nvar->avarArr->size;
      G->flags[v->index] |= BARR_MASK;
    }

    for(i = 0; i < v->ins->size; i++) {
      e = (smtEdge_t *) v->ins->space[i];
      w = e->src;
      weight = e->weight + G->epsilonless; 

      if (G->brdists[w->index] > G->brdists[v->index] + weight) {

	G->brdists[w->index] = G->brdists[v->index] + e->weight;
	G->bpaths[w->index] = v;
	if (!(G->flags[w->index] & BFRINGE_MASK)) {
	  smt_enqueue(Q, (long) w);
	  G->flags[w->index] |= BFRINGE_MASK;
	}	
      }
    }
  }

  return;
}

void
smt_idl_gather_forward_reachable_vertex_with_bfs(
  smtGraph_t * G, 
  smtVertex_t * dest,
  int * num_avar)
{
  smtVertex_t * v, * w;
  smtEdge_t * e;
  smtNvar_t * nvar;
  smtQueue_t * Q = G->queue;
  double weight;
  int i;

  G->fidists[dest->index] = 0;
  G->fpaths[dest->index] = 0;

  smt_init_queue(Q);

  smt_enqueue(Q, (long) dest);
    
  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {
    
    G->flags[v->index] &= RESET_FFRINGE_MASK;
    
    if (!(G->flags[v->index] & FARR_MASK)) {
      G->fvArr = sat_array_insert(G->fvArr, (long) v);
      nvar = (smtNvar_t *) G->nvarArr->space[v->index];
      *num_avar += nvar->avarArr->size;
      G->flags[v->index] |= FARR_MASK;
    }

    for(i = 0; i < v->outs->size; i++) {
      e = (smtEdge_t *) v->outs->space[i];
      w = e->dest;
      weight = e->weight;

      if (G->fidists[w->index] > G->fidists[v->index] + (int) weight) {

	G->fidists[w->index] = G->fidists[v->index] + (int) weight;
	G->fpaths[w->index] = v;
	if (!(G->flags[w->index] & FFRINGE_MASK)) {
	  smt_enqueue(Q, (long) w);
	  G->flags[w->index] |= FFRINGE_MASK;
	}	
      }
    }
  }

  return;
}

void
smt_rdl_gather_forward_reachable_vertex_with_bfs(
  smtGraph_t * G, 
  smtVertex_t * dest,
  int * num_avar)
{
  smtVertex_t * v, * w;
  smtEdge_t * e;
  smtNvar_t * nvar;
  smtQueue_t * Q = G->queue;
  double weight;
  int i;

  G->frdists[dest->index] = 0;
  G->fpaths[dest->index] = 0;

  smt_init_queue(Q);

  smt_enqueue(Q, (long) dest);
    
  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {
    
    G->flags[v->index] &= RESET_FFRINGE_MASK;
    
    if (!(G->flags[v->index] & FARR_MASK)) {
      G->fvArr = sat_array_insert(G->fvArr, (long) v);
      nvar = (smtNvar_t *) G->nvarArr->space[v->index];
      *num_avar += nvar->avarArr->size;
      G->flags[v->index] |= FARR_MASK;
    }

    for(i = 0; i < v->outs->size; i++) {
      e = (smtEdge_t *) v->outs->space[i];
      w = e->dest;
      weight = e->weight + G->epsilonless;

      if (G->frdists[w->index] > G->frdists[v->index] + weight) {

	G->frdists[w->index] = G->frdists[v->index] + e->weight;
	G->fpaths[w->index] = v;
	if (!(G->flags[w->index] & FFRINGE_MASK)) {
	  smt_enqueue(Q, (long) w);
	  G->flags[w->index] |= FFRINGE_MASK;
	}	
      }
    }
  }

  return;
}

void
smt_idl_theory_propagation_with_bv_arr(smtManager_t * sm, smtAvar_t * avar)
{
  smtGraph_t * G = sm->cG;
  smtAvar_t * tmp_avar;
  smtNvar_t * nvar;/*, * rnvar, * lnvar;*/
  smtNvar_t * tmp_rnvar, * tmp_lnvar;
  smtVertex_t * v;
  satArray_t * bvArr;
  double weight, tmp_weight;
  int dist, id;
  int i, j;

  id = avar->id;

  if (sm->avalues[id] == 1) {
    /*lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];*/
    weight = avar->constant;
  } else {
    assert(sm->avalues[id] == 0);
    /*lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];*/
    weight = -avar->constant - sm->epsilon;
  }  

  bvArr = G->bvArr;

  for(i = 0; i < bvArr->size; i++) { 
    v = (smtVertex_t *) bvArr->space[i];
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    
    for(j = 0; j < nvar->avarArr->size; j++) {
      tmp_avar = (smtAvar_t *) nvar->avarArr->space[j]; 
      sm->stats->num_tprop_call++;
      if (sm->avalues[tmp_avar->id] != 2) continue;
      tmp_lnvar = (smtNvar_t *) tmp_avar->nvars->space[0];
      tmp_rnvar = (smtNvar_t *) tmp_avar->nvars->space[1];
      tmp_weight = tmp_avar->constant;
     
      if (nvar == tmp_rnvar && G->fidists[tmp_lnvar->id-1] != INT_MAX) {
	dist = G->fidists[tmp_lnvar->id-1] + 
	  G->bidists[tmp_rnvar->id-1] + (int) weight;
	
	if (dist <= tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> x - y <= weight (dist <= weight)  */
	  sm->avalues[tmp_avar->id] = 1;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_rnvar, tmp_lnvar);
	}
      } else if (nvar == tmp_lnvar && 
		 G->fidists[tmp_rnvar->id-1] != INT_MAX) {
	dist = G->bidists[tmp_lnvar->id-1] + 
	  G->fidists[tmp_rnvar->id-1] + (int) weight;

	if (-dist > tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> y - x <= weight (dist + weight < 0)  */
	  sm->avalues[tmp_avar->id] = 0;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_lnvar, tmp_rnvar);
	}
      }
    }
  }

  return;
}

void
smt_idl_theory_propagation_with_fv_arr(smtManager_t * sm, smtAvar_t * avar)
{
  smtGraph_t * G = sm->cG;
  smtAvar_t * tmp_avar;
  smtNvar_t * nvar;
  smtNvar_t * tmp_rnvar, * tmp_lnvar;
  smtVertex_t * v;
  satArray_t * fvArr;
  double weight, tmp_weight;
  int dist, id;
  int i, j;

  id = avar->id;

  if (sm->avalues[id] == 1) {
    /*lnvar = (smtNvar_t *) avar->nvars->space[0];
    rnvar = (smtNvar_t *) avar->nvars->space[1];*/
    weight = avar->constant;
  } else {
    assert(sm->avalues[id] == 0);
    /*lnvar = (smtNvar_t *) avar->nvars->space[1];
    rnvar = (smtNvar_t *) avar->nvars->space[0];*/
    weight = -avar->constant - sm->epsilon;
  }  

  fvArr = G->fvArr;

  for(i = 0; i < fvArr->size; i++) { 
    v = (smtVertex_t *) fvArr->space[i];
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    
    for(j = 0; j < nvar->avarArr->size; j++) {
      tmp_avar = (smtAvar_t *) nvar->avarArr->space[j]; 
      sm->stats->num_tprop_call++;
      if (sm->avalues[tmp_avar->id] != 2) continue;
      tmp_lnvar = (smtNvar_t *) tmp_avar->nvars->space[0];
      tmp_rnvar = (smtNvar_t *) tmp_avar->nvars->space[1];
      tmp_weight = tmp_avar->constant;
     
      if (nvar == tmp_lnvar && G->bidists[tmp_rnvar->id-1] != INT_MAX) {
	dist = G->fidists[tmp_lnvar->id-1] + 
	  G->bidists[tmp_rnvar->id-1] + (int) weight;
	
	if (dist <= (int) tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> x - y <= weight (dist <= weight)  */
	  sm->avalues[tmp_avar->id] = 1;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_rnvar, tmp_lnvar);
	}
      } else if (nvar == tmp_rnvar && 
		 G->bidists[tmp_lnvar->id-1] != INT_MAX) {
	dist = G->bidists[tmp_lnvar->id-1] + 
	  G->fidists[tmp_rnvar->id-1] + (int) weight;

	if (-dist > (int) tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> y - x <= weight (dist + weight < 0)  */
	  sm->avalues[tmp_avar->id] = 0;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_lnvar, tmp_rnvar);
	}
      }
    }
  }

  return;
}

void
smt_rdl_theory_propagation_with_bv_arr(smtManager_t * sm, smtAvar_t * avar)
{
  smtGraph_t * G = sm->cG;
  smtAvar_t * tmp_avar;
  smtNvar_t * nvar; /*, * rnvar, * lnvar;*/
  smtNvar_t * tmp_rnvar, * tmp_lnvar;
  smtVertex_t * v;
  satArray_t * bvArr;
  double weight, tmp_weight, dist;
  int id;
  int i, j;

  id = avar->id;

  if (sm->avalues[id] == 1) {
    /*lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];*/
    weight = avar->constant;
  } else {
    assert(sm->avalues[id] == 0);
    /*lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];*/
    weight = -avar->constant - sm->epsilon;
  }  

  bvArr = G->bvArr;

  for(i = 0; i < bvArr->size; i++) { 
    v = (smtVertex_t *) bvArr->space[i];
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    
    for(j = 0; j < nvar->avarArr->size; j++) {
      tmp_avar = (smtAvar_t *) nvar->avarArr->space[j]; 
      sm->stats->num_tprop_call++;
      if (sm->avalues[tmp_avar->id] != 2) continue;
      tmp_lnvar = (smtNvar_t *) tmp_avar->nvars->space[0];
      tmp_rnvar = (smtNvar_t *) tmp_avar->nvars->space[1];
      tmp_weight = tmp_avar->constant;
     
      if (nvar == tmp_rnvar && G->frdists[tmp_lnvar->id-1] < HUGE_VAL) {
	dist = G->frdists[tmp_lnvar->id-1] + 
	  G->brdists[tmp_rnvar->id-1] + weight;
	
	if (dist <= tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> x - y <= weight (dist <= weight)  */
	  sm->avalues[tmp_avar->id] = 1;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_rnvar, tmp_lnvar);
	}
      } else if (nvar == tmp_lnvar && 
		 G->frdists[tmp_rnvar->id-1] < HUGE_VAL) {
	dist = G->brdists[tmp_lnvar->id-1] + 
	  G->frdists[tmp_rnvar->id-1] + weight;

	if (-dist > tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> y - x <= weight (dist + weight < 0)  */
	  sm->avalues[tmp_avar->id] = 0;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_lnvar, tmp_rnvar);
	}
      }
    }
  }

  return;
}

void
smt_rdl_theory_propagation_with_fv_arr(smtManager_t * sm, smtAvar_t * avar)
{
  smtGraph_t * G = sm->cG;
  smtAvar_t * tmp_avar;
  smtNvar_t * nvar; /*, * rnvar, * lnvar;*/
  smtNvar_t * tmp_rnvar, * tmp_lnvar;
  smtVertex_t * v;
  satArray_t * fvArr;
  double weight, tmp_weight, dist;
  int id;
  int i, j;

  id = avar->id;

  if (sm->avalues[id] == 1) {
    /*lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];*/
    weight = avar->constant;
  } else {
    assert(sm->avalues[id] == 0);
    /*lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];*/
    weight = -avar->constant - sm->epsilon;
  }  

  fvArr = G->fvArr;

  for(i = 0; i < fvArr->size; i++) { 
    v = (smtVertex_t *) fvArr->space[i];
    nvar = (smtNvar_t *) G->nvarArr->space[v->index];
    
    for(j = 0; j < nvar->avarArr->size; j++) {
      tmp_avar = (smtAvar_t *) nvar->avarArr->space[j]; 
      sm->stats->num_tprop_call++;
      if (sm->avalues[tmp_avar->id] != 2) continue;
      tmp_lnvar = (smtNvar_t *) tmp_avar->nvars->space[0];
      tmp_rnvar = (smtNvar_t *) tmp_avar->nvars->space[1];
      tmp_weight = tmp_avar->constant;
     
      if (nvar == tmp_lnvar && G->brdists[tmp_rnvar->id-1] < HUGE_VAL) {
	dist = G->frdists[tmp_lnvar->id-1] + 
	  G->brdists[tmp_rnvar->id-1] + weight;
	
	if (dist <= tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> x - y <= weight (dist <= weight)  */
	  sm->avalues[tmp_avar->id] = 1;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_rnvar, tmp_lnvar);
	}
      } else if (nvar == tmp_rnvar && 
		 G->brdists[tmp_lnvar->id-1] < HUGE_VAL) {
	dist = G->brdists[tmp_lnvar->id-1] + 
	  G->frdists[tmp_rnvar->id-1] + weight;

	if (-dist > tmp_weight) {
	  sm->stats->num_tprop++;
	  /*  x - y <= dist ---> y - x <= weight (dist + weight < 0)  */
	  sm->avalues[tmp_avar->id] = 0;
	  smt_get_theory_prop_reason(sm, avar, tmp_avar, 
					tmp_lnvar, tmp_rnvar);
	}
      }
    }
  }

  return;
}

void
smt_dl_simple_theory_propagation(smtManager_t * sm)
{
  smtAvar_t * avar, * sis_avar;
  smtNvar_t * lnvar, * sis_lnvar;
  double weight, sis_weight;
  int lit, id;
  int i, j;

#if 0
  level = sm->cm->decisionStackSeperator->size;
  skip_tp_cond = level!=0 && 
    (smt_gv->color & RDL_MASK && sm->bvarArr->size>100 &&
    sm->avarArr->size > 50*sm->bvarArr->size); 
  if (skip_tp_cond) return;
#endif

  for(i = sm->cur_index; i < sm->litArr->size; i++) {
    lit = sm->litArr->space[i];
    id = (lit > 0)? lit : -lit;
    avar = (smtAvar_t *) sm->id2var->space[id];

    sm->stats->num_simp_tprop_call++;
    
    if (avar->flag & IMPLIED_MASK || avar->flag & TPROP_MASK)
      continue;
    
    if (sm->avalues[id] == 1) {
      lnvar = (smtNvar_t *) avar->nvars->space[0];
      weight = avar->constant;
    } else {
      assert(sm->avalues[id] == 0);
      lnvar = (smtNvar_t *) avar->nvars->space[1];
      weight = -avar->constant - sm->epsilon;
    }

    for(j = 0; j < avar->sis_avars->num; j++) { /* check sister avar */
      sis_avar = array_fetch(smtAvar_t *, avar->sis_avars, j);
      if (sm->avalues[sis_avar->id] != 2) continue;

      sis_lnvar = (smtNvar_t *) sis_avar->nvars->space[0];
      sis_weight = sis_avar->constant;

      if (sis_lnvar == lnvar) {
	if (sis_weight >= weight) {
	  sm->stats->num_simp_tprop++;
	  sm->avalues[sis_avar->id] = 1;
	  smt_get_theory_prop_reason(sm, avar, sis_avar, 0, 0);
	}
      } else {
	if (sis_weight < -weight) {
	  sm->stats->num_simp_tprop++;
	  sm->avalues[sis_avar->id] = 0;
	  smt_get_theory_prop_reason(sm, avar, sis_avar, 0, 0);
	}
      }
    }
  }

  return; 
}

void
smt_get_theory_prop_reason(
  smtManager_t * sm,
  smtAvar_t * implying, 
  smtAvar_t * implied,
  smtNvar_t * bnvar,
  smtNvar_t * fnvar)
{
  satManager_t * cm = sm->cm;
  smtGraph_t * G = sm->cG;
  smtVertex_t * bv, * fv, * path;
  smtEdge_t * e;
  smtAvar_t * avar;
  int sign, id;

  /*
   * ((implying /\ avars_in_bpath /\ avars_in_fpath)' \/ implied) 
   */

  sm->lemma->size = 0;

  /* implied avar */
  id = implied->id;
  sign = (sm->avalues[id] == 1) ? 0 : 1;
  sm->lemma = sat_array_insert(sm->lemma, (id<<1)+sign); 

  if (bnvar) {
    /* reasons from backward shortest path */
    bv = &(G->vHead[bnvar->id-1]);
    path = G->bpaths[bv->index];
    
    while(path) {
      e = smt_find_edge(bv, path);
      avar = e->avar;
      id = avar->id;
      sign = (sm->avalues[id] == 1) ? 1 : 0; /* negated */
      sm->lemma = sat_array_insert(sm->lemma, (id<<1)+sign); 
      bv = path;
      path = G->bpaths[bv->index];
    }
  }
   
  if (fnvar) {
    /* reasons from forward shortest path */
    fv = &(G->vHead[fnvar->id-1]);
    path = G->fpaths[fv->index];
    
    while(path) {
      e = smt_find_edge(path, fv);
      avar = e->avar;
      id = avar->id;
      sign = (sm->avalues[id] == 1) ? 1 : 0; /* negated */
      sm->lemma = sat_array_insert(sm->lemma, (id<<1)+sign);
      fv = path;
      path = G->fpaths[fv->index];
    }
  }

  /* reason from currently assigned implying */
  id = implying->id;
  sign = (sm->avalues[id] == 1) ? 1 : 0; /* negated */
  sm->lemma = sat_array_insert(sm->lemma, (id<<1)+sign);
  assert(sm->lemma->size);

  smt_add_theory_clause(cm, implied, sm->lemma); 

  return;
}

void
smt_idl_init_reachable_vertex(smtGraph_t * G)
{
  smtVertex_t * v;
  int i;

  for(i = 0; i < G->bvArr->size; i++) {
    v = (smtVertex_t *) G->bvArr->space[i];

    G->flags[v->index] = 0;
    G->bpaths[v->index] = 0;
    G->fpaths[v->index] = 0;
    G->bidists[v->index] = INT_MAX;
    G->fidists[v->index] = INT_MAX;
  }

  for(i = 0; i < G->fvArr->size; i++) {
    v = (smtVertex_t *) G->fvArr->space[i];

    G->flags[v->index] = 0;
    G->bpaths[v->index] = 0;
    G->fpaths[v->index] = 0;
    G->bidists[v->index] = INT_MAX;
    G->fidists[v->index] = INT_MAX;
  }
}

void
smt_rdl_init_reachable_vertex(smtGraph_t * G)
{
  smtVertex_t * v;
  int i;

  for(i = 0; i < G->bvArr->size; i++) {
    v = (smtVertex_t *) G->bvArr->space[i];

    G->flags[v->index] = 0;
    G->bpaths[v->index] = 0;
    G->fpaths[v->index] = 0;
    G->brdists[v->index] = HUGE_VAL;
    G->frdists[v->index] = HUGE_VAL;
  }

  for(i = 0; i < G->fvArr->size; i++) {
    v = (smtVertex_t *) G->fvArr->space[i];

    G->flags[v->index] = 0;
    G->bpaths[v->index] = 0;
    G->fpaths[v->index] = 0;
    G->brdists[v->index] = HUGE_VAL;
    G->frdists[v->index] = HUGE_VAL;
  }
}

#endif
