/**CFile***********************************************************************

  FileName    [smtMp.c]

  PackageName [smt]

  Synopsis    [Routines for smt mp function.]

  Author      [Hyondeuk Kim]

  Copyright   [Copyright (c) 1995-2004, Regents of the University of Colorado

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

  Neither the name of the University of Colorado nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.]

******************************************************************************/

#ifdef HAVE_LIBGMP

#include "smt.h"

smtMp_t *
smt_mp_init(smtManager_t * sm)
{
  smtMp_t * mp = (smtMp_t *) malloc(sizeof(smtMp_t));
  
  sm->mp = mp;

  smt_mp_init_pool(mp);

  mpq_init(mp->zero);

  mpq_init(mp->one);

  mpq_init(mp->m_one);

  mpq_set_d(mp->zero, 0);

  mpq_set_d(mp->one, 1);

  mpq_set_d(mp->m_one, -1);

  smt_mp_init_atom_constants(sm);

  smt_mp_init_nvar_values(sm);
  
  return mp;
}

void
smt_mp_init_pool(smtMp_t * mp)
{
  mpq_t * pool;
  int i;

  mp->plimit = 10;

  pool = (mpq_t *) malloc(sizeof(mpq_t) * mp->plimit);

  mp->pool = pool;

  for(i = 0; i < mp->plimit; i++) {
    mpq_init(pool[i]);
  }
}

void
smt_mp_init_nvar_values(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  int i, size;

  mp->num_nvar = sm->nvarArr->size;

  size = mp->num_nvar + 1;

  mp->values = (mpq_t *) malloc(sizeof(mpq_t) * size);
  mp->pvalues = (mpq_t *) malloc(sizeof(mpq_t) * size);

  for(i = 0; i < size; i++) {
    mpq_init(mp->values[i]);
    mpq_init(mp->pvalues[i]);
    mpq_set_d(mp->values[i], 0);
    mpq_set_d(mp->pvalues[i], 0);
  }
}

void
smt_mp_init_atom_constants(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  smtAvar_t * avar;
  int i, size, id;

  mp->num_avar = sm->avarArr->size;
  
  size = mp->num_avar + 1;

  mp->constants = (mpq_t *) malloc(sizeof(mpq_t) * size);
  memset(mp->constants, 0, sizeof(mpq_t) * size);
  mp->weights = (mpq_t *) malloc(sizeof(mpq_t) * size);
  memset(mp->weights, 0, sizeof(mpq_t) * size);

  for(i = 0; i < sm->avarArr->size; i++) {
    avar = (smtAvar_t *) sm->avarArr->space[i];
    id = avar->id;
    mpq_init(mp->constants[id]);
    mpq_init(mp->weights[id]);
    mpq_set_d(mp->weights[id], 0);
  }

  return;
}

void
smt_mp_free(smtMp_t * mp)
{
  smt_mp_free_pool(mp);

  mpq_clear(mp->zero);

  mpq_clear(mp->one);

  mpq_clear(mp->m_one);

  smt_mp_free_atom_constants(mp);

  smt_mp_free_nvar_values(mp);
  
  free(mp);
}

void
smt_mp_free_pool(smtMp_t * mp)
{
  mpq_t * pool = mp->pool;
  int i;

  for(i = 0; i < mp->plimit; i++) {
    mpq_clear(pool[i]);
  }

  free(pool);
}

void
smt_mp_free_atom_constants(smtMp_t * mp)
{
  int i, id;

  for(i = 0; i < mp->num_avar; i++) {
    id = i + 1;
    mpq_clear(mp->constants[id]);
    mpq_clear(mp->weights[id]);
  }

  free(mp->constants);
  free(mp->weights);

  return;
}

void
smt_mp_free_nvar_values(smtMp_t * mp)
{
  int i, size;

  size = mp->num_nvar + 1;

  for(i = 0; i < size; i++) {
    mpq_clear(mp->values[i]);
    mpq_clear(mp->pvalues[i]);
  }

  free(mp->values);
  free(mp->pvalues);
}

void
smt_mp_assign_atom_constants(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  smtFml_t * avfml;
  smtAvar_t * avar;
  int i;

  for(i = 0; i < sm->avfmlArr->size; i++) {
    avfml = (smtFml_t *) sm->avfmlArr->space[i];
    avar = avfml->avar;
    if (!avar) continue;
    smt_mp_assign_atom_constant(mp, avfml);    
  }

  return;
}

void
smt_mp_assign_atom_constant(smtMp_t * mp, smtFml_t * avfml)
{
  mpq_t * pool = mp->pool;
  mpq_t * mp_constant = 0, * coeff = 0;
  smtAvar_t * avar = avfml->avar;
  smtFml_t * lfml, * rfml;
  int nminus;

  /* mp vars */
  mp_constant = &pool[0];
  coeff = &pool[1];
  
  lfml = (smtFml_t *) avfml->subfmlArr->space[0];
  rfml = (smtFml_t *) avfml->subfmlArr->space[1];
    
  mpq_set_d(*mp_constant, 0);
  
  mpq_set_d(*coeff, 1);
  nminus = 0;
  
  smt_mp_get_atom_constant(mp, lfml, mp_constant, coeff, nminus);
  
  mpq_set_d(*coeff, 1);
  nminus = 1;
  
  smt_mp_get_atom_constant(mp, rfml, mp_constant, coeff, nminus);

  /* assign atom constant */
  mpq_set(mp->constants[avar->id], *mp_constant);
  
  return;
}

void
smt_mp_get_atom_constant(
  smtMp_t * mp,
  smtFml_t * fml,
  mpq_t * mp_constant,
  mpq_t * coeff,
  int nminus)
{
  mpq_t * pool = mp->pool;
  mpq_t * tmp_a = 0, * tmp_b = 0;
  smtFml_t * lfml, * rfml, * tfml;
  char * str_a, * str_b;
  int i;

  /* mp vars */
  tmp_a = &pool[2];
  tmp_b = &pool[3];
  
  if (fml->type == NUM_c) {
    mpq_set_str(*tmp_a, (char *) fml->subfmlArr->space[0], 10);
    if (nminus%2 == 1) {
      mpq_mul(*tmp_a, *coeff, *tmp_a);
    } else {
      mpq_neg(*tmp_a, *tmp_a);
      mpq_mul(*tmp_a, *coeff, *tmp_a);
    }

    mpq_add(*mp_constant, *mp_constant, *tmp_a);
    
  } else if (fml->type == MINUS_c) {
    nminus++;
    tfml = (smtFml_t *) fml->subfmlArr->space[0];
    smt_mp_get_atom_constant(mp, tfml, mp_constant, coeff, nminus);
    
  } else if (fml->type == SUB_c) {
    lfml = (smtFml_t *) fml->subfmlArr->space[0];
    rfml = (smtFml_t *) fml->subfmlArr->space[1];
    
    smt_mp_get_atom_constant(mp, lfml, mp_constant, coeff, nminus);
    smt_mp_get_atom_constant(mp, rfml, mp_constant, coeff, nminus + 1);

  } else if (fml->type == ADD_c) {
    for(i = 0; i < fml->subfmlArr->size; i++) {
      tfml = (smtFml_t *) fml->subfmlArr->space[i];
      smt_mp_get_atom_constant(mp, tfml, mp_constant, coeff, nminus);
    }
    
  } else if (fml->type == MUL_c) {
    lfml = (smtFml_t *) fml->subfmlArr->space[0];
    rfml = (smtFml_t *) fml->subfmlArr->space[1];    
    
    if (lfml->type == NUM_c && rfml->type == NUM_c) {
      mpq_set_str(*tmp_a, (char *) lfml->subfmlArr->space[0], 10);
      mpq_set_str(*tmp_b, (char *) rfml->subfmlArr->space[0], 10);

      mpq_mul(*tmp_b, *tmp_a, *tmp_b);
	
      if(nminus%2 == 1) {
	mpq_mul(*tmp_b, *tmp_b, *coeff);
      } else {
	mpq_neg(*tmp_a, *coeff);
	mpq_mul(*tmp_b, *tmp_a, *tmp_b);
      }

      mpq_add(*mp_constant, *mp_constant, *tmp_b);      
    } else {
      fprintf(stdout, "ERROR: WRONG LINEAR ARITHMETIC ATOM TYPE\n");
      str_a = smt_convert_fml_to_string(lfml);
      str_b = smt_convert_fml_to_string(rfml);
      fprintf(stdout, "fml = %s * %s\n", str_a, str_b);
      fprintf(stdout, "unknown\n");
      exit(0);      
    }

  } else if (fml->type == DIV_c) {
    lfml = (smtFml_t *) fml->subfmlArr->space[0];
    rfml = (smtFml_t *) fml->subfmlArr->space[1];
    
    if (lfml->type == NUM_c && rfml->type == NUM_c) {
      mpq_set_str(*tmp_a, (char *) lfml->subfmlArr->space[0], 10);
      mpq_set_str(*tmp_b, (char *) rfml->subfmlArr->space[0], 10);
      
      mpq_div(*tmp_b, *tmp_a, *tmp_b);
	
      if(nminus%2 == 1) {
	mpq_mul(*tmp_b, *tmp_b, *coeff);
      } else {
	mpq_neg(*tmp_a, *coeff);
	mpq_mul(*tmp_b, *tmp_a, *tmp_b);
      }

      mpq_add(*mp_constant, *mp_constant, *tmp_b);      
    } else {
      fprintf(stdout, "ERROR: WRONG LINEAR ARITHMETIC ATOM TYPE\n");
      str_a = smt_convert_fml_to_string(lfml);
      str_b = smt_convert_fml_to_string(rfml);
      fprintf(stdout, "fml = %s * %s\n", str_a, str_b);
      fprintf(stdout, "unknown\n");
      exit(0);      
    }
  }
  
  return;
}

int
smt_mp_dl_theory_solve(smtManager_t * sm)
{
  smt_mp_bellman_ford_main(sm);

  if (!(sm->flag & SAT_MASK)) { 

    return 2; /* theory unsat */

  }

  if (sm->litArr->size == sm->avarArr->size) {

    smt_mp_check_solution(sm);
    
  }
    
  return 0; /* sat without theory prop */
}

void
smt_mp_bellman_ford_main(smtManager_t * sm)
{
  sm->stats->num_bf_call++;

  smt_mp_generate_constraint_graph(sm);

  smt_mp_bellman_ford_algorithm(sm);
}

void
smt_mp_generate_constraint_graph(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  mpq_t * pool = mp->pool;
  mpq_t * weight = 0, * epsilon = 0;
  smtGraph_t * G;
  smtVertex_t * src, * dest;
  smtEdge_t * e = 0;
  smtAvar_t * avar;
  smtNvar_t * lnvar, *rnvar;
  satArray_t *cur_edges; 
  int cur_index, lit, id;
  int vindex, eindex;
  int i;

  /* mp vars */
  weight = &pool[0];
  epsilon = &pool[1];
  
  if (sm->cG == 0) smt_init_constraint_graph(sm);    

  G = sm->cG;
  eindex = G->esize;
  cur_edges = G->cur_edges;
  cur_edges->size = 0;

  /* generate edges */
  cur_index = sm->cur_index;

  for(i = cur_index; i < sm->litArr->size; i++) {
    lit = sm->litArr->space[i];
    id = (lit>0) ? lit : -lit;
    avar = (smtAvar_t *) sm->id2var->space[id];
    /* filter theory propagated avar */
    if (avar->flag & TPROP_MASK) continue;
    assert(avar->type != EQ_c);
      
    if (sm->avalues[id] == 1) {
      lnvar = (smtNvar_t *) avar->nvars->space[0];
      rnvar = (smtNvar_t *) avar->nvars->space[1];
      mpq_set(*weight, mp->constants[avar->id]);
    } else {
      assert(sm->avalues[id] == 0);
      lnvar = (smtNvar_t *) avar->nvars->space[1];
      rnvar = (smtNvar_t *) avar->nvars->space[0];
      mpq_neg(*weight, mp->constants[e->avar->id]);
      mpq_set_d(*epsilon, sm->epsilon);
      mpq_sub(*weight, *weight, *epsilon);
    }

    vindex = lnvar->id - 1;
    dest = &(G->vHead[vindex]);
    vindex = rnvar->id - 1;
    src =  &(G->vHead[vindex]);

    e = smt_find_edge(src, dest);

    if (e) {
      if ( mpq_cmp(*weight, mp->weights[avar->id]) < 0 ) {
	e->avar->flag |= IMPLIED_MASK;
	e->implied = sat_array_insert(e->implied, (long) e->avar);
	e->avar = avar;
	mpq_set(mp->weights[avar->id], *weight);
	cur_edges = sat_array_insert(cur_edges, (long) e);

      } else {
	avar->flag |= IMPLIED_MASK;
	e->implied = sat_array_insert(e->implied, (long) avar);
      }
    } else {
      e = smt_add_edge(G, src, dest, avar, eindex++);
      mpq_set(mp->weights[avar->id], *weight);
      cur_edges = sat_array_insert(cur_edges, (long) e);
      G->esize++;
    }
  }

  return;
}

void
smt_mp_bellman_ford_algorithm(smtManager_t * sm)      
{
  smtMp_t * mp = sm->mp;
  mpq_t * pool = mp->pool;
  mpq_t * new_dist = 0, * cur_dist = 0, * weight = 0;
  smtGraph_t * G = sm->cG;
  smtVertex_t * v = 0, * src = 0, * dest = 0, *parent = 0;
  smtEdge_t * e;
  smtQueue_t * Q;
  int i, qsize, cycleFound = 0;

  /* mp vars */
  new_dist = &pool[0];
  cur_dist = &pool[1];
  weight = &pool[2];
  
  smt_mp_init_bellman_ford_algorithm(sm);  

  Q = G->queue;

  qsize = Q->size;

  while( (v = (smtVertex_t *) smt_dequeue(Q)) ) {

    G->flags[v->index] &= RESET_FRINGE_MASK;

    if (G->flags[v->index] & VISITED_MASK)
      continue;

    for(i = 0; i < v->outs->size; i++) {
      e = (smtEdge_t *) v->outs->space[i];
      src = e->src;
      dest = e->dest;

      mpq_set(*weight, mp->weights[e->avar->id]);
      mpq_add(*new_dist, mp->values[src->index], *weight);
      mpq_set(*cur_dist, mp->values[dest->index]);

      if ( mpq_cmp(*new_dist, *cur_dist) < 0 ) {
	/* check if src is in the subtree of dest.
	   if this is the case, the negative cycle is detected */
	parent = G->paths[v->index];
	while(1) {
	  if (parent == 0)
            break;
	  else if(parent == dest) {
	    G->paths[dest->index] = v;
	    G->flags[dest->index] &= RESET_VISITED_MASK;
	    cycleFound = 1;
	    break;
	  }
	  parent = G->paths[parent->index];
	}
	if (cycleFound)	break;

	smt_delete_subtree(G, dest);

	/* relaxation */
	mpq_set(mp->values[dest->index], *new_dist);
	
	G->paths[dest->index] = src;
	G->flags[dest->index] &= RESET_VISITED_MASK;
	
        if (!(G->flags[dest->index] & FRINGE_MASK)) {
          G->flags[dest->index] |= FRINGE_MASK; 
          smt_enqueue(Q, (long) dest);
        }
      }
    }

    if(cycleFound)	break;
  }

  if(cycleFound) {

    sm->stats->num_bf_conf++;

    sm->flag &= RESET_SAT_MASK;

    smt_collect_edges_in_neg_cycle(G, dest);

    smt_get_lemma_from_neg_cycle(sm, G->neg_edges);

    smt_mp_retrieve_previous_distance(sm);

  } else if (qsize) {
    
    smt_update_value_with_current_distance(sm);

  }

  return;
}

void
smt_mp_init_bellman_ford_algorithm(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  mpq_t * pool = mp->pool;
  mpq_t * new_dist = 0, * cur_dist = 0, * weight = 0;
  smtGraph_t * G = sm->cG;
  smtVertex_t * src, * dest;
  smtEdge_t * e;
  smtQueue_t * Q;
  int i;

  /* mp vars */
  new_dist = &pool[0];
  cur_dist = &pool[1];
  weight = &pool[2];
  
  memset(G->flags, 0, sizeof(int) * G->vsize);
  memset(G->paths, 0, sizeof(smtVertex_t *) * G->vsize);

  Q = G->queue;
  smt_init_queue(Q);

  for(i = 0; i < G->cur_edges->size; i++) {
    e = (smtEdge_t *) G->cur_edges->space[i];
    src = e->src;
    dest = e->dest;

    mpq_set(*weight, mp->weights[e->avar->id]);
    mpq_add(*new_dist, mp->values[src->index], *weight);
    mpq_set(*cur_dist, mp->values[dest->index]);

    if ( mpq_cmp(*new_dist, *cur_dist) < 0 ) {
      if(!(G->flags[src->index] & FRINGE_MASK)) {
	G->flags[src->index] |= FRINGE_MASK; /* not in queue */
	smt_enqueue(Q, (long) src);
      }
    }
  }

  return;
}

void
smt_mp_retrieve_previous_distance(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  int i, size;

  size = sm->nvarArr->size + 1;

  for(i = 0; i < size; i++) {
    mpq_set(mp->pvalues[i], mp->values[i]);
  }

  return;
}

void
smt_mp_print_atom_constants(smtManager_t * sm)
{
  smtMp_t * mp = sm->mp;
  smtAvar_t * avar;
  int i, id;

  for(i = 0; i < sm->avarArr->size; i++) {
    avar = (smtAvar_t *) sm->avarArr->space[i];
    id = avar->id;
    fprintf(stdout, "%s\n", avar->name);
    gmp_printf("constant:%#Qd\n", mp->constants[id]);
  }
}

void
smt_mp_print_value(mpq_t * value)
{
  gmp_printf("value:%#Qd\n", *value);  
}
#endif
