/**CFile***********************************************************************

  FileName    [AigCmd.c]

  PackageName [Aig]

  Synopsis    [Functions to initialize and shut down the Aig manager.]

  Author      [Mohammad Awedh]

  Copyright [ This file was created at the University of Colorado at
  Boulder.  The University of Colorado at Boulder makes no warranty
  about the suitability of this software for any purpose.  It is
  presented on an AS IS basis.]

******************************************************************************/

#include "aig.h"
#include "aigInt.h"

static char rcsid[] UNUSED = "$Id: aigCmd.c,v 1.1.1.1 2008-11-14 20:40:11 hhkim Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Creates a new Aig manager.]

  Description [Creates a new Aig manager, Create a NodeArray of size equal to
  maxSize * AigNodeSize. Returns a pointer to the manager if successful; 
  NULL otherwise.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
Aig_Manager_t *
Aig_initAig(
  int maxSize /* maximum number of nodes in the AIG graph*/
)
{
  int i;
  
  Aig_Manager_t *bm = ALLOC(Aig_Manager_t,1);
  
  maxSize = maxSize * AigNodeSize;  /* each node is a size of AigNodeSize */
  if (bm == NIL( Aig_Manager_t)){
    return bm;
  }
  
  maxSize = (maxSize/AigNodeSize)*AigNodeSize;
  if (maxSize >  AigArrayMaxSize ) {
  }
  bm->NodesArray = ALLOC(AigEdge_t, maxSize);
  bm->maxNodesArraySize = maxSize;
  bm->nodesArraySize = AigFirstNodeIndex;
  
  fanout(0) = 0;
  canonical(0) = 0;
  flags(0) = 0;
  aig_value(0) = 2;
  aig_next(0) = Aig_NULL;
  
  bm->SymbolTable = st_init_table(strcmp,st_strhash);
  bm->nameList = ALLOC(char *, bm->maxNodesArraySize/AigNodeSize);
  bm->HashTable =  ALLOC(AigEdge_t, Aig_HashTableSize);
  
  for (i=0; i<Aig_HashTableSize; i++)
    bm->HashTable[i]= Aig_NULL; 
  
  bm->timeframe = 0;
  bm->timeframeWOI = 0;
  
  return(bm);

} /* end of Aig_Init */

/**Function********************************************************************

  Synopsis [Quit Aig manager]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Aig_quit(
  Aig_Manager_t *bm)
{
  int           i;
  char          *name;
  st_generator  *stGen;
  AigEdge_t    varIndex;

  /**
  if (bm->mVarList != NIL(array_t)){
    array_free(bm->mVarList);
  }
  if (bm->bVarList != NIL(array_t)){
    array_free(bm->bVarList);
  }
  **/

  FREE(bm->HashTable);
  st_foreach_item(bm->SymbolTable, stGen, &name, &varIndex) {
    FREE(name);
  }
  st_free_table(bm->SymbolTable);
  /* i is too small to represent 80000
  for (i=0; i< bm->maxNodesArraySize/AigNodeSize ; i++){
    FREE(bm->nameList[i]);
  }
  */
  for (i=AigFirstNodeIndex ; i< bm->nodesArraySize ; i+=AigNodeSize){
    if (fanout(i)) free((AigEdge_t *) fanout(i));
  }

  if (bm->nameList) FREE(bm->nameList);
  if (bm->NodesArray) FREE(bm->NodesArray);

  bm->nameList = 0;
  bm->NodesArray = 0;
  bm->timeframe = 0;
  bm->timeframeWOI = 0;

  if (bm) {
    FREE(bm);
  }

  bm = 0;
}

/**Function********************************************************************

  Synopsis    []

  Description []

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
void 
Aig_NodePrint(
   Aig_Manager_t  *bm,
   AigEdge_t node)
{

  if (node == Aig_Zero){
    printf("0");
    return;
  }
  if (node == Aig_One){
    printf("1");
    return;
  }  
  
  if (Aig_IsInverted(node)){
    printf(" NOT(");
  }
  
  if ( rightChild(node) == Aig_NULL){
    printf("Var Node");
    if (Aig_IsInverted(node)){
      printf(")");
    }
    return;
  }
  printf("(");
  Aig_NodePrint(bm, leftChild(node));
  printf("AND");
  Aig_NodePrint(bm, rightChild(node));
  printf(")");
  if (Aig_IsInverted(node)){
    printf(")");
  }
}
