/**CHeaderFile*****************************************************************

  FileName    [hrc.h]

  PackageName [hrc]

  Synopsis    [Hierarchical representation of a design.]

  Description [The concept of a hierarchy is directly related to the
  constructs in a blifmv file. It is built around three important data
  structures viz. Hrc_Model_t, Hrc_Subckt_t and Hrc_Node_t. These will be
  referred to as model, subckt and node.<p>

  A model corresponds directly to a model definition in a blifmv file. It can
  be viewed as a black box with some I/O pins. A model could call another
  model as a module within itself using names for the I/O pins which are, in
  general, different from those actually used inside the model that is
  called. The correspondence between the two sets of names is stored in a
  subckt. A call to a model will henceforth be referred to as an instantiation
  of the model.<p>
               
  Since a given model could be called by many other models, it is necessary to
  distinguish between its different instantiations. An instantiation is
  represented by a node.  A hierarchy is a tree of nodes. The root node of the
  tree corresponds to the single instantiation of the root model in a blifmv
  file. An instantiation of a model results in the instantiation of all models
  recursively called by it. Thus, the instantiation of root model results in a
  tree structure being formed.<p>
               
  There is also a structure called a hierarchy manager which contains a list
  of all models and pointers to the root node and the current node. The
  current node represents the current position in the hierarchy. The designer
  can, if he wishes, make changes in this node only, or modify the whole
  sub-tree below it.]

  SeeAlso     [hrcInt.h]

  Author      [Yuji Kukimoto, Shaz Qadeer]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: hrc.h,v 1.5 2005/04/16 04:23:47 fabio Exp $]

******************************************************************************/

#ifndef _HRC
#define _HRC

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

typedef struct HrcModelStruct Hrc_Model_t;
typedef struct HrcManagerStruct Hrc_Manager_t;
typedef struct HrcNodeStruct Hrc_Node_t;
typedef struct HrcSubcktStruct Hrc_Subckt_t;
typedef struct HrcLatchStruct Hrc_Latch_t;
typedef void (*Hrc_ApplInfoFreeFn) (void *);
typedef void (*Hrc_ApplInfoChangeFn) (Hrc_Node_t *, Hrc_Node_t *, void **);


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**Macro***********************************************************************

  Synopsis     [Iterates over the models in the hierarchy manager.]

  Description  [This macro iterates over the models in the hierarchy manager.
                It is an error to modify the models while iterating over them.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define Hrc_ManagerForEachModel(                                    \
  /* Hrc_Manager_t * */ manager   /* manager to iterate models */,  \
  /* st_generator * */  gen       /* local variable for iterator */,\
  /* char * */          modelName /* name of model */,              \
  /* Hrc_Model_t * */   model     /* model * */                     \
)                                                                   \
  st_foreach_item(Hrc_ManagerReadModelTable(manager), gen, &(modelName), &(model))
    
/**Macro***********************************************************************

  Synopsis     [Iterates over the formal inputs of a node.]

  Description  [This macro iterates over the formal inputs of a node. It is an
                error to modify the formal inputs while iterating over them.]

  SideEffects  [This macro instantiates macros from the array package.  Hence
                it is advisable not to nest this macro within array macros.]

  SeeAlso      [Hrc_NodeForEachFormalOutput()]

******************************************************************************/
#define Hrc_NodeForEachFormalInput(                             \
  /* Hrc_Node_t * */     node /* node to iterate inputs */,     \
  /* int */              i    /* local variable for iterator */,\
  /* Var_Variable_t * */ var  /* input variable of node */      \
)                                                               \
  arrayForEachItem(Var_Variable_t *, Hrc_NodeReadFormalInputs(node), i, var)

/**Macro***********************************************************************

  Synopsis     [Iterates over the formal outputs of a node.]

  Description  [This macro iterates over the formal outputs of a node. It is an
                error to modify the formal outputs while iterating over them.]

  SideEffects  [This macro instantiates macros from the array package.  Hence
                it is advisable not to nest this macro within array macros.]

  SeeAlso      [Hrc_NodeForEachFormalInput()]

******************************************************************************/
#define Hrc_NodeForEachFormalOutput(                            \
  /* Hrc_Node_t * */     node /* node to iterate outputs */,    \
  /* int */              i    /* local variable for iterator */,\
  /* Var_Variable_t * */ var  /* output variable of node */     \
)                                                               \
  arrayForEachItem(Var_Variable_t *, Hrc_NodeReadFormalOutputs(node), i, var)

/**Macro***********************************************************************

  Synopsis     [Iterates over the actual inputs of a model.]

  Description  [This macro iterates over the actual inputs of a node. It is an
                error to modify the actual inputs while iterating over them.]

  SideEffects  [This macro instantiates macros from the array package.  Hence
                it is advisable not to nest this macro within array macros.]

  SeeAlso      [Hrc_NodeForEachActualOutput()]

******************************************************************************/
#define Hrc_NodeForEachActualInput(                               \
  /* Hrc_Node_t * */     node /* node to iterate actual inputs */,\
  /* int */              i    /* local variable for iterator */,  \
  /* Var_Variable_t * */ var  /* actual input variable of node */ \
)                                                                 \
  arrayForEachItem(Var_Variable_t *, Hrc_NodeReadActualInputs(node), i, var)

/**Macro***********************************************************************

  Synopsis     [Iterates over the actual outputs of a node.]

  Description  [This macro iterates over the actual outputs of a node. It is an
                error to modify the actual outputs while iterating over them.]

  SideEffects  [This macro instantiates macros from the array package.  Hence
                it is advisable not to nest this macro within array macros.]

  SeeAlso      [Hrc_NodeForEachActualInput()]

******************************************************************************/
#define Hrc_NodeForEachActualOutput(                               \
  /* Hrc_Node_t * */     node /* node to iterate actual outputs */,\
  /* int */              i    /* local variable for iterator */,   \
  /* Var_Variable_t * */ var  /* actual output variable of node */ \
)                                                                  \
  arrayForEachItem(Var_Variable_t *, Hrc_NodeReadActualOutputs(node), i, var)

/**Macro***********************************************************************

  Synopsis     [Iterates over the name tables of a node.]

  Description  [This macro iterates over the name tables of a node. It is an
                error to modify the name tables while iterating over them.]

  SideEffects  [This macro instantiates macros from the array package.  Hence
                it is advisable not to nest this macro within array macros.]

  SeeAlso      [Hrc_NodeForEachLatch()]

******************************************************************************/
#define Hrc_NodeForEachNameTable(                             \
  /* Hrc_Node_t * */  node  /* node to iterate tables */,     \
  /* int */           i     /* local variable for iterator */,\
  /* Tbl_Table_t * */ table /* table of node */               \
)                                                             \
  arrayForEachItem(Tbl_Table_t *, Hrc_NodeReadNameTables(node), i, table)

/**Macro***********************************************************************

  Synopsis     [Iterates over the subckts of a model.]

  Description  [This macro iterates over the sub-circuits of a node. It is an
                error to modify the sub-circuits while iterating over them.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define Hrc_ModelForEachSubckt(                                  \
  /* Hrc_Model_t */    model  /* model to iterate subcircuits */,\
  /* st_generator * */ gen    /* local variable for iterator */, \
  /* char * */         name   /* instance name of subcircuit */, \
  /* Hrc_Subckt_t * */ subckt /* subcircuit */                   \
)                                                                \
  st_foreach_item(Hrc_ModelReadSubcktTable(model), gen, &(name), &(subckt))
    
/**Macro***********************************************************************

  Synopsis     [Iterates over the children of a node.]

  Description  [This macro iterates over the children of a node. It is an
                error to modify the children while iterating over them.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define Hrc_NodeForEachChild(                                        \
  /* Hrc_Node_t * */   node      /* node to iterate child nodes */, \
  /* st_generator * */ gen       /* local variable for iterator */, \
  /* char * */         childName /* name of child */,               \
  /* Hrc_Node_t * */   child     /* child */                        \
)                                                                    \
  st_foreach_item(Hrc_NodeReadChildTable(node), gen, &(childName), &(child))
    

/**Macro***********************************************************************

  Synopsis     [Iterates over the latches of a node.]

  Description  [This macro iterates over the latches of a node. It is an
                error to modify the latches while iterating over them.]

  SideEffects  []

  SeeAlso      [Hrc_NodeForEachNameTable()]

******************************************************************************/
#define Hrc_NodeForEachLatch(                                      \
  /* Hrc_Node_t * */   node      /* node to iterate latches */,    \
  /* st_generator * */ gen       /* local variable for iterator */,\
  /* char * */         latchName /* name of latch */,              \
  /* Hrc_Latch_t * */  latch     /* latch */                       \
)                                                                  \
  st_foreach_item(Hrc_NodeReadLatchTable(node), gen, &(latchName), &(latch))
    
/**Macro***********************************************************************

  Synopsis     [Iterates over the formal variables of a node.]

  Description  [This macro iterates over the formal variables of a node. It
                is an error to modify the formal variables while iterating
                over them.]

  SideEffects  []

  SeeAlso      [Hrc_NodeForEachFormalInput(), Hrc_NodeForEachFormalOuput()]

******************************************************************************/
#define Hrc_NodeForEachVariable(                                   \
  /* Hrc_Node_t * */     node    /* node to iterate variables */,  \
  /* st_generator * */   gen     /* local variable for iterator */,\
  /* char * */           varName /* name of variable */,           \
  /* Var_Variable_t * */ var     /* variable */                    \
)                                                                  \
  st_foreach_item(Hrc_NodeReadVariableTable(node), gen, &(varName), &var)
    
#include "tbl.h"

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN void Hrc_Init(void);
EXTERN void Hrc_End(void);
EXTERN Hrc_Node_t * Hrc_ModelCreateHierarchy(Hrc_Manager_t *hmgr, Hrc_Model_t *model, char *instanceName);
EXTERN boolean Hrc_NodeAddApplInfo(Hrc_Node_t *node, char * key, Hrc_ApplInfoFreeFn freeFn, Hrc_ApplInfoChangeFn changeFn, void * data);
EXTERN void * Hrc_NodeReadApplInfo(Hrc_Node_t *node, char * key);
EXTERN boolean Hrc_NodeFreeApplInfo(Hrc_Node_t *node, char * key);
EXTERN void Hrc_ModelWriteBlifMv(FILE *fp, Hrc_Model_t *model, boolean isRootModel, char *rootInstanceName);
EXTERN Hrc_Node_t * Hrc_ManagerReadRootNode(Hrc_Manager_t *manager);
EXTERN Hrc_Node_t * Hrc_ManagerReadCurrentNode(Hrc_Manager_t *manager);
EXTERN st_table * Hrc_ManagerReadModelTable(Hrc_Manager_t *manager);
EXTERN Hrc_Model_t * Hrc_ManagerFindModelByName(Hrc_Manager_t *manager, char *modelName);
EXTERN Hrc_Node_t * Hrc_ModelReadMasterNode(Hrc_Model_t *model);
EXTERN char * Hrc_ModelReadName(Hrc_Model_t *model);
EXTERN st_table * Hrc_ModelReadSubcktTable(Hrc_Model_t *model);
EXTERN Hrc_Model_t * Hrc_SubcktReadModel(Hrc_Subckt_t *subckt);
EXTERN char * Hrc_SubcktReadInstanceName(Hrc_Subckt_t *subckt);
EXTERN array_t * Hrc_SubcktReadActualInputVars(Hrc_Subckt_t *subckt);
EXTERN array_t * Hrc_SubcktReadActualOutputVars(Hrc_Subckt_t *subckt);
EXTERN Hrc_Manager_t * Hrc_NodeReadManager(Hrc_Node_t *node);
EXTERN char * Hrc_NodeReadModelName(Hrc_Node_t *node);
EXTERN char * Hrc_NodeReadInstanceName(Hrc_Node_t *node);
EXTERN Hrc_Node_t * Hrc_NodeReadParentNode(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumFormalInputs(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumFormalOutputs(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumVariables(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumTables(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumLatches(Hrc_Node_t *node);
EXTERN int Hrc_NodeReadNumChildren(Hrc_Node_t *node);
EXTERN array_t * Hrc_NodeReadFormalInputs(Hrc_Node_t *node);
EXTERN array_t * Hrc_NodeReadFormalOutputs(Hrc_Node_t *node);
EXTERN array_t * Hrc_NodeReadActualInputs(Hrc_Node_t *node);
EXTERN array_t * Hrc_NodeReadActualOutputs(Hrc_Node_t *node);
EXTERN array_t * Hrc_NodeReadNameTables(Hrc_Node_t *node);
EXTERN st_table * Hrc_NodeReadChildTable(Hrc_Node_t *node);
EXTERN st_table * Hrc_NodeReadLatchTable(Hrc_Node_t *node);
EXTERN st_table * Hrc_NodeReadVariableTable(Hrc_Node_t *node);
EXTERN void * Hrc_NodeReadUndef(Hrc_Node_t *node);
EXTERN Hrc_Latch_t * Hrc_NodeFindLatchByName(Hrc_Node_t *node, char *latchName);
EXTERN Var_Variable_t * Hrc_NodeFindVariableByName(Hrc_Node_t *node, char *varName);
EXTERN Hrc_Node_t * Hrc_NodeFindChildByName(Hrc_Node_t *node, char *instanceName);
EXTERN Var_Variable_t * Hrc_LatchReadInput(Hrc_Latch_t *latch);
EXTERN Var_Variable_t * Hrc_LatchReadOutput(Hrc_Latch_t *latch);
EXTERN Tbl_Table_t * Hrc_LatchReadResetTable(Hrc_Latch_t *latch);
EXTERN void * Hrc_LatchReadUndef(Hrc_Latch_t *latch);
EXTERN void Hrc_ManagerSetRootNode(Hrc_Manager_t *manager, Hrc_Node_t *node);
EXTERN void Hrc_ManagerSetCurrentNode(Hrc_Manager_t *manager, Hrc_Node_t *currentNode);
EXTERN boolean Hrc_NodeAddFormalInput(Hrc_Node_t *node, Var_Variable_t *var);
EXTERN boolean Hrc_NodeAddFormalOutput(Hrc_Node_t *node, Var_Variable_t *var);
EXTERN void Hrc_NodeAddNameTable(Hrc_Node_t *node, Tbl_Table_t *nameTable);
EXTERN void Hrc_NodeSetUndef(Hrc_Node_t *node, void *value);
EXTERN boolean Hrc_LatchSetResetTable(Hrc_Latch_t *latch, Tbl_Table_t *table);
EXTERN void Hrc_LatchSetUndef(Hrc_Latch_t *latch, void *value);
EXTERN Hrc_Manager_t * Hrc_ManagerAlloc(void);
EXTERN Hrc_Model_t * Hrc_ModelAlloc(Hrc_Manager_t *manager, char *modelName);
EXTERN Hrc_Node_t * Hrc_NodeAlloc(Hrc_Manager_t *manager, char *modelName, char *instanceName, Hrc_Node_t *parentNode);
EXTERN Hrc_Node_t * Hrc_NodeDup(Hrc_Node_t *node, char *instanceName);
EXTERN Hrc_Latch_t * Hrc_LatchCreate(Hrc_Model_t *model, Var_Variable_t *latchInput, Var_Variable_t *latchOutput);
EXTERN void Hrc_ManagerFree(Hrc_Manager_t *manager);
EXTERN boolean Hrc_ModelDelete(Hrc_Manager_t *manager, char *modelName);
EXTERN Hrc_Node_t * Hrc_ManagerFindNodeByPathName(Hrc_Manager_t *manager, char *path, boolean pathFlag);
EXTERN Var_Variable_t * Hrc_VariableFindActualFromFormal(Hrc_Node_t *node, Var_Variable_t *formalVar, Hrc_Node_t *referenceNode);
EXTERN array_t * Hrc_ManagerObtainComponentModels(Hrc_Manager_t *manager);
EXTERN char * Hrc_NodeFindHierarchicalName(Hrc_Node_t *node, boolean pathFlag);
EXTERN boolean Hrc_TreeReplace(Hrc_Node_t *oldNode, Hrc_Node_t *newNode);
EXTERN boolean Hrc_NodeTestIsInTree(Hrc_Node_t *node1, Hrc_Node_t *node2);
EXTERN boolean Hrc_NodeTestIsUninterpreted(Hrc_Node_t *node);
EXTERN boolean Hrc_ModelTestIsUninterpreted(Hrc_Model_t* model);
EXTERN boolean Hrc_NodeTestRecursivelyIsUninterpreted(Hrc_Node_t* parent);
EXTERN boolean Hrc_NodeTestIsUninterpretedNodeInHierarchy(Hrc_Manager_t* hmgr);
EXTERN boolean Hrc_NodeCheckVariableConsistency(Hrc_Node_t *node);
EXTERN boolean Hrc_ModelAddSubckt(Hrc_Model_t *callerModel, Hrc_Model_t *calleeModel, char *instanceName, array_t *actualInputVars, array_t *actualOutputVars);
EXTERN void Hrc_ModelAddNameTable(Hrc_Model_t *model, Tbl_Table_t *table);
EXTERN boolean Hrc_NodeAddChild(Hrc_Node_t *parent, Hrc_Node_t *child, array_t *actualInputs, array_t *actualOutputs);
EXTERN boolean Hrc_NodeAddLatch(Hrc_Node_t *node, Hrc_Latch_t *latch);
EXTERN boolean Hrc_NodeAddVariable(Hrc_Node_t *node, Var_Variable_t *var);
EXTERN boolean Hrc_NodeDeleteVariable(Hrc_Node_t *node, Var_Variable_t *var);
EXTERN boolean Hrc_NodeDeleteLatch(Hrc_Node_t *node, Hrc_Latch_t *latch);
EXTERN void Hrc_ModelWriteSmv(FILE *fp, Hrc_Model_t *model, boolean isRootModel, char *rootInstanceName);

/**AutomaticEnd***************************************************************/

#endif /* _HRC */
