/**CFile***********************************************************************

  FileName    [hrcHierarchy.c]

  PackageName [hrc]

  Synopsis    [Creates a hierarchy from model definitions.]

  Description []

  SeeAlso     []

  Author      [Yuji Kukimoto]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "hrcInt.h"

static char rcsid[] UNUSED = "$Id: hrcHierarchy.c,v 1.4 2005/04/16 04:23:47 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static Hrc_Node_t * _HrcCreateHierarchyAux(Hrc_Manager_t *hmgr, Hrc_Model_t *model, char *instanceName, st_table *pathTable);
static int _HrcConnectParentChild(Hrc_Node_t *parent, Hrc_Node_t *child, Hrc_Subckt_t *subckt);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Creates a hierarchy from model definitinos.]

  Description [Creates a hierarchy from model definitinos. The second
  argument is a pointer to the root model. The last argument is the name
  of the root node. A hash table pathTable is used to check to see if
  there is no cyclic model definition.]

  SideEffects []

  SeeAlso     [_HrcCreateHierarchyAux]

******************************************************************************/

Hrc_Node_t *
Hrc_ModelCreateHierarchy(
  Hrc_Manager_t *hmgr,
  Hrc_Model_t *model,
  char *instanceName)
{
  st_table *pathTable = st_init_table(st_ptrcmp,st_ptrhash);
  Hrc_Node_t *rootNode = _HrcCreateHierarchyAux(hmgr,model,instanceName,pathTable);
  st_free_table(pathTable);
  return rootNode;
}
    

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [Recursively creates a hierarchy from model definitinos.]

  Description []

  SideEffects [pathTable is updated to keep track of the models used
  from the root to the current node.]

  SeeAlso     [Hrc_CreateHierarchy]

******************************************************************************/

static Hrc_Node_t *
_HrcCreateHierarchyAux(
  Hrc_Manager_t *hmgr,
  Hrc_Model_t *model,
  char *instanceName,
  st_table *pathTable)
{
  Hrc_Node_t *rootNode, *childNode;
  Hrc_Model_t *modelToBeInstantiated;
  Hrc_Subckt_t *subckt;
  st_generator *gen;

  if (st_is_member(pathTable,(char *)model) == 1){
    error_append("Model ");
    error_append(Hrc_ModelReadName(model));
    error_append(" is defined by using itself.\n");
    return NIL(Hrc_Node_t);
  }
  rootNode = Hrc_NodeDup(Hrc_ModelReadMasterNode(model),instanceName);
  (void)st_insert(pathTable,model,NULL);

  Hrc_ModelForEachSubckt(model,gen,instanceName,subckt){
    modelToBeInstantiated = Hrc_SubcktReadModel(subckt);
    childNode = _HrcCreateHierarchyAux(hmgr,modelToBeInstantiated,Hrc_SubcktReadInstanceName(subckt),pathTable);
    if (childNode == NIL(Hrc_Node_t)){
      return NIL(Hrc_Node_t);
    }
    if (_HrcConnectParentChild(rootNode,childNode,subckt) == 0){
      return NIL(Hrc_Node_t);
    }
  }
  (void)st_delete(pathTable,&model,NULL);
  return rootNode;
}


/**Function********************************************************************

  Synopsis    [Connects a parent node and a child node using Hrc_Subckt_t
  information.]

  Description []

  SideEffects [The parent node and the child node are connected together
  in the hierarchy.]

  SeeAlso     [Hrc_NodeAddChild]

******************************************************************************/
static int
_HrcConnectParentChild(
  Hrc_Node_t *parent, 
  Hrc_Node_t *child,
  Hrc_Subckt_t *subckt)
{
  array_t *actualInputVars, *actualOutputVars;
  array_t *realActualInputVars, *realActualOutputVars;
  int i;
  Var_Variable_t *var, *realVar;

  /* actual inputs and outputs in the master hnode corresponsing to the parent */
  actualInputVars = Hrc_SubcktReadActualInputVars(subckt);
  actualOutputVars = Hrc_SubcktReadActualOutputVars(subckt);

  /* collect all the actual inputs in the parent */
  realActualInputVars = array_alloc(Var_Variable_t *,0);
  for (i=0; i < array_n(actualInputVars); i++){
    var = array_fetch(Var_Variable_t *,actualInputVars,i);
    realVar = Hrc_NodeFindVariableByName(parent,Var_VariableReadName(var)); 
    array_insert_last(Var_Variable_t *,realActualInputVars,realVar);
  }
  /* collect all the actual outputs in the parent */
  realActualOutputVars = array_alloc(Var_Variable_t *,0);
  for (i=0; i < array_n(actualOutputVars); i++){
    var = array_fetch(Var_Variable_t *,actualOutputVars,i);
    realVar = Hrc_NodeFindVariableByName(parent,Var_VariableReadName(var)); 
    array_insert_last(Var_Variable_t *,realActualOutputVars,realVar);
  }
  /* connects the parent and the child using an hrc function */
  if (Hrc_NodeAddChild(parent,child,realActualInputVars,realActualOutputVars) == 0){
     return 0;
  }
  return 1;
}
  















