/**CFile***********************************************************************

  FileName    [hrcInOut.c]

  PackageName [hrc]

  Synopsis    [This file provides the functions for accessing the fields
               of the data structures of the hrc package.]

  SeeAlso     []

  Author      [Shaz Qadeer]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "hrcInt.h"
#include "io.h"

static char rcsid[] UNUSED = "$Id: hrcInOut.c,v 1.6 2005/04/28 14:16:09 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [Associates application specific data with a node.]

  Description [Adds the key/data pair to the node's applInfoTable. Key may
  be an arbitrary string; to avoid possible conflicts with other applications,
  key should be prefixed with the application's package name, for example
  "Pkg_HrcNodeApplKey".  A copy is made of the string key before it is
  inserted into applInfoTable. The contents of data are arbitrary; the
  hierarchy package does not interpret the data. The freeFn is called on data
  when a node is freed, or when Hrc_NodeFreeApplInfo() is called with the same
  value of key. The freeFn takes (void *) as its only argument, and returns
  void. The changeFn is called when a subtree in the hierarchy rooted at
  oldNode is replaced by another tree rooted at newNode. It takes three
  arguments - oldNode, newNode and &(applInfo->data), and returns void. Note
  the order of the three arguments. <p>

  If this function is called with a value of key that already exists in the
  applInfoTable then the data for the previous entry is freed using the
  freeFn, its fields are overwritten by the new ones passed as arguments to
  the function and TRUE is returned. Otherwise, FALSE is returned. 

  <p>
  WARNING:  This function does not make a copy of data before inserting it
  into the table.  By calling this function, the application is surrendering
  responsibility to the hierarchy to free the data.  Once this function is
  called on data, the data should be freed only by calling
  Hrc_NodeFreeApplInfo(), Hrc_ManagerFree() or Hrc_TreeReplace().  In other
  words, it is an error for the application to explicitly free the data
  without using one of these functions. </p>]

  SideEffects []

  SeeAlso     [Hrc_NodeFreeApplInfo() Hrc_NodeReadApplInfo()]

******************************************************************************/
boolean
Hrc_NodeAddApplInfo(
  Hrc_Node_t *node,
  char * key,
  Hrc_ApplInfoFreeFn  freeFn,
  Hrc_ApplInfoChangeFn changeFn,
  void * data)
{
  ApplInfo_t *applInfo;
  boolean status;
  
  if (st_lookup(node->applInfoTable, key, &applInfo)) {
    (*applInfo->freeFn)(applInfo->data);
    status = TRUE;
  }
  else {
    char       *keyCopy  = util_strsav(key);

    applInfo = ALLOC(ApplInfo_t, 1);
    st_insert(node->applInfoTable, keyCopy, applInfo);
    status = FALSE;
  }
  applInfo->freeFn = freeFn;
  applInfo->changeFn = changeFn;
  applInfo->data   = data;
  return status;
}


/**Function********************************************************************

  Synopsis    [Returns data corresponding to key if it exists, else returns NULL.]

  Description [If data corresponding to key is associated with the node, then
  returns a pointer to data (as void *).  Otherwise, returns NULL.]

  SideEffects []

  SeeAlso     [Hrc_NodeAddApplInfo() Hrc_NodeFreeApplInfo()]

******************************************************************************/
void *
Hrc_NodeReadApplInfo(
  Hrc_Node_t *node,
  char * key)
{
  int         status;
  ApplInfo_t *applInfo;
  
  status = st_lookup(node->applInfoTable, key, &applInfo);
  if (status == 1) {
    return (applInfo->data);
  }
  else {
    return (NIL(void));
  }
}


/**Function********************************************************************

  Synopsis    [Frees application-specific data from a node.]

  Description [If data corresponding to key exists in the node, then frees
  the data using the associated free function, removes the key/data pair from
  the network's applInfoTable, and returns TRUE.  If key does not exist, then
  the function does nothing and returns FALSE.
  <p>
  WARNING: This function should be called only if a corresponding call to
  Hrc_NodeAddApplInfo() has already been made.</p>]

  SideEffects []

  SeeAlso     [Hrc_NodeAddApplInfo() Hrc_NodeReadApplInfo()]

******************************************************************************/
boolean
Hrc_NodeFreeApplInfo(
  Hrc_Node_t *node,
  char * key)
{
  int         status;
  ApplInfo_t *applInfo;
  
  status = st_lookup(node->applInfoTable, key, &applInfo);

  if (status == 1) {
    st_delete(node->applInfoTable, &key, &applInfo);
    (*applInfo->freeFn)(applInfo->data);
    FREE(key);  /* frees the string contained in the table */
    FREE(applInfo);
  }

  return ((status) ? TRUE : FALSE);
}

/**Function********************************************************************

  Synopsis    [Writes out a model definition.]

  Description [Writes out a model definition. The second argument is a pointer
  to a model to be written out. If isRootModel is TRUE, the function prints
  out .root to specify that the model is a root model. The last argument
  specifies the instance name of the root model and is used only if isRootModel
  is set to TRUE.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_ModelWriteBlifMv(
  FILE *fp,
  Hrc_Model_t *model,
  boolean isRootModel,
  char *rootInstanceName)
{
  Hrc_Node_t *hnode, *subcktHnode;
  Hrc_Model_t *subcktModel;
  int i, is_enum, range;
  Var_Variable_t *var, *actualVar;
  Hrc_Latch_t *latch;
  Hrc_Subckt_t *subckt;
  Tbl_Table_t *table;
  char *varName, *latchName, *instanceName;
  st_generator *gen;
  array_t *subcktActualInputVars, *subcktActualOutputVars;

  (void)fprintf(fp,".model %s\n",Hrc_ModelReadName(model));
  if (isRootModel){
    (void)fprintf(fp,".root %s\n",rootInstanceName);
  }
  hnode = Hrc_ModelReadMasterNode(model);

  /* .inputs */
  if (Hrc_NodeReadNumFormalInputs(hnode) != 0){
    (void)fprintf(fp,".inputs ");
    Hrc_NodeForEachFormalInput(hnode,i,var){
      (void)fprintf(fp,"%s ",Var_VariableReadName(var));
    }
    (void)fprintf(fp,"\n");
  }

  /* .outputs */
  if (Hrc_NodeReadNumFormalOutputs(hnode) != 0){
    (void)fprintf(fp,".outputs ");
    Hrc_NodeForEachFormalOutput(hnode,i,var){
      (void)fprintf(fp,"%s ",Var_VariableReadName(var));
    }
    (void)fprintf(fp,"\n");
  }

  /* .mv */
  Hrc_NodeForEachVariable(hnode,gen,varName,var){
    is_enum = Var_VariableTestIsEnumerative(var);
    range = Var_VariableReadNumValues(var);
    
    if (is_enum == 1){
      if (range == 2){
        /* Boolean enumerative variables need no .mv declaration. */
        continue;
      }
      else {
        (void)fprintf(fp,".mv %s %d\n",Var_VariableReadName(var),range);
      }
    }
    else {
      /* variable var is symbolic */
      (void)fprintf(fp,".mv %s %d ",Var_VariableReadName(var),range);
      for (i=0; i < range; i++){
        (void)fprintf(fp,"%s ",Var_VariableReadSymbolicValueFromIndex(var,i));
      }
      (void)fprintf(fp,"\n");
    }
  }

  /* .latches */
  Hrc_NodeForEachLatch(hnode,gen,latchName,latch){
    (void)fprintf(fp,".latch %s %s\n",
       Var_VariableReadName(Hrc_LatchReadInput(latch)),
       Var_VariableReadName(Hrc_LatchReadOutput(latch)));
    Tbl_TableWriteBlifMvToFile(Hrc_LatchReadResetTable(latch),1,fp);
  }

  /* .subckt */
  Hrc_ModelForEachSubckt(model,gen,instanceName,subckt){
    subcktModel = Hrc_SubcktReadModel(subckt);
    subcktHnode = Hrc_ModelReadMasterNode(subcktModel);
    subcktActualInputVars = Hrc_SubcktReadActualInputVars(subckt);
    subcktActualOutputVars = Hrc_SubcktReadActualOutputVars(subckt);

    (void)fprintf(fp,".subckt %s %s ",
       Hrc_ModelReadName(subcktModel), Hrc_SubcktReadInstanceName(subckt));
    Hrc_NodeForEachFormalInput(subcktHnode,i,var){
      actualVar = array_fetch(Var_Variable_t *,subcktActualInputVars,i);
      (void)fprintf(fp,"%s = %s ",Var_VariableReadName(var),Var_VariableReadName(actualVar)); 
    }   
    Hrc_NodeForEachFormalOutput(subcktHnode,i,var){
      actualVar = array_fetch(Var_Variable_t *,subcktActualOutputVars,i);
      (void)fprintf(fp,"%s = %s ",Var_VariableReadName(var),Var_VariableReadName(actualVar)); 
    }   
    (void)fprintf(fp,"\n");
  }

  /* .names */
  Hrc_NodeForEachNameTable(hnode,i,table){
    Tbl_TableWriteBlifMvToFile(table,0,fp);
  }

  (void)fprintf(fp,".end\n");
}


/**Function********************************************************************

  Synopsis    [Writes out a model definition in NuSMV format.]

  Description [Writes out a model definition. The second argument is a
  pointer to a model to be written out. If isRootModel is TRUE, the
  function prints out .root to specify that the model is a root
  model. The last argument specifies the instance name of the root
  model and is used only if isRootModel is set to TRUE.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_ModelWriteSmv(
  FILE *fp,
  Hrc_Model_t *model,
  boolean isRootModel,
  char *rootInstanceName)
{
  Hrc_Node_t *hnode, *subcktHnode;
  Hrc_Model_t *subcktModel;
  int i, is_enum, range;
  Var_Variable_t *var, *actualVar;
  Hrc_Latch_t *latch;
  Hrc_Subckt_t *subckt;
  Tbl_Table_t *table;
  char *varName, *latchName, *instanceName;
  st_generator *gen;
  array_t *subcktActualInputVars, *subcktActualOutputVars;

  hnode = Hrc_ModelReadMasterNode(model);
  
  if (isRootModel){
    (void)fprintf(fp,"\n\nMODULE main --(");
    if (Hrc_NodeReadNumFormalInputs(hnode) != 0){
      fprintf(stderr, 
	      "Warning: ignored %d input parameter(s) in the main module (root model)\n", 
	      Hrc_NodeReadNumFormalInputs(hnode));
    }
    if (Hrc_NodeReadNumFormalOutputs(hnode) != 0){
      fprintf(stderr, 
	      "Warning: ignored %d output parameter(s) in the main module (root model)\n", 
	      Hrc_NodeReadNumFormalOutputs(hnode));
    }
  } else {
    (void)fprintf(fp,"\n\nMODULE %s (",Hrc_ModelReadName(model));
  }

  /* Input parameters */
  if (Hrc_NodeReadNumFormalInputs(hnode) != 0){
    Hrc_NodeForEachFormalInput(hnode,i,var){
      Io_SmvPrintVar(fp, var);
      if(i < Hrc_NodeReadNumFormalInputs(hnode)-1 ||
	 Hrc_NodeReadNumFormalOutputs(hnode) > 0) {
	(void)fprintf(fp,", ");
      }
    }
  }
  /* Output parameters */
  if (Hrc_NodeReadNumFormalOutputs(hnode) != 0){
    Hrc_NodeForEachFormalOutput(hnode,i,var){
      Io_SmvPrintVar(fp, var);
      if(i < Hrc_NodeReadNumFormalOutputs(hnode)-1) {
	(void)fprintf(fp,", ");
      }
    }
  }
  (void)fprintf(fp,")\n");

  /* Variables */
  Hrc_NodeForEachVariable(hnode,gen,varName,var){
    if (!isRootModel && (Var_VariableTestIsPI(var) || Var_VariableTestIsPO(var))) {
      fprintf(fp,"-- PI or PO: ");
    }

    is_enum = Var_VariableTestIsEnumerative(var);
    range = Var_VariableReadNumValues(var);
    
    (void)fprintf(fp,"VAR ");
    Io_SmvPrintVar(fp, var);
    if (is_enum == 1){
      if (range == 2){
	(void)fprintf(fp," : boolean;\n");
      }
      else {
	(void)fprintf(fp," : 0..%d;\n",range-1);
      }
    }
    else {
      /* variable var is symbolic */
      (void)fprintf(fp," : {");
      for (i=0; i < range; i++){
	(void)fprintf(fp,"%s",Var_VariableReadSymbolicValueFromIndex(var,i));
	if(i < range-1)
	  (void)fprintf(fp, ", ");
      }
      (void)fprintf(fp,"};\n");
    }
  }

  /* Subcircuits */
  Hrc_ModelForEachSubckt(model,gen,instanceName,subckt){
    subcktModel = Hrc_SubcktReadModel(subckt);
    subcktHnode = Hrc_ModelReadMasterNode(subcktModel);
    subcktActualInputVars = Hrc_SubcktReadActualInputVars(subckt);
    subcktActualOutputVars = Hrc_SubcktReadActualOutputVars(subckt);

    /* Note: mv uses named parameters; we do not check that there are
       enough parameters or that they occur in the correct order. This
       doesn't seem to matter at the moment. */
    (void)fprintf(fp,"VAR %s : %s(",
       Hrc_SubcktReadInstanceName(subckt), Hrc_ModelReadName(subcktModel));
    Hrc_NodeForEachFormalInput(subcktHnode,i,var){
      actualVar = array_fetch(Var_Variable_t *,subcktActualInputVars,i);
      Io_SmvPrintVar(fp, actualVar);
      if(i < Hrc_NodeReadNumFormalInputs(subcktHnode)-1 ||
	 Hrc_NodeReadNumFormalOutputs(subcktHnode) > 0) {
	(void)fprintf(fp,", ");
      }
    }   
    Hrc_NodeForEachFormalOutput(subcktHnode,i,var){
      actualVar = array_fetch(Var_Variable_t *,subcktActualOutputVars,i);
      Io_SmvPrintVar(fp, actualVar);
      if(i < Hrc_NodeReadNumFormalOutputs(subcktHnode)-1) {
	(void)fprintf(fp,", ");
      }
    }   
    (void)fprintf(fp,");\n");
  }

  /* .latches */
  (void)fprintf(fp,"\nASSIGN\n");
  Hrc_NodeForEachLatch(hnode,gen,latchName,latch){
    (void)fprintf(fp,"next(");
    Io_SmvPrintVar(fp, Hrc_LatchReadOutput(latch));
    (void)fprintf(fp,") := ");
    Io_SmvPrintVar(fp, Hrc_LatchReadInput(latch));
    (void)fprintf(fp,";\ninit(");
    Io_SmvPrintVar(fp, Hrc_LatchReadOutput(latch));
    (void)fprintf(fp,") := ");
    Tbl_TableWriteSmvToFile(Hrc_LatchReadResetTable(latch),1,fp);
    (void)fprintf(fp,"\n");
  }

  /* .names */
  Hrc_NodeForEachNameTable(hnode,i,table){
    Tbl_TableWriteSmvToFile(table,0,fp);
  }

}

/**Function********************************************************************

  Synopsis    [Returns pointer to root node in the hierarchy.]

  Description [The function simply accesses the rootNode field of the
               hierarchy manager.]

  SideEffects []

  SeeAlso     [Hrc_ManagerReadCurrentNode()]

******************************************************************************/
Hrc_Node_t *
Hrc_ManagerReadRootNode(
  Hrc_Manager_t *manager)
{
  return manager->rootNode;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the current node.]

  Description [The function simply accesses the currentNode field of the
               hierarchy manager.]

  SideEffects []

  SeeAlso     [Hrc_ManagerReadRootNode()]

******************************************************************************/
Hrc_Node_t *
Hrc_ManagerReadCurrentNode(
  Hrc_Manager_t *manager)
{
  return manager->currentNode;
}

/**Function********************************************************************

  Synopsis    [Returns the hash table of all the models.]

  Description [A pointer to the table containing all the models currently
               registered with the manager is returned. This table should
               not be modified in any way.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
st_table *
Hrc_ManagerReadModelTable(
  Hrc_Manager_t *manager)
{
  return manager->modelTable;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to model corresponding to the given name.]

  Description [The function looks up the name in the hash table of models.
               If the name is found, the corresponding model is returned
               otherwise NULL is returned.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Hrc_Model_t *
Hrc_ManagerFindModelByName(
  Hrc_Manager_t *manager,
  char *modelName)
{
  Hrc_Model_t *model;
    
  if(st_lookup(manager->modelTable, modelName, &model)) {
    return model;
  }
  else {
    return NIL(Hrc_Model_t);
  }
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the master node of a model.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Hrc_Node_t *
Hrc_ModelReadMasterNode(
  Hrc_Model_t *model)
{
  return model->masterNode;
}

/**Function********************************************************************

  Synopsis    [Returns the model name.]

  Description [A pointer to model name is returned. Note that this string
               should not be freed by the user.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
char *
Hrc_ModelReadName(
  Hrc_Model_t *model)
{
  return model->modelName;
}

/**Function********************************************************************

  Synopsis    [Returns the hash table of sub-circuits of the model.]

  Description [A pointer to the table containing all the sub-circuits of a 
               model is returned. Note that this table should not be modified
               in any way.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
st_table *
Hrc_ModelReadSubcktTable(
  Hrc_Model_t *model)
{
  return model->subcktTable;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to model of the sub-circuit.]

  Description [A pointer to the model whose instantiation created this
               sub-circuit is returned.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Hrc_Model_t *
Hrc_SubcktReadModel(
  Hrc_Subckt_t *subckt)
{
  return subckt->model;
}
    
/**Function********************************************************************

  Synopsis    [Returns pointer to instance name of a sub-circuit.]

  Description [A pointer to instance name of a sub-circuit is returned. Note
               that this string should not be freed by the user.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
char *
Hrc_SubcktReadInstanceName(
  Hrc_Subckt_t *subckt)
{
  return subckt->instanceName;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the array of actual input variables used for
               instantiating the model.]

  Description [A pointer to the array of actual input variables is returned.
               Note that the array should not be modified in any way.]

  SideEffects []

  SeeAlso     [Hrc_SubcktReadActualOutputVars()]

******************************************************************************/
array_t *
Hrc_SubcktReadActualInputVars(
  Hrc_Subckt_t *subckt)
{
  return subckt->actualInputVars;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the array of actual output variables used for
               instantiating the model.]

  Description [A pointer to the array of actual output variables is returned.
               Note that the array should not be modified in any way.]

  SideEffects []

  SeeAlso     [Hrc_SubcktReadActualInputVars()]

******************************************************************************/
array_t *
Hrc_SubcktReadActualOutputVars(
  Hrc_Subckt_t *subckt)
{    
  return subckt->actualOutputVars;
}

/**Function********************************************************************

  Synopsis    [Returns the manager of a node.]

  Description [The function returns a pointer to the manager of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Hrc_Manager_t *
Hrc_NodeReadManager(
  Hrc_Node_t *node)
{
  return node->manager;
}

/**Function********************************************************************

  Synopsis    [Returns the model name of a node.]

  Description [A pointer to the name of the model whose instantiation
               created this node is returned. Note that the string should not
               be freed by the user.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
char *
Hrc_NodeReadModelName(
  Hrc_Node_t *node)
{
  return node->modelName;
}

/**Function********************************************************************

  Synopsis    [Returns the instance name of a node.]

  Description [A pointer to the name used when instantiating a node is
               returned. Note that the string should not be freed by
               the user.]   

  SideEffects []

  SeeAlso     []

******************************************************************************/
char *
Hrc_NodeReadInstanceName(
  Hrc_Node_t *node)
{
  return node->instanceName;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the parent node of a node.]

  Description [A pointer to the parent node of a node is returned.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Hrc_Node_t *
Hrc_NodeReadParentNode(
  Hrc_Node_t *node)
{
  return node->parentNode;
}

/**Function********************************************************************

  Synopsis    [Returns the number of formal inputs of a node.]

  Description [The number of formal inputs of a node is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeReadNumFormalOutputs()]

******************************************************************************/
int
Hrc_NodeReadNumFormalInputs(
  Hrc_Node_t *node)
{
  return array_n(node->formalInputs);
}

/**Function********************************************************************

  Synopsis    [Returns the number of formal outputs of a node.]

  Description [The number of formal outputs of a node is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeReadNumFormalInputs()]

******************************************************************************/
int
Hrc_NodeReadNumFormalOutputs(
  Hrc_Node_t *node)
{
  return array_n(node->formalOutputs);
}

/**Function********************************************************************

  Synopsis    [This function returns the number of formal variables in a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int
Hrc_NodeReadNumVariables(
  Hrc_Node_t *node)
{
  return st_count(node->varTable);
}

/**Function********************************************************************

  Synopsis    [This function returns the number of tables in the node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int
Hrc_NodeReadNumTables(
  Hrc_Node_t *node)
{
  return array_n(node->nameTables);
}

/**Function********************************************************************

  Synopsis    [This function returns the number of latches in a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int
Hrc_NodeReadNumLatches(
  Hrc_Node_t *node)
{
  return st_count(node->latchTable);
}

/**Function********************************************************************

  Synopsis    [This function returns the number of children of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
int
Hrc_NodeReadNumChildren(
  Hrc_Node_t *node)
{
  return st_count(node->childTable);
}

/**Function********************************************************************

  Synopsis    [Returns the array of formal inputs of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the formal inputs of a node.
               It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
array_t *
Hrc_NodeReadFormalInputs(
  Hrc_Node_t *node)
{
  return node->formalInputs;
}

/**Function********************************************************************

  Synopsis    [Returns the array of formal outputs of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the formal outputs of a
               node. It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
array_t *
Hrc_NodeReadFormalOutputs(
  Hrc_Node_t *node)
{
  return node->formalOutputs;
}

/**Function********************************************************************

  Synopsis    [Returns the array of actual inputs of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the actual inputs of a node.
               It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
array_t *
Hrc_NodeReadActualInputs(
  Hrc_Node_t *node)
{
  return node->actualInputs;
}

/**Function********************************************************************

  Synopsis    [Returns the array of actual outputs of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the actual outputs of a
               node. It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
array_t *
Hrc_NodeReadActualOutputs(
  Hrc_Node_t *node)
{
  return node->actualOutputs;
}

/**Function********************************************************************

  Synopsis    [Returns the array of name tables of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the name tables in a
               node. It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
array_t *
Hrc_NodeReadNameTables(
  Hrc_Node_t *node)
{
  return node->nameTables;
}
    
/**Function********************************************************************

  Synopsis    [Returns the hash table of children of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the children of a node.
               It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
st_table *
Hrc_NodeReadChildTable(
  Hrc_Node_t *node)
{
  return node->childTable;
}

/**Function********************************************************************

  Synopsis    [Returns the hash table of latches of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the latches in a node.
               It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
st_table *
Hrc_NodeReadLatchTable(
  Hrc_Node_t *node)
{
  return node->latchTable;
}

/**Function********************************************************************

  Synopsis    [Returns the hash table of variables of a node.]

  Description [Note that this function has been exported only to provide
               the exported macro to iterate over the variables in a node.
               It is not supposed to be used for any other purpose.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
st_table *
Hrc_NodeReadVariableTable(
  Hrc_Node_t *node)
{
  return node->varTable;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the undef field of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void *
Hrc_NodeReadUndef(
  Hrc_Node_t *node)
{
  return node->undef;
}
/**Function********************************************************************

  Synopsis    [Returns pointer to a latch corresponding to a name.]

  Description [The name is looked up in the hash table of latches. If it is
               found, a pointer to the corresponding latch is returned
               otherwise NULL is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeFindVariableByName(), Hrc_NodeFindChildByName()]

******************************************************************************/
Hrc_Latch_t *
Hrc_NodeFindLatchByName(
  Hrc_Node_t *node,
  char *latchName)
{
  Hrc_Latch_t *latch;
    
  if(st_lookup(node->latchTable, latchName, &latch)) {
    return latch;
  }
  else {
    return NIL(Hrc_Latch_t);
  }
}

/**Function********************************************************************

  Synopsis    [Returns pointer to a variable corresponding to a name.]

  Description [The name is looked up in the hash table of variables. If it is
               found, a pointer to the corresponding variable is returned
               otherwise NULL is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeFindLatchByName(), Hrc_NodeFindChildByName()]

******************************************************************************/
Var_Variable_t *
Hrc_NodeFindVariableByName(
  Hrc_Node_t *node,
  char *varName)
{
  Var_Variable_t *var;
  
  if(st_lookup(node->varTable, varName, &var)) {
    return var;
  }
  else {
    return NIL(Var_Variable_t);
  }    
}

/**Function********************************************************************

  Synopsis    [Returns pointer to child node corresponding to an instance
               name.]

  Description [The name is looked up in the hash table of children. If it is
               found, a pointer to the corresponding child is returned
               otherwise NULL is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeFindLatchByName(), Hrc_NodeFindVariableByName()]

******************************************************************************/
Hrc_Node_t *
Hrc_NodeFindChildByName(
    Hrc_Node_t *node,
    char *instanceName)
{
  Hrc_Node_t *childNode;

  if(st_lookup(node->childTable, instanceName, &childNode)) {
    return childNode;
  }
  else {
    return NIL(Hrc_Node_t);
  }    
}

/**Function********************************************************************

  Synopsis    [Returns pointer to input variable of a latch.]

  SideEffects []

  SeeAlso     [Hrc_LatchReadOutput()]

******************************************************************************/
Var_Variable_t *
Hrc_LatchReadInput(
  Hrc_Latch_t *latch)
{
  return latch->latchInput;
}
    
/**Function********************************************************************

  Synopsis    [Returns pointer to output variable of a latch.]

  SideEffects []

  SeeAlso     [Hrc_LatchReadInput()]

******************************************************************************/
Var_Variable_t *
Hrc_LatchReadOutput(
  Hrc_Latch_t *latch)
{
  return latch->latchOutput;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the reset table of a latch.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Tbl_Table_t *
Hrc_LatchReadResetTable(
    Hrc_Latch_t *latch)
{
  return latch->resetTable;
}

/**Function********************************************************************

  Synopsis    [Returns pointer to the undef field in a latch.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void *
Hrc_LatchReadUndef(
  Hrc_Latch_t *latch)
{
  return latch->undef;
}

/**Function********************************************************************

  Synopsis    [Sets the rootNode field of a manager.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_ManagerSetRootNode(
  Hrc_Manager_t *manager,
  Hrc_Node_t *node)
{
  manager->rootNode = node;
}

/**Function********************************************************************

  Synopsis    [Sets the currentNode field of a manager.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_ManagerSetCurrentNode(
  Hrc_Manager_t *manager,
  Hrc_Node_t *currentNode)
{
  manager->currentNode = currentNode;
}

/**Function********************************************************************

  Synopsis    [Adds a formal input variable to a node.]

  Description [The formal input variable is added to the array of formal
               inputs and to the hash table of variables and TRUE is returned.
               If the variable is a primary input already, nothing is done
               and FALSE is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeAddFormalOutput()]

******************************************************************************/
boolean
Hrc_NodeAddFormalInput(
  Hrc_Node_t *node,
  Var_Variable_t *var)
{
  char *name = Var_VariableReadName(var);

  if(!Var_VariableTestIsPI(var)) {
    array_insert_last(Var_Variable_t *, node->formalInputs, var);
    st_insert(node->varTable, name, (char *) var);
    (void)Var_VariableSetPI(var);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************

  Synopsis    [Adds a formal output variable to a node.]

  Description [The formal output variable is added to the array of formal
               outputs and to the hash table of variables and TRUE is
               returned. If the variable is a primary output already,
               nothing is done and FALSE is returned.]

  SideEffects []

  SeeAlso     [Hrc_NodeAddFormalInput()]

******************************************************************************/
boolean
Hrc_NodeAddFormalOutput(
  Hrc_Node_t *node,
  Var_Variable_t *var)
{
  char *name = Var_VariableReadName(var);

  if(!Var_VariableTestIsPO(var)) {
    array_insert_last(Var_Variable_t *, node->formalOutputs, var);
    st_insert(node->varTable, name, (char *) var);
    (void)Var_VariableSetPO(var);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************

  Synopsis    [Adds a name table to a node.]

  SideEffects []

  SeeAlso     [Hrc_NodeAddFormalInput(), Hrc_NodeAddFormalOutput()]

******************************************************************************/
void
Hrc_NodeAddNameTable(
  Hrc_Node_t *node,
  Tbl_Table_t *nameTable)
{
  array_insert_last(Tbl_Table_t *, node->nameTables, nameTable);
}

/**Function********************************************************************

  Synopsis    [Sets the undef field of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_NodeSetUndef(
  Hrc_Node_t *node,
  void *value)
{
  node->undef = value;
}

/**Function********************************************************************

  Synopsis    [Sets the reset table of a latch.]

  Description [If the resetTable field of the latch is NULL, it is set to
               table and TRUE is returned otherwise FALSE is returned. ]

  SideEffects []

  SeeAlso     []
  
******************************************************************************/

boolean
Hrc_LatchSetResetTable(
  Hrc_Latch_t *latch,
  Tbl_Table_t *table)
{
  if (latch->resetTable != NIL(Tbl_Table_t)){
    return FALSE;
  }
  latch->resetTable = table;
  return TRUE;
}

/**Function********************************************************************

  Synopsis    [Set the undef field of a latch.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_LatchSetUndef(
  Hrc_Latch_t *latch,
  void *value)
{
  latch->undef = value;
}
    

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

























