/**CHeaderFile*****************************************************************

  FileName    [hrcInt.h]

  PackageName [hrc]

  Synopsis    [The internal declarations needed for the hierarchy]

  Description [This file contains the definitions of important structures
               needed for defining the hierarchy.]

  SeeAlso     [hrc.h]

  Author      [Yuji Kukimoto, Shaz Qadeer]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: hrcInt.h,v 1.2 2001/03/12 22:28:29 jinh Exp $]

******************************************************************************/

#ifndef _HRCINT
#define _HRCINT
 
#include "hrc.h"
#include "cmd.h"
#include <string.h>

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Enum************************************************************************

  Synopsis    [Types of components present in a circuit.]

******************************************************************************/
typedef enum {
  HrcLatch_c,             /* component is a latch */
  HrcSubckt_c,            /* component is a subckt */
  HrcUndef_c              /* component is an arbitrary gate */
} HrcCktComponentType;

/**Struct**********************************************************************

  Synopsis    [Structure to store uninterpreted application info.]

  SeeAlso     [Hrc_NodeAddApplInfo()]

******************************************************************************/
typedef struct ApplInfoStruct {
  Hrc_ApplInfoFreeFn  freeFn;  /* application function to free data */
  Hrc_ApplInfoChangeFn changeFn; /* application function to change data */
  void               *data;    /* application data */
} ApplInfo_t;

/**Struct**********************************************************************

  Synopsis    [Definition of a hierarchy manager]

  Description [rootNode is a pointer to the root node of the hierarchy tree.
               It corresponds to the instantiation of the root model in a
               blifmv file. currentNode points to the current position in
               the hierarchy. This pointer is initialized to root node. Its
               value changes whenever the cd command for changing the
               hierarchical level is used. modelTable contains all the
               defined models.]

  SeeAlso     []

******************************************************************************/
struct HrcManagerStruct {
  Hrc_Node_t *rootNode;   /* pointer to root Hrc_Node_t */
  Hrc_Node_t *currentNode;/* pointer to current Hrc_Node_t */
  st_table *modelTable;   /* hash table from model name(char *)
                           * to model (Hrc_Model_t *) */
};

/**Struct**********************************************************************

  Synopsis    [Definition of a model]

  Description [hrcNode is a pointer to a the master node of the model. The
               master node contains information about all the elements in
               the model. subcktTable contains all information about the
               various instantiations of other models from within the model.]

  SeeAlso     [HrcNodeStruct, HrcSubcktStruct]

******************************************************************************/
struct HrcModelStruct {
  char *modelName;           /* unique model name */
  Hrc_Node_t *masterNode;       /* pointer to the master hrcNode of a model */
  st_table *subcktTable;     /* hash table from instance name (char *) of
                              * model being called to subckt (Hrc_Subckt_t *)*/
};

/**Struct**********************************************************************

  Synopsis    [Definition of a sub-circuit]

  Description [A subckt is created whenever a model instantiates another. The
               two arrays actualInputVars and actualOutputVars map contain
               the variables used to refer to the I/O pins of the instantiated
               model.]

  SeeAlso     [HrcModelStruct]

******************************************************************************/
struct HrcSubcktStruct {
  Hrc_Model_t *model;       /* the subckt resulted from an instantiation of
                             * this model */
  char *instanceName;       /* name by which the instantiation took place */
  array_t *actualInputVars; /* array of actual input variable pointers 
                             * Var_Variable_t *'s */
  array_t *actualOutputVars;/* array of actual output variable pointers
                             * Var_Variable_t *'s */
};

/**Struct**********************************************************************

  Synopsis    [Definition of a node in the hierarchy tree]

  Description [This structure forms the basic unit of the hierarchy tree.
               Note that the elements of arrays formalInputs and actualInputs,
               and formalOutputs and actualOutputs are in one to one
               correspondence.]

  SeeAlso     [HrcModelStruct]

******************************************************************************/
struct HrcNodeStruct {
  Hrc_Manager_t *manager;    /* manager to which this node belongs */
  char *modelName;           /* corresponding model name */
  char *instanceName;        /* instance name */
  Hrc_Node_t *parentNode;    /* pointer to the unique parent node */
  array_t *formalInputs;     /* formal input symbolic variables 
                              * (Var_Variable_t *) */
  array_t *formalOutputs;    /* formal output symbolic variables 
                              * (Var_Variable_t *) */
  array_t *actualInputs;     /* actual connections (Var_Variable_t *) with 
                              * the inputs of parent node */
  array_t *actualOutputs;    /* actual connections (Var_Variable_t *) with 
                              * the outputs of parent node */  
  array_t *nameTables;       /* array of Tbl_Table_t *'s corresponding
                              * to .names in the blifmv file */
  st_table *childTable;      /* hash table from child instance name (char *) 
                              * to child (Hrc_Node_t *) */
  st_table *latchTable;      /* hash table from name of output of latch
                              * (char *)  to latch (Hrc_Latch_t *) */
  st_table *varTable;        /* hash table from symbolic variable name
                              * (char *) to (Var_Variable_t *) */
  st_table *applInfoTable;   /* hash table for application specific
                                information from (char *) to (ApplInfo_t *) */ 
  void *undef;               /* for programmer's use */
};

/**Struct**********************************************************************

  Synopsis    [Definition of a latch]

  Description []

  SeeAlso     []

******************************************************************************/
struct HrcLatchStruct {
  Var_Variable_t *latchInput;  /* input of the latch (next state) */
  Var_Variable_t *latchOutput; /* output of the latch (present state) */
  Tbl_Table_t *resetTable;     /* pointer to reset table for this latch */
  void *undef;                 /* for programmer's use */
};

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN void HrcNodeFreeRecursively(Hrc_Node_t *node);
EXTERN void HrcNodeFreeInternalMemory(Hrc_Node_t *node);
EXTERN Hrc_Node_t * HrcNodeDeleteChild(Hrc_Node_t *node, char *childName);

/**AutomaticEnd***************************************************************/

#endif /* _HRCINT */





















