%{
/**CFile***********************************************************************

  FileName    [io.l]

  PackageName [io]

  Synopsis    [Lexical analyzer for BLIF-MV.]

  Description []

  SeeAlso     []

  Author      [Yuji Kukimoto, Rajeev Ranjan, Huey-Yih Wang]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "ioYacc.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define IoMAXSTACKDEPTH (32)

/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

YY_BUFFER_STATE globalFilePtrStack[IoMAXSTACKDEPTH];
int globalCurrentStackDepth = 0;

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd***************************************************************/

static char * _IoFileNameExtract(char *string);

%}

/*---------------------------------------------------------------------------*/
/* Lexical analyzer rules                                                    */
/*---------------------------------------------------------------------------*/

%option noyywrap

/* -, =, (, ), {, }, ! and ; are intentionally removed from symbol
   - is for dashes in tables
   = is for the = construct in tables
   ( and ) are for lists in tables
   { and } are for ranges in tables
   ! is for complements in tables
   ; is for delimitters in CTL formulas
removing - has another effect. Since - is not a symbol, -> in the table
construct is always found by the lexer
 */
blank  [ \t\r\b]
alpha  [A-Za-z]
number [0-9]
symbol [\^\?\|\/\[\]\+\*\$\<\>~@\_#\$%\:\"\'\.]
/*symbol [\^\?\|\/\[\]\+\*\$\<\>~@\_#\$%\:\;\"\'\.]*/
/*symbol [\^\?\|\/\+\*\$\<\>~@\_#\$%\:\;\"\'\.]*/
alnumsymbol  {alpha}|{number}|{symbol}
alnumsymbolblank  {alnumsymbol}|{blank}

%%

"->"	{
	return ARROW;
	}       
"="	{ 
        return ASSIGN; 
	}
^{blank}*\n	{
		globalLineNumber++;
		continue;
		}
\n	{
	globalLineNumber++;
	return NEWLINE;
	}
\\\n{blank}*\n	{
		globalLineNumber += 2;
		return NEWLINE;
		}
\\$		{
#ifdef __cplusplus
                (void)yyinput();
#else
		(void)input();
#endif
		globalLineNumber++;
		continue;
		}

\.include{blank}+{alnumsymbol}*\n {
			char *fileName, *realFileName;

			fileName = _IoFileNameExtract(IoYytext);
                        if (globalCurrentStackDepth >= IoMAXSTACKDEPTH){
                           error_append("Error: .include nested too deeply.\n");
                           IoError();
                        }
			globalFilePtrStack[globalCurrentStackDepth++] = YY_CURRENT_BUFFER;
			IoYyin = Cmd_FileOpen(fileName,"r",&realFileName,1);
			if (IoYyin == NIL(FILE)) {
				error_append("Error: Include file ");
				error_append(fileName);
				error_append(" is not found.\n");
				IoError();
			}
                        IoYy_switch_to_buffer(IoYy_create_buffer(IoYyin,YY_BUF_SIZE));
                        BEGIN(INITIAL);
			continue;
			}
<<EOF>>		{
		if (--globalCurrentStackDepth < 0){
		yyterminate();
		}
		else {
                  IoYy_delete_buffer(YY_CURRENT_BUFFER);
                  IoYy_switch_to_buffer(globalFilePtrStack[globalCurrentStackDepth]);
                }
                }
                  

\.{alpha}*	{
		if (strcmp(IoYytext,".model") == 0) {
			return MODEL;
		}
		else if (strcmp(IoYytext,".inputs") == 0) {
		        return INPUTS;
		}
		else if (strcmp(IoYytext,".outputs") == 0) {
			return OUTPUTS;
		}
		else if (strcmp(IoYytext, ".mv" ) == 0 ){
			return MV;
		}
		else if (strcmp(IoYytext,".latch") == 0) {
			return LATCH;
		}
		else if (strcmp(IoYytext,".table") == 0 || strcmp(IoYytext,".names") == 0) {
			return NAMES;
		}
		else if (strcmp(IoYytext,".reset") == 0 || strcmp(IoYytext,".r") == 0) {
			return RESET;
		}
		else if (strcmp(IoYytext,".subckt") == 0) {
			return SUBCKT;
		}
		else if (strcmp(IoYytext,".default") == 0 || strcmp(IoYytext,".def") == 0) {
			return DEFAULT;
		}
		else if (strcmp(IoYytext,".root") == 0) {
			return ROOT;
		}
		else if (strcmp(IoYytext,".end") == 0) {
			return END;
		}
		else {
			error_append("Error: Unknown construct ");
			error_append(IoYytext);
			error_append("\n");
			IoError();
		}
		}
{alnumsymbol}+	{
		return IDENTIFIER;
		}
^{blank}*#[^\n]*\n	{
		/* for lines containing only comments can be skipped */
		globalLineNumber++;
		continue;
		}
{blank}+	{
		/* skip white spaces */
	        continue;
		}
#[^\n]*\n		{
			/* for lines containing meaningful info 
			followed by comments; should return NEWLINE */ 
			globalLineNumber++;
			return NEWLINE;
			}
.	{
	return IoYytext[0];
	}

%%


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Chops off the initial .include from a given string.]

  Description []

  SideEffects []

  SeeAlso     []

******************************************************************************/

static char *
_IoFileNameExtract(char *string)
{
  char *fileName;

  fileName = string + strlen(".include ");
  fileName[strlen(fileName)-1] = '\0';
  return fileName;
}
