/**CFile***********************************************************************

  FileName    [truesimCmd.c]

  PackageName [truesim]

  Synopsis    [Command interface for the truesim package.]

  Description [Command interface for the truesim package. This package
  implements delay-based simulation of a combinational circuit in addition
  to zero-delay simulation. The 'sim' package does zero-delay simulation,
  but handles both combinational and sequential circuits. This package
  can take as input circuits described in BLIF format only.

  In delay-based simulation each node of the network has a certain
  amount of propagation delay through it. The delay and load for each node
  of the circuit can be specified by the user. In the absence of delay
  values, the fanout count of the node is assumed to be the delay. In
  this implementation the delay values do not have units. For example,
  a node with 2 fanouts has a delay of 2, an node with fanout 4 has
  delay of 4, etc. Primary outputs are assumed to have a delay of 1,
  unlesss specfied otherwise. Load values are used by other
  applications to compute power dissipation (by using the node switching
  information via simulation.)

  Boolean vectors can be specified for the primary inputs of the circuit.
  Alternatively, bit vectors can also be generated randomly. The bit
  vectors are applied to the circuit one at a time. The effects of one
  pattern (a bit-vector for the primary inputs) are propagated through
  the circuit and subsequent patterns are applied only after all the
  signals in the circuit have settled.]

  Author      [Balakrishna Kumthekar <kumtheka@colorado.edu>]

  Copyright [This file was created at the University of Colorado at
  Boulder.  The University of Colorado at Boulder makes no warranty about
  the suitability of this software for any purpose.  It is presented on
  an AS IS basis.]

******************************************************************************/

#include "truesimInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define DEFAULT_NUM_PATTERNS 1000
/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

static jmp_buf timeOutEnv;

/* Global variable used within truesim package */
int createdPart;
int truesimRptHeader;
int truesimVerbose;

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int CommandTruesim(Hrc_Manager_t **hmgr, int argc, char **argv);
static void TimeOutHandle(void);
static int TestIsNetworkMultipleValued(Ntk_Network_t *network);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [This function initializes the truesim package.]

  SideEffects [None]

  SeeAlso     [Truesim_End]

******************************************************************************/
void
Truesim_Init(void)
{
  Cmd_CommandAdd("truesim", CommandTruesim, 0);
}

/**Function********************************************************************

  Synopsis    [This function ends the truesim package.]

  SideEffects [None]

  SeeAlso     [Truesim_Init]

******************************************************************************/
void
Truesim_End(void)
{

} /* End of Truesim_End */


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Implements the truesim command.]

  CommandName [truesim]

  CommandSynopsis [Simulate the flattened network]

  CommandArguments [\[ -D &lt;filename&gt; \] \[ -d \] 
  \[ -f &lt;filename&gt; \] \[ -g &lt;filename&gt; \] \[ -h \]  
  \[ -n &lt;N&gt; \] \[ -p &lt;filename&gt; \] \[ -r &lt;N&gt; \] 
  \[ -t &lt;time&gt; \] \[ -v &lt;N&gt; \]]

  CommandDescription [Simulates a network with a set of input
  vectors. Before calling this command, the user should create a partition
  (using the command <tt>build_partition_mdds</tt>). The simulation
  vectors can be provided by the user (using -f vectors_file), or
  generated randomly. <p>

  Command options:<p>

  <dl>
  <dt>-D &lt;filename&gt;
  <dd> File to dump randomly generated vectors. Primary input
  probability file SHOULD be specified with -p option.


  <dt>-d 
  <dd>Perform event-driven true delay simulation. The fanout count of
  the nodes is used as an estimate for the node delay. Primary inputs
  are assumed to have arrival times of zero.


  <dt>-f &lt;filename&gt; 
  <dd> File containing simulation vectors. The format is simple but
  strict and hence, few checks are made. The file format is as below:

  <p>.i <code>c n d o e p f q g r h s i t j u k a l b m</code><br>
  .s <br>
  0 1 1 1 1 1 0 1 1 1 1 1 1 1 1 1 0 1 1 1 1 ; <br>
  0 1 1 0 0 0 0 1 0 1 1 0 1 1 1 1 0 0 0 0 1 ; <br>
  0 0 1 0 0 0 0 1 0 1 0 0 0 1 0 1 0 0 0 0 1 ; <br>
  0 1 1 0 0 0 0 1 1 1 1 1 1 1 1 1 0 1 0 0 1 ; <br>

  <p> The .i statement specifies the primary inputs of the network. The
  patterns start after .s key word. Each vector is a space separated list
  of bits and ends in a semi-colon. The length of any vector should be
  equal to the number of signals specified in the .i statement. A line
  starting with # is a comment. <p>


  <dt>-g &lt;filename&gt; 
  <dd> File to output random probabilities for primary inputs. Use this
  option ONLY to produce a vector of signal probabilities for primary
  inputs. This option does not proceed with simulation.


  <dt>-h
  <dd>Print a help message that details all options.


  <dt> -n &lt;N&gt;
  <dd> Number of patterns to simulate. The default is 1000. Use this
  option ONLY with -p option.


  <dt>-p &lt;filename&gt;
  <dd> File containing primary input probabilities. The format is as
  shown below:

  <p> 
  PI_Name1 0.67 <br>
  PI_Name2 0.45 <br>


  <dt>-r &lt;N&gt; 
  <dd> Print PI/PO names after every N patterns simulated, for better
  readability of simulation output. This option is effective only when
  the verbosity flag is on.

  <dt>-T &lt;filename&gt;
  <dd> File containing node delay and load values. The file format is
  as shown below:

  <p>
  node1 delay1 load1 <br>
  node2 delay2 load2 <br>

  <p> The delay vaues are assumed as specified in a common unit. The same
  is true for load values too.

  
  <dt>-t &lt;time&gt;
  <dd> Time in seconds allowed to finish the simulation.  The default
  is no limit.


  <dt>-v &lt;N&gt;
  <dd>  Specify verbosity level, N.
  </dl> <p>]

  SideEffects []

******************************************************************************/
static int
CommandTruesim(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  Ntk_Network_t *network = NIL(Ntk_Network_t);
  graph_t *partition;
  boolean trueDelay;
  int genVectors,N,status;
  int c; /* For scanning command line arguments */
  long timeOutPeriod,initialTime,finalTime;
  char *probFile,*simFile,*probGenFile,*dumpFile;
  char *delayFile;
  FILE *fp = NIL(FILE);;
  
  /* These are the default values. */
  createdPart = 0; /* =1 when partiton is created by the command */
  trueDelay = FALSE; /* Use zero-delay simulation by default */
  timeOutPeriod   = 0; /* Unlimited time */ 
  genVectors = -1; /* Do not generate bit-vectors for simulation */
  N = -1; /* Number of patterns to simulate */
  probFile = NIL(char); /* File with probability values for PI */
  simFile = NIL(char); /* Simulation bit-vector file */
  probGenFile = NIL(char); /* File with prob. values for PI; used during
  			    * bit-vector generation */
  dumpFile = NIL(char); /* File to dump generated bit-vectors */
  delayFile = NIL(char); /* File with delay vaules for nodes of the
  			  * circuit */
  truesimVerbose = 0; /* Verbosity level */
  truesimRptHeader = -1; /* Print PI/PO names while printing simulation
  		      * results */

  if (bdd_get_package_name() != CUDD) {
    (void) fprintf(vis_stderr,
		   "** truesim error: The truesim package can be used only with CUDD package\n");
    (void) fprintf(vis_stderr,
		   "** truesim error: Please link with CUDD package\n");
    return 0;
  }
	
  util_getopt_reset();

  while((c = util_getopt(argc, argv, "D:df:g:hn:p:r:T:t:v:")) != EOF) {
    switch(c) {
    case 'D':
      if (!util_optarg) {
	(void) fprintf(vis_stderr,
		       "** truesim error: File to output pattern vectors not specified.\n");
	goto usage;
      }
      dumpFile = util_strsav(util_optarg);
      break;
    case 'd':
      trueDelay = TRUE;
      break;
    case 'f': 
      if (!util_optarg) {
	(void) fprintf(vis_stderr,
		       "** truesim error: Simulation file not specified.\n");
	goto usage;
      } else {
	simFile = util_strsav(util_optarg);
	if ((fp = Cmd_FileOpen(simFile,"r",NIL(char *),1)) == NIL(FILE)) {
	  (void) fprintf(vis_stderr,
			 "** truesim error: Could not open %s for reading.\n",
			 simFile);
	  goto endgame;
	} else {
	  fclose(fp);
	}
	/* Stop generation of vectors, in case -p is also used with -f */
	N = -1;
	genVectors = 0;
	if (probFile) {
	  FREE(probFile);
	  probFile = NIL(char);
	}
      }
      break;
    case 'g': 
      if (!util_optarg) {
	(void) fprintf(vis_stderr,
		       "** truesim error: File to output input probs. not specified.\n");
	goto usage;
      }
      probGenFile = util_strsav(util_optarg);
      break;
    case 'h':
      goto usage;
    case 'n':
      if (genVectors != 0) {
	N = atoi(util_optarg);
	genVectors = 1;
      } else {
	(void) fprintf(vis_stderr,
		       "** truesim warning: Simulation file already specified.\n");
	(void) fprintf(vis_stderr,
		       "** truesim warning: Ignoring -n option.\n");
      }
      break;
    case 'p': 
      if (genVectors != 0) {
	if (!util_optarg) {
	  (void) fprintf(vis_stderr,
			 "** truesim error: Probability file not specified.\n");
	  goto usage;
	} else {
	  probFile = util_strsav(util_optarg);
	  if ((fp = Cmd_FileOpen(probFile,"r",NIL(char *),1)) == NIL(FILE)) {
	    (void) fprintf(vis_stderr,
			   "** truesim error: Could not open %s for reading.\n",
			   probFile);
	    goto endgame;
	  } else {
	    fclose(fp);
	  }
	}
	genVectors = 1;
      } else {
	(void) fprintf(vis_stderr,
		       "** truesim warning: Simulation file already specified.\n");
	(void) fprintf(vis_stderr,
		       "** truesim warning: Ignoring -p option.\n");
      }
      break;
    case 'r':
      truesimRptHeader = atoi(util_optarg);
      break;
    case 'T': 
      if (!util_optarg) {
	(void) fprintf(vis_stderr,
		       "** truesim error: Delay file not specified.\n");
	goto usage;
      } else {
	delayFile = util_strsav(util_optarg);
	if ((fp = Cmd_FileOpen(delayFile,"r",NIL(char *),1)) == NIL(FILE)) {
	  (void) fprintf(vis_stderr,
			 "** truesim error: Could not open %s for reading.\n",
			 delayFile);
	  goto endgame;
	} else {
	  fclose(fp);
	}
      }
      break;
    case 't':
      timeOutPeriod = atoi(util_optarg);
      break;
    case 'v':
      truesimVerbose = atoi(util_optarg);
      break;
    default:
      goto usage;
    }
  }

  if(Hrc_ManagerReadCurrentNode(*hmgr) == NIL(Hrc_Node_t)) {
    (void) fprintf(vis_stderr,"** truesim error: The hierarchy manager is empty.");
    (void) fprintf(vis_stderr," Read in design.\n");
    goto endgame;
  }

  network = (Ntk_Network_t *) 
    Hrc_NodeReadApplInfo(Hrc_ManagerReadCurrentNode(*hmgr), 
			 NTK_HRC_NODE_APPL_KEY);

  if(network == NIL(Ntk_Network_t)) {
    (void) fprintf(vis_stderr,"** truesim error: There is no network. ");
    (void) fprintf(vis_stderr,"Use flatten_hierarchy.\n");
    goto endgame;
  }

  /* Check if the current network has signals with multiple values. */
  if (TestIsNetworkMultipleValued(network)) {
    (void) fprintf(vis_stderr,"** truesim error: Circuit has multiple valued variables.\n");
    (void) fprintf(vis_stderr,"** truesim error: This algorithm applies to boolean signals only.\n");
    goto endgame;
  }

  if(Ntk_NetworkReadNumPrimaryInputs(network) !=
     Ntk_NetworkReadNumInputs(network)) {
    (void) fprintf(vis_stderr,"** truesim error: This algorithm applies only for circuits descrbied in BLIF format.\n");
    goto endgame;
  }

  /* Generate a file with random values for primary inputs and return. No
     simulation is performed. */
  if (probGenFile) {
    Truesim_GeneratePrimaryInputProbs(network,probGenFile);
    FREE(probGenFile);
    if (probFile)
      FREE(probFile);
    if (simFile)
      FREE(simFile);
    if (dumpFile)
      FREE(dumpFile);
    if (delayFile)
      FREE(delayFile);
    return 0;
  }

  /* Make sure that we either generate random vectors or we are
     supplied with simulation vectors */
  if (genVectors == -1) {
    (void) fprintf(vis_stderr,
		   "** truesim error: Neither simulation vector file nor ");
    (void) fprintf(vis_stderr,
		   "PI probabilities provided.\n");
    (void) fprintf(vis_stderr,
		   "** truesim error: No simulation will be performed.\n");
    goto endgame;
  }

  if (dumpFile && !genVectors) {
    if (!simFile) {
      (void) fprintf(vis_stderr,
		     "** truesim error: PI probability file not specified.\n");
      (void) fprintf(vis_stderr, 
		     "** truesim error: Use -p option.\n");
      (void) fprintf(vis_stderr,
		     "** truesim error: No simulation will be performed.\n");
      goto endgame;
    } else {
      (void) fprintf(vis_stderr,
      		     "** truesim warning: Simulation file %s specified.\n",
		     simFile);
      (void) fprintf(vis_stderr,
      		     "** truesim warning: Ignoring dump file %s\n",dumpFile);
    }
  }
  
  /* Make sure N is specified or a default value is used */
  if (N == -1 && genVectors == 1) {
    (void) fprintf(vis_stderr,
		   "** truesim warning: Number of patterns to be simulated is not specified.\n");
    (void) fprintf(vis_stderr,
		   "Assuming N = 1000 patterns to be simulated.\n");
    N = DEFAULT_NUM_PATTERNS;
  }

  /* Access a 'total' partition */
  partition = (graph_t *) Ntk_NetworkReadApplInfo(network, 
						  PART_NETWORK_APPL_KEY);
  createdPart = 0; /* Using partition of the network. */
  if (partition == NIL(graph_t) || 
      (Part_PartitionReadMethod(partition) != Part_Total_c)) {
    partition = Part_NetworkCreatePartition(network, 
					    NIL(Hrc_Node_t),
					    "dummy", (lsList) 0, 
					    (lsList) 0, NIL(mdd_t),
					    Part_Total_c,
					    (lsList) 0, 
					    FALSE, FALSE, TRUE);
    if (partition == NIL(graph_t)) {
      (void) fprintf(vis_stderr,"** truesim error: Could not create a partition.\n");
      goto endgame;
    }
    Ntk_NetworkAddApplInfo(network,PART_NETWORK_APPL_KEY,
			   (Ntk_ApplInfoFreeFn)Part_PartitionFreeCallback,
			   (void *)partition);
    createdPart = 1; /* Using new partition */
  }

  /* Start the timer.*/
  if (timeOutPeriod > 0){
    (void) signal(SIGALRM, (void(*)(int))TimeOutHandle);
    (void) alarm(timeOutPeriod);
    if (setjmp(timeOutEnv) > 0) {
      (void) fprintf(vis_stderr, "** truesim warning: Timeout occurred after ");
      (void) fprintf(vis_stderr, "%ld seconds.\n", timeOutPeriod);
      alarm(0);
      goto endgame;
    }
  }

  initialTime = util_cpu_time();
  status = TruesimSimulateNetwork(network,simFile,probFile,delayFile,
				  dumpFile,trueDelay,genVectors,N);
  finalTime = util_cpu_time();
  if(status) {
    (void) fprintf(vis_stdout, "%-20s%10ld\n", "** truesim info: analysis time =",
		   (finalTime-initialTime)/1000);
  }
  else {
    (void) fprintf(vis_stdout, "** truesim error: Simulation was not successful.\n");
  }

  alarm(0);
  /* Clean up */
  if (probGenFile)
    FREE(probGenFile);
  if (probFile)
    FREE(probFile);
  if (simFile)
    FREE(simFile);
  if (dumpFile)
    FREE(dumpFile);
  if (delayFile)
    FREE(delayFile);
  return 0;  /* normal exit */

endgame:
  /* Clean up */
  if (probGenFile)
    FREE(probGenFile);
  if (probFile)
    FREE(probFile);
  if (simFile)
    FREE(simFile);
  if (dumpFile)
    FREE(dumpFile);
  if (delayFile)
    FREE(delayFile);
  
  if (createdPart)
    Ntk_NetworkFreeApplInfo(network,PART_NETWORK_APPL_KEY);
  
  return 1; /* Error exit */

usage:

  (void) fprintf(vis_stderr, "\nusage: Also see 'help truesim' for more details.\n\n");
  (void) fprintf(vis_stderr, "    -D <filename>\tFile to dump randomly generated pattern vectors.\n");
  (void) fprintf(vis_stderr, "            \t\tPrimary input probability file SHOULD be specified with \n");
  (void) fprintf(vis_stderr, "            \t\t-p option.\n\n");
  (void) fprintf(vis_stderr, "    -d      \t\tPerform true delay simulation.\n\n");
  (void) fprintf(vis_stderr, "    -f <filename>\tFile containing simulation vectors.\n\n");
  (void) fprintf(vis_stderr, "    -g <filename>\tFile to output random probabilities for PI \n");
  (void) fprintf(vis_stderr, "            \t\tUse this option ONLY to produce a vector of \n");
  (void) fprintf(vis_stderr, "            \t\tprobability values for PI. No simulation is performed. \n\n");
  (void) fprintf(vis_stderr, "    -h      \t\tCommand usage.\n\n");
  (void) fprintf(vis_stderr, "    -n <N>  \t\tNumber of patterns to simulate. Default is 1000.\n");
  (void) fprintf(vis_stderr, "            \t\tShould ONLY be used with -p option.\n\n");
  (void) fprintf(vis_stderr, "    -p <filename>\tFile containing PI probabilities.\n");
  (void) fprintf(vis_stderr, "            \t\tSee help for file format.\n\n");
  (void) fprintf(vis_stderr, "    -r <N>  \t\tPrint the node name header ever N iterations.\n");
  (void) fprintf(vis_stderr, "            \t\tThis is effective only when verbosity flag is on.\n\n");
  (void) fprintf(vis_stderr, "    -T <filename>\tFile containing node delay and load values.\n");
  (void) fprintf(vis_stderr, "            \t\tSee help for file format.\n\n");
  (void) fprintf(vis_stderr, "    -t <time> \t\tTime in seconds allowed to finish the simulation.\n\n");
  (void) fprintf(vis_stderr, "    -v <N>  \t\tVerbosity level.\n");

  return 1;    /* error exit */
}

/**Function********************************************************************

  Synopsis    [Handle function for timeout.]

  Description [This function is called when the time out occurs.]

  SideEffects []

******************************************************************************/
static void
TimeOutHandle(void)
{
  longjmp(timeOutEnv, 1);
}


/**Function********************************************************************

  Synopsis    [Checks whether the network has multiple valued signals.]

  Description [Checks whether the network has multiple valued
  signals. Returns 1 if true, else 0.]

  SideEffects [None]

  SeeAlso []

******************************************************************************/
static int
TestIsNetworkMultipleValued(Ntk_Network_t *network)
{
  Ntk_Node_t *node;
  lsGen gen;
  Var_Variable_t *var;
  int numValues;

  Ntk_NetworkForEachNode(network,gen,node) {
    var = Ntk_NodeReadVariable(node);
    numValues = Var_VariableReadNumValues(var);
    if (numValues > 2)
      return 1;
  }
  return 0;
}
