/**CFile***********************************************************************

  FileName    [hrcModify.c]

  PackageName [hrc]

  Synopsis    [This files provides the basic functions concerned with
               modifying the hierarchy. This ]

  Description []

  SeeAlso     []

  Author      [Shaz Qadeer]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "hrcInt.h"

static char rcsid[] UNUSED = "$Id: hrcModify.c,v 1.4 2005/04/16 04:23:47 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/



/**Function********************************************************************

  Synopsis    [Adds a subckt to a model.]

  Description [If a subckt with the same instanceName is already present, the
               function returns FALSE. Otherwise, a new sub-circuit is
               allocated and initialized and TRUE is returned. Note that
               it is the user's responsibility to free the string
               instanceName. actualInputVars and actualOutputVars are
               pointer-copied.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_ModelAddSubckt(
  Hrc_Model_t *callerModel,
  Hrc_Model_t *calleeModel,
  char *instanceName,
  array_t *actualInputVars,
  array_t *actualOutputVars)
{
  Hrc_Subckt_t *subckt; 
  
  if(!st_is_member(callerModel->subcktTable, instanceName)) {
    subckt = ALLOC(Hrc_Subckt_t, 1);     
    subckt->model = calleeModel;
    subckt->instanceName = util_strsav(instanceName);
    subckt->actualInputVars = actualInputVars;
    subckt->actualOutputVars = actualOutputVars;
    st_insert(callerModel->subcktTable, subckt->instanceName, (char *) subckt);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************

  Synopsis    [Adds a name table to a model.]

  Description [The function adds a name table to the master node of a model.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
Hrc_ModelAddNameTable(
  Hrc_Model_t *model,
  Tbl_Table_t *table)
{
  array_insert_last(Tbl_Table_t *, model->masterNode->nameTables, table);
}

/**Function********************************************************************

  Synopsis    [Adds a child node to a node.]

  Description [The function checks if a child with the same instance name
               is already present in parent. If it is present, FALSE is
               returned. Otherwise, the child is added and TRUE is returned.
               Note that before using this function, child should have
               created using Hrc_NodeAlloc().]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_NodeAddChild(
  Hrc_Node_t *parent,
  Hrc_Node_t *child,
  array_t *actualInputs,
  array_t *actualOutputs)
{
  if(!st_is_member(parent->childTable, child->instanceName)) {
    st_insert(parent->childTable, child->instanceName, (char *) child);
    child->parentNode = parent;
    child->actualInputs = actualInputs;
    child->actualOutputs = actualOutputs;
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************

  Synopsis    [Adds a latch to a node.]

  Description [The function adds a latch to a node if the latch is not 
               present already and returns TRUE. Otherwise, it returns FALSE.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_NodeAddLatch(
  Hrc_Node_t *node,
  Hrc_Latch_t *latch)
{
  char *latchOutputName = Var_VariableReadName(latch->latchOutput);
  
  if(!st_is_member(node->latchTable, latchOutputName)) {
    st_insert(node->latchTable, latchOutputName, (char *) latch);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************

  Synopsis    [Adds a variable to a node.]
  
  Description [The function adds a variable to a node if the variable is not 
               present already and returns TRUE. Otherwise, it returns FALSE.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_NodeAddVariable(
  Hrc_Node_t *node,
  Var_Variable_t *var)
{
  char *varName = Var_VariableReadName(var);
  
  if(!st_is_member(node->varTable, varName)) {
    st_insert(node->varTable, varName, (char *) var);
    return TRUE;
  }
  else {
    return FALSE;
  }
}

/**Function********************************************************************
  
  Synopsis    [Deletes a variable from a node.]

  Description [If the variable is present in node it is deleted and TRUE is
               returned. Otherwise, nothing is done and FALSE is returned.
               Note that this function should only be used for deleting an
               internal variable of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_NodeDeleteVariable(
  Hrc_Node_t *node,
  Var_Variable_t *var)
{
  char *varName = Var_VariableReadName(var);
  
  return st_delete(node->varTable, &varName, &var);
}

/**Function********************************************************************

  Synopsis    [Deletes a latch from a node.]

  Description [If the latch is present in node it is deleted and TRUE is
               returned. Otherwise, nothing is done and FALSE is returned.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
boolean
Hrc_NodeDeleteLatch(
  Hrc_Node_t *node,
  Hrc_Latch_t *latch)
{
  char *latchOutputName = Var_VariableReadName(latch->latchOutput);
  
  return st_delete(node->latchTable, &latchOutputName, &latch);
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis    [Deletes a child from a node.]

  Description [If a child of the name childName exists in node, it is deleted
               from the childTable of node and a pointer to this child is
               returned. Otherwise, NULL is returned.]

  SideEffects []

  SeeAlso     [Hrc_TreeReplace()]

******************************************************************************/
Hrc_Node_t *
HrcNodeDeleteChild(
  Hrc_Node_t *node,
  char *childName)
{
  Hrc_Node_t *childNode;
  
  if(st_delete(node->childTable, &childName, &childNode)) {
    return childNode;
  }
  else {
    return NIL(Hrc_Node_t);
  }
}


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
























































































