/**CFile***********************************************************************

  FileName    [mvfaigUtil.c]

  PackageName [mvfaig]

  Synopsis    [Routines to create, manipulate and free multi-valued functions.]

  SeeAlso     [mvfAig.h]

  Author      [Mohammad Awedh]

  Copyright  [.]

******************************************************************************/

#include "mvfaigInt.h"

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Initializes the mvfAig package.]

  SideEffects []

  SeeAlso     [MvfAig_End]

******************************************************************************/
void
MvfAig_Init(void)
{
}


/**Function********************************************************************

  Synopsis    [Ends the mvfAig package.]

  SideEffects []

  SeeAlso     [MvfAig_Init]

******************************************************************************/
void
MvfAig_End(void)
{
}


/**Function********************************************************************

  Synopsis    [Allocates a multi-valued function of n components.]

  Description [Allocates a multi-valued function of n components.  Each
  component is initialized to the zero bAig.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
MvfAig_Function_t *
MvfAig_FunctionAlloc(
  int n)
{
  int      i;
  array_t *bAndInvArray = array_alloc(bAigEdge_t, n);

  for (i = 0; i < n; i++) {
    array_insert(bAigEdge_t, bAndInvArray, i, bAig_Zero);
  }
  return ((MvfAig_Function_t *) bAndInvArray);
}

/**Function********************************************************************

  Synopsis    [Frees a multi-valued output function.]

  Description [Frees a multi-valued output function. Does nothing if
  function is NULL.]
  
  SideEffects []

  SeeAlso     [MvfAig_FunctionAlloc]

******************************************************************************/
void
MvfAig_FunctionFree(
  MvfAig_Function_t *function)
{
  array_free((array_t *) function);
}

/**Function********************************************************************

  Synopsis    [Duplicates a multi-valued output function.]

  Description [Returns a new multi-valued output function, whose constituent
  MAIGs have been duplicated. Assumes that function is not NULL.] 

  SideEffects []

  SeeAlso     [Mvf_FunctionFree]

******************************************************************************/
MvfAig_Function_t *
MvfAig_FunctionDuplicate(
  MvfAig_Function_t *function)
{
  return ((MvfAig_Function_t *) array_dup((array_t *) function));
}


/**Function********************************************************************

  Synopsis    [Frees an array of multi-valued output functions.]

  Description [Frees an array of multi-valued output functions.  Does nothing
  if functionArray is NULL.]

  SideEffects []

  SeeAlso     [Mvf_FunctionFree]

******************************************************************************/
void
MvfAig_FunctionArrayFree(
  array_t *functionArray)
{
  int i;

  if (functionArray != NIL(array_t)) {
    for (i = 0; i < array_n(functionArray); i++) {
      MvfAig_Function_t *function = array_fetch(MvfAig_Function_t*, functionArray, i);
      MvfAig_FunctionFree(function);
    }
    array_free(functionArray);
  }
}


/**Function********************************************************************

  Synopsis    [Returns the number of components of a multi-valued function.]

  Description [Returns the number of components of a multi-valued function.
  This is the same number as the value of the parameter passed to
  Mvf_FunctionAlloc.]

  SideEffects []

  SeeAlso     [Mvf_FunctionAlloc]

******************************************************************************/
int
MvfAig_FunctionReadNumComponents(
  MvfAig_Function_t *function)
{
  return (array_n((array_t *) function));
}


/**Function********************************************************************

  Synopsis [Returns the ith component of a multi-valued function.]

  Description [Returns the mAndInv giving the minterms for which a
  multi-valued function evaluates to its ith value. ]

  SideEffects []

  SeeAlso     []

******************************************************************************/
mAigEdge_t
MvfAig_FunctionReadComponent(
  MvfAig_Function_t *function,
  int i)
{
  bAigEdge_t component = array_fetch(bAigEdge_t, (array_t *) function, i);
  return (component);
}


/**Function********************************************************************

  Synopsis [Creates the multi-output function for a variable.]

  Description [Given a variable, creates a function with as many components as
  values of the variable.  The ith component of the function is true exactly
  when the variable is equal to the ith value (i.e. fi(x) = (x==i), where x is
  the variable specified by mddId).  For the case where x is binary valued,
  the result is \[!x, x\]. Assumes that mddId is non-negative.]

  SideEffects []

  SeeAlso [Mvf_FunctionAlloc]

******************************************************************************/
MvfAig_Function_t *
MvfAig_FunctionCreateFromVariable(
  mAig_Manager_t *manager,
  mAigEdge_t mVarId)
{
  int        i;
  array_t   *mVarList    = mAigReadMulVarList(manager);
  mAigMvar_t mVar;
  array_t   *result;

  assert( (mVarId >= 0) && (mVarId <= array_n(mVarList)));
  mVar   = array_fetch(mAigMvar_t, mVarList, mVarId);
  result = array_alloc(bAigEdge_t, mVar.values);  /* array of all possible values of this variable */
  for(i = 0; i < mVar.values; i++) {
    bAigEdge_t literal;

    literal = mAig_EquC(manager, mVarId, i);       /* AndInv graph = 1 if the variable = i*/
    array_insert(bAigEdge_t, result, i, literal);
    {
      /*
	The name of the node in AIG is node name = node value.
       */
      /*
      char *valueStr = util_inttostr(i);
      char *name     = util_strcat3(mVar.name,"=", valueStr);
      char *tmpName;
      int  index;
      
      bAig_NodeSetName(manager, literal, name);
      FREE(valueStr);
      */
    }
  }
  return ((MvfAig_Function_t *) result);
} 


/**Function********************************************************************

  Synopsis    [Adds a set of minterms to the ith component of a function.]

  Description [Adds a set of minterms, represented by the onset of an mAig g,
  to the onset of the ith component of a function.  The mAig g is not freed.]

  SideEffects []

  SeeAlso     [Mvf_FunctionAlloc]

******************************************************************************/
void
MvfAig_FunctionAddMintermsToComponent(
  mAig_Manager_t    *manager,
  MvfAig_Function_t *function,
  int               i,
  mAigEdge_t        g)
{
  mAigEdge_t oldComponent;
  mAigEdge_t newComponent;
  array_t    *mAigArray = (array_t *) function;

  assert((i >= 0) && (i < array_n(mAigArray)));

  oldComponent = array_fetch(mAigEdge_t , mAigArray, i);
  newComponent = mAig_Or(manager, oldComponent, g);
  array_insert(mAigEdge_t , mAigArray, i, newComponent);
}

/**Function********************************************************************

  Synopsis    [Returns the set of minterms on which two functions agree.]

  Description [Returns the set of minterms on which two functions agree.  For
  f = \[f1, f2, ..., fn\] and g = \[g1, g2, ..., gn\], the returned set is:
  AND(i = 1, ..., n) (fi XNOR gi).  For the special case where f and g are
  binary valued, this function computes (f XNOR g).  It is an error if the two
  functions have a different number of components.]

  SideEffects []

******************************************************************************/
mAigEdge_t 
MvfAig_FunctionsComputeEquivalentSet(
  mAig_Manager_t *manager, 
  MvfAig_Function_t *function1,
  MvfAig_Function_t *function2)
{
  int          i;
  array_t     *mAigArray1     = (array_t *) function1;
  array_t     *mAigArray2     = (array_t *) function2;
  int          numComponents = array_n(mAigArray1);
  mAigEdge_t   product       = mAig_One;

  assert(numComponents == array_n(mAigArray1));
  assert(numComponents == array_n(mAigArray2));
  
  for (i = 0; i < numComponents; i++) {
    mAigEdge_t mAig1 = array_fetch(mAigEdge_t, mAigArray1, i);
    mAigEdge_t mAig2 = array_fetch(mAigEdge_t, mAigArray2, i);
    mAigEdge_t xnor = mAig_Eq(manager, mAig1, mAig2);
    mAigEdge_t temp = product;

    product = mAig_And(manager, temp, xnor);
  }

  return product;
}

/**Function********************************************************************

  Synopsis    [Computes the domain of a multi-valued function.]

  Description [Returns an mAig representing the set of minterms which turn on
  some component of a function.  In other words, returns the union of the
  onsets of the components.  The domain is the tautology if and only if the
  function is completely specified.]

  SideEffects []

  SeeAlso     [Mvf_FunctionTestIsCompletelySpecified]

******************************************************************************/
mAigEdge_t 
MvfAig_FunctionComputeDomain(
  mAig_Manager_t *manager,
  MvfAig_Function_t *function)
{
  int          i;
  array_t     *mAigArray     = (array_t *) function;
  int          numComponents = array_n(mAigArray);
  mAigEdge_t   sum           = mAig_Zero;

  for (i = 0; i < numComponents; i++) {
    mAigEdge_t ithComponent = array_fetch(mAigEdge_t , mAigArray, i);
    mAigEdge_t temp = sum;
    
    sum = mAig_Or(manager, temp, ithComponent);
  }
  return sum;
}

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/









