/**CHeaderFile*****************************************************************

  FileName    [ord.h]

  PackageName [ord]

  Synopsis    [Routines for ordering MDD variables of a flattened network.]

  Description [The routines in this package relate to ordering MDD variables
  corresponding to the nodes of a network.  The enumerated type Ord_OrderType
  is used to specify to which set of nodes certain ordering routines should
  be applied.  The following matrix shows which node types are included in
  each ordering type:
  <pre>
                  |             Ord_OrderType
   node type      |  All   InputAndLatch   NextStateNode
  ----------------+--------------------------------------- 
  primary input   |   x          x 
  pseudo input    |   x          x
  latch           |   x          x
  shadow          |   x          x               x  
  combinational   |   x
  
  </pre>

  In addition, the order type Partial can be used to specify an arbitrary
  subset of nodes.<p>

  There are various methods for ordering the roots of a network, and for
  ordering the nodes of a network.  These are explained in the documentation
  for the static_order command.]
  
  Author      [Adnan Aziz, Tom Shiple, Serdar Tasiran]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: ord.h,v 1.10 2002/09/08 23:26:15 fabio Exp $]

******************************************************************************/

#ifndef _ORD
#define _ORD

/*---------------------------------------------------------------------------*/
/* Nested includes                                                           */
/*---------------------------------------------------------------------------*/
#include <time.h>
#include <math.h>
#include "ntk.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                     */
/*---------------------------------------------------------------------------*/
/**Enum************************************************************************

  Synopsis    [Used to specify particular subsets of network nodes.]

  Description [Used to specify particular subsets of network nodes. This
  enumerated type is used to specify which nodes are to be ordered, or which
  nodes are supplied in an ordering file. "All" nodes of a network is exactly
  the set returned by Ntk_NetworkReadNodes (this includes shadow nodes). ]

  SeeAlso     [Ord_NetworkOrderVariables]

******************************************************************************/
typedef enum {
  Ord_All_c,           /* all nodes in network */
  Ord_NextStateNode_c, /* latch NS nodes */
  Ord_Partial_c,       /* arbitrary subset of all nodes in network */
  Ord_InputAndLatch_c, /* primary and pseudo inputs, latches, and latch NS */
  Ord_Unassigned_c     /* not specified */
} Ord_OrderType;


/**Enum************************************************************************

  Synopsis    [Used to specify a particular method to order the roots of a network.]

  Description [Used to specify a particular method to order the roots of a
  network.  The nodes of a network are explored in DFS order, starting from
  the ordered list of roots.]

  SeeAlso     [Ord_NetworkOrderRoots]

******************************************************************************/
typedef enum {
  Ord_RootsByDepth_c,   /* decreasing order of logic depth */
  Ord_RootsByPerm_c,    /* permutation that minimizes a communication-based
                           cost function */
  Ord_RootsByDefault_c  /* hardcoded default method (one of the above) */
} Ord_RootMethod;


/**Enum************************************************************************

  Synopsis    [Used to specify a particular method to order the nodes of a network.]

  Description [Used to specify a particular method to order the nodes of a
  network.  These methods produce a total ordering on all of the nodes of a
  network, including the next state nodes.]

  SeeAlso     [Ord_NetworkOrderNodes]

******************************************************************************/
typedef enum {
  Ord_NodesByInterleaving_c, /* Interleaving algorithm of Fujii et al. */
  Ord_NodesByMergingLeft_c,  /* DFS by depth, merging left orderings of fanins */
  Ord_NodesByMergingRight_c, /* DFS by depth, merging right orderings of fanins */
  Ord_NodesByAppending_c,    /* DFS by depth, appending nodes as visited */
  Ord_NodesByDefault_c       /* hardcoded default method (one of the above) */
} Ord_NodeMethod;


/**Enum************************************************************************

  Synopsis    [Used to specify a particular method to merge two lists.]

  SeeAlso     [Ord_ListMergeListUsingTable]

******************************************************************************/
typedef enum {
  Ord_ListMergeLeft_c, /* merge left two lists */
  Ord_ListMergeRight_c /* merge right two lists */
} Ord_ListMergeMethod;


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN void Ord_Init(void);
EXTERN void Ord_End(void);
EXTERN boolean Ord_NetworkTestAreVariablesOrdered(Ntk_Network_t * network, Ord_OrderType orderType);
EXTERN int Ord_NetworkPrintVariableOrder(FILE * fp, Ntk_Network_t * network, Ord_OrderType orderType);
EXTERN char ** Ord_NetworkGetCombInputNamesInOrder(Ntk_Network_t *network);
EXTERN boolean Ord_FileReadNodeList(FILE * fp, Ntk_Network_t * network, lsList * nodeList, int verbose);
EXTERN void Ord_NetworkOrderVariables(Ntk_Network_t *network, Ord_RootMethod rootOrderMethod, Ord_NodeMethod nodeOrderMethod, boolean nsAfterSupport, Ord_OrderType generatedOrderType, Ord_OrderType suppliedOrderType, lsList suppliedNodeList, int verbose);
EXTERN lsList Ord_NetworkOrderNodes(Ntk_Network_t *network, Ord_RootMethod rootOrderMethod, Ord_NodeMethod nodeOrderMethod, boolean nsAfterSupport, Ord_OrderType generatedOrderType, Ord_OrderType suppliedOrderType, lsList suppliedNodeList, int verbose);
EXTERN void Ord_NetworkAssignMddIdForNode(Ntk_Network_t *network, Ntk_Node_t *node);
EXTERN void Ord_ListMergeLeftListUsingTable(lsList list1, lsList list2, st_table *dataToHandle1);
EXTERN void Ord_ListMergeRightListUsingTable(lsList list1, lsList list2, st_table *dataToHandle1);
EXTERN void Ord_ListMergeListUsingTable(lsList list1, lsList list2, st_table *dataToHandle1, Ord_ListMergeMethod method);
EXTERN void Ord_ListAppendList(lsList list1, lsList list2);
EXTERN void Ord_ListMergeList(lsList list1, lsList list2, Ord_ListMergeMethod method);
EXTERN lsList Ord_NetworkOrderRoots(Ntk_Network_t *network, Ord_RootMethod rootMethod, lsList partialOrder, boolean verbose);

/**AutomaticEnd***************************************************************/

#endif /* _ORD */
