/**CHeaderFile*****************************************************************

  FileName    [sim.h]

  PackageName [sim]

  Synopsis    [Simulation of a flattened network.]

  Description [The sim package provides functions for simulation. It also
  provides the "simulate" command in the VIS environment. The simulate command
  performs a simulation of a network or a part of a network.
  <p>
  sim conceives simulation through three operations:
  <p>
  1- Build simulation vectors.
  <p>
  2- Simulate.
  <p>
  3- Print the result.
  <p>
  Exported functions make it possible to build an internal data-structure, and
  perform the three operations given above. Simulation vectors may be provided
  by the user. Using exported functions, it is also possible to generate
  simulation vectors randomly or reading it from a file. Low level functions
  are also available that evaluate, for example, a network node using a
  simulation vector. An example of a simulation vectors file may be generated
  using the simulate command with random-vectors-generation option, in VIS.
  <p>
  WARNING : In order to improve the speed of simulation, it is assumed that
  each combinational output function is completely specified, and that each
  simulation vector specifies a minterm over the combinational inputs this
  assumption is not checked. Incorrect results may be produced if this
  assumption is violated.]

  Author      [Shaker Sarwary and Tom Shiple]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: sim.h,v 1.5 2002/09/08 23:33:35 fabio Exp $]

******************************************************************************/

#ifndef _SIM
#define _SIM

/*---------------------------------------------------------------------------*/
/* Nested includes                                                           */
/*---------------------------------------------------------------------------*/
#include "ntk.h"

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/* Sim main Data structure type */

typedef struct SimSimStruct Sim_Sim_t;

/**Enum************************************************************************

  Synopsis    [Pseudo-inputs source.]

  Description [Pseudo-inputs source. There are 4 possible source :
  ---> undefined : If input vectors are read from a file, then user must
  specify the pseudo-inputs. If vectors are generated randomly, then
  pseudo-inputs are chosen randomly.
  ---> user : Pseudo-inputs are specified in the input vectors file.
  ---> random : Pseudo-inputs are generated randomly. If vectors are read from
  a file, this option cannot be used.
  ---> first : The first value of pseudo-input in the array of value of the
  variable is chosen. If vectors are read from a file, this option cannot be used.

******************************************************************************/
typedef enum {
  Sim_Undef_c,
  Sim_User_c,
  Sim_Random_c,
  Sim_First_c
} Sim_PseudoSrc;


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN void Sim_SimPrint(Sim_Sim_t * sim, FILE * of, boolean printMode, int printInputsFlag, int printOutputsFlag, int printPseudoInputsFlag, int printStatesFlag);
EXTERN Sim_Sim_t * Sim_FileParseDeclaration(Ntk_Network_t * network, FILE * fp, char * inputFile, boolean verbose);
EXTERN int Sim_FileParseVectors(FILE * fp, Sim_Sim_t * sim, int num);
EXTERN void Sim_Init(void);
EXTERN void Sim_End(void);
EXTERN Sim_Sim_t * Sim_SimCreate(Ntk_Network_t * network, st_table * nodeToMvfTable, char * inputFile, int lineNumber, array_t * nodesArray, int currentStateHead, int internalPartitionHead, int nextStateHead, int outputHead, array_t * initState, array_t * vectorArray, boolean verbose);
EXTERN void Sim_SimReset(Sim_Sim_t * sim);
EXTERN void Sim_SimFree(Sim_Sim_t * sim);
EXTERN array_t * Sim_NetworkCreateNodesArray(Ntk_Network_t * network, int * currentStateHead, int * internalPartitionHead, int * nextStateHead, int * outputHead);
EXTERN void Sim_SimGenerateRandomVectors(Sim_Sim_t * sim, int numberVector, Sim_PseudoSrc pseudoInputSource);
EXTERN void Sim_SimGenerateRandomInitState(Sim_Sim_t * sim);
EXTERN void Sim_SimSimulate(Sim_Sim_t * sim);
EXTERN int Sim_nodeToMvfTableEvaluateNode(st_table * nodeToMvfTable, Ntk_Node_t * node, mdd_t * vectorMdd);
EXTERN array_t * Sim_nodeToMvfTableEvaluateNodesArray(st_table * nodeToMvfTable, array_t * nodesArray, mdd_t * vectorMdd);
EXTERN mdd_t * Sim_RandomSimulate(Ntk_Network_t * network, int num, boolean verbose);
EXTERN Ntk_Network_t * Sim_SimReadNetwork(Sim_Sim_t * sim);
EXTERN st_table * Sim_SimReadNodeToMvfTable(Sim_Sim_t * sim);
EXTERN char * Sim_SimReadInputFile(Sim_Sim_t * sim);
EXTERN int Sim_SimReadLineNumber(Sim_Sim_t * sim);
EXTERN array_t * Sim_SimReadNodesArray(Sim_Sim_t * sim);
EXTERN int Sim_SimReadCurrentStateHead(Sim_Sim_t * sim);
EXTERN int Sim_SimReadNextStateHead(Sim_Sim_t * sim);
EXTERN int Sim_SimReadOutputHead(Sim_Sim_t * sim);
EXTERN array_t * Sim_SimReadInitState(Sim_Sim_t * sim);
EXTERN array_t * Sim_SimReadVectorArray(Sim_Sim_t * sim);
EXTERN boolean Sim_SimTestIsVerbose(Sim_Sim_t * sim);
EXTERN st_table * Sim_NetworkBuildNodeToMvfTable(Ntk_Network_t *network, array_t *nodesArray, int internalPartitionHead, int nextStateHead);

/**AutomaticEnd***************************************************************/

#endif /* _SIM */


