/**CFile***********************************************************************

  FileName    [spfdAPI.c]

  PackageName [spfd]

  Synopsis [Routines to read, delete and update data structures used
  in the package.]

  Description [Routines to read, delete and update data structures used
  in the package.]

  SeeAlso     [spfdUtil.c spfdClean.c]

  Author      [Balakrishna Kumthekar]

  Copyright [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#include "spfdInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

extern int sfpdVerbose; 
extern float alpha; 

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Returns whether the node is locked.]

  SideEffects        [None]

******************************************************************************/
boolean
SpfdNodeReadLocked(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  
  return nodeData->locked;
  
} /* End of SpfdNodeReadLocked */


/**Function********************************************************************

  Synopsis           [Read the auxillary BDD id associated with the node.]

  SideEffects        [None]

******************************************************************************/
int
SpfdNodeReadAuxId(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  
  return nodeData->auxId;
  
} /* End of SpfdNodeReadAuxId */


/**Function********************************************************************

  Synopsis           [Set the auxillary BDD id for the node]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetAuxId(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  int auxId)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  nodeData->auxId = auxId;

} /* End of SpfdNodeSetAuxId */


/**Function********************************************************************

  Synopsis           [Read the node's spfd.]

  SideEffects        [None]

******************************************************************************/
bdd_node *
SpfdNodeReadSpfd(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  
  return nodeData->spfd;
  
} /* End of SpfdNodeReadSpfd */


/**Function********************************************************************

  Synopsis           [Free the BDD representing the spfd of the node.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeDeleteSpfd(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->spfd)
    bdd_recursive_deref(applData->ddManager,nodeData->spfd);
  nodeData->spfd = NIL(bdd_node);
  
  return;
  
} /* End of SpfdNodeDeleteSpfd */


/**Function********************************************************************

  Synopsis           [Set the node's spfd]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetSpfd(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  bdd_node *spfd)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->spfd) {
    (void) fprintf(vis_stdout,
		   "** spfd warning: <%s: set spfd> Possible memory leak.\n",
		   Ntk_NodeReadName(node));
  }
  nodeData->spfd = spfd;
  
} /* End of SpfdNodeSetSpfd */


/**Function********************************************************************

  Synopsis [Read the array of BDD variables (parameters) associated
  with the spfd of the node.]

  SideEffects        [None]

******************************************************************************/
bdd_node **
SpfdNodeReadParameters(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  return nodeData->parameters;
  
} /* End of SpfdNodeReadParameters */


/**Function********************************************************************

  Synopsis [Free the array of BDD variables (parameters) associated
  with the spfd of the node.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeDeleteParameters(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->parameters)
    FREE(nodeData->parameters);

  nodeData->parameters = NIL(bdd_node *);
  nodeData->numParams = 0;
  
  return;
  
} /* End of SpfdNodeDeleteParameters */


/**Function********************************************************************

  Synopsis [Set the array of BDD variables (parameters) associated
  with the spfd of the node.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetParameters(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  bdd_node **parameters,
  int numParams)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->parameters) {
    (void) fprintf(vis_stdout,
		   "** spfd warning: <%s: set parameters> Possible memory leak.\n",
		   Ntk_NodeReadName(node));
  }

  nodeData->parameters = parameters;
  nodeData->numParams = numParams;
  
} /* End of SpfdNodeSetParameters */


/**Function********************************************************************

  Synopsis           [Read the sorted array of fanin nodes.]

  SideEffects        [None]

******************************************************************************/
array_t *
SpfdNodeReadFaninOrder(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  
  return nodeData->faninOrder;
  
} /* End of SpfdNodeReadFaninOrder */


/**Function********************************************************************

  Synopsis           [Delete the sorted array of fanin nodes.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeDeleteFaninOrder(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->faninOrder)
    array_free(nodeData->faninOrder);

  nodeData->faninOrder = NIL(array_t);
  
  return;
  
} /* End of SpfdNodeDeleteFaninOrder */


/**Function********************************************************************

  Synopsis [Set the array faninOrder required during SPFD computation.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetFaninOrder(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  array_t *faninOrder)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  if (nodeData->faninOrder) {
    (void) fprintf(vis_stdout,
                   "** spfd warning: <%s: set fanin order> Possible memory leak.\n",
		   Ntk_NodeReadName(node));
  }
  nodeData->faninOrder = faninOrder;

} /* End of SpfdNodeSetFaninOrder */

/**Function********************************************************************

  Synopsis [Read the global BDD of the node. The global BDD is derived
  from the node's spfd.]

  SideEffects        [SpfdNodeDeleteGlobalAlternative]

******************************************************************************/
bdd_node *
SpfdNodeReadGlobalAlternative(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  
  return nodeData->alternative;
  
} /* End of SpfdNodeReadGlobalAlternative */


/**Function********************************************************************

  Synopsis [Delete the global BDD of the node. The global BDD is
  derived from the node's spfd.]

  SideEffects [None]

  SeeAlso            [SpfdNodeReadGlobalAlternative]

******************************************************************************/
void
SpfdNodeDeleteGlobalAlternative(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->alternative)
    bdd_recursive_deref(applData->ddManager,nodeData->alternative);

  nodeData->alternative = NIL(bdd_node);
  
  return;
  
} /* End of SpfdNodeDeleteGlobalAlternative */


/**Function********************************************************************

  Synopsis           [Set the BDD as the node's global BDD.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetGlobalAlternative(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  bdd_node *alternative)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  if (nodeData->alternative) {
    (void) fprintf(vis_stdout,
		   "** spfd warning: <%s: set global alternative>"
		   " Possible memory leak.\n",
		   Ntk_NodeReadName(node));
  }
  nodeData->alternative = alternative;
  
} /* End of SpfdNodeSetGlobalAlternative */


/**Function********************************************************************

  Synopsis [Read the local BDD, which is in terms of the node's
  immediate fanin.]

  SideEffects        [None]

******************************************************************************/
bdd_node *
SpfdNodeReadLocalAlt(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  return nodeData->localAlt;
  
} /* End of SpfdNodeReadLocalAlt */


/**Function********************************************************************

  Synopsis [Delete the local BDD, which is in terms of the node's
  immediate fanin.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeDeleteLocalAlt(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  if (nodeData->localAlt)
    bdd_recursive_deref(applData->ddManager,nodeData->localAlt);

  nodeData->localAlt = NIL(bdd_node);
  
  return;
  
} /* End of SpfdNodeDeleteLocalAlt */

 
/**Function********************************************************************

  Synopsis [Set localAlt as the local BDD. localAlt is in terms of the
  node's immediate fanin.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetLocalAlt(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  bdd_node *localAlt)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);
  if (nodeData->localAlt) {
    (void) fprintf(vis_stdout,
		   "** spfd warning: <%s: set local alt> Possible memory leak.\n",
		   Ntk_NodeReadName(node));
  }
  nodeData->localAlt = localAlt;
  
} /* End of SpfdNodeSetLocalAlt */
  

/**Function********************************************************************

  Synopsis           [Set the node as locked.]

  SideEffects        [None]

******************************************************************************/
void
SpfdNodeSetLocked(
  SpfdApplData_t *applData,
  Ntk_Node_t *node,
  boolean locked)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  nodeData->locked = locked;
  
} /* End of SpfdNodeSetLocked */


/**Function********************************************************************

  Synopsis [Read the number of parameters associated with the node's
  SPFD.]

  SideEffects        [None]

******************************************************************************/
int
SpfdNodeReadNumParams(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  st_table *nodeToData = applData->nodeToData;
  SpfdNodeData_t *nodeData;

  st_lookup(nodeToData,(char *)node,&nodeData);

  return nodeData->numParams;
  
} /* End of SpfdNodeReadNumParams */

/**Function********************************************************************

  Synopsis [This is used to get the index of the source of the net
  data structure used in VPR.]

  SideEffects        [None]

******************************************************************************/
int
SpfdNodeReadNetIndex(
  SpfdApplData_t *applData,
  Ntk_Node_t *node)
{
  SpfdPlaceData_t *placeData;
  int index = -1;
  
  placeData = applData->placeData;
  st_lookup(placeData->nodeToNetIndex,(char *)node,&index);

  return index;
  
} /* End of SpfdNodeReadNetIndex */

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


