/**CHeaderFile*****************************************************************

  FileName    [truesimInt.h]

  PackageName [truesim]

  Synopsis    [Internal data structures for the truesim package.]

  Description [Internal data structures for the truesim package.]

  SeeAlso     [truesim.h]

  Author      [Balakrishna Kumthekar]

  Copyright   [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#ifndef _TRUESIMINT
#define _TRUESIMINT

#include "truesim.h"
#include <string.h>

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define TRUESIM_NETWORK_APPL_KEY "Truesim_NetworkApplKey"
#define TRUESIM_ERROR_VALUE -1

/* Different types of event. */
#define SIM_EVENT_0   0
#define SIM_EVENT_1   1
/* Threshold events are book-keeping events. They are not 
 * simulation events */
#define THRESHOLD_1   2 
#define THRESHOLD_2   3    
#define THRESHOLD_3   4
#define THRESHOLD_I   5
#define SIM_EVENT_C   6 /* Even to describe cancellation of a previously
			 * scheduled event */

/* Constant used during random number generation */
#define MODULUS1 2147483563 /* Borrowed from util package */


/* Basic wheel parameters for the event queue. */
#define LOG_HOR_LEN         3
#define LOG_WHEEL_SIZE      10
#define MAX_TIME            2147483647 /* 2^32 -1  */

/* Derived wheel parameters */
#define HOR_LEN   (1 << LOG_HOR_LEN) /* # of time 'ticks' in
				      * each slot of the time wheel. */
#define WHEEL_SIZE  (1 << LOG_WHEEL_SIZE) /* # slots in the wheel */
#define PERIOD  (HOR_LEN * WHEEL_SIZE) /* # time ticks in the wheel */
#define HALF_PERIOD  (PERIOD / 2)

/* # events to allocate at a time. */
#define MEM_CHUNK 1023


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Structure to store simulation events.]

  Description [Structure to store simulation events. The Event structure is
  used only for real-delay simulation.]

  SeeAlso     [EventQueue]

******************************************************************************/
typedef struct Event {
  long type; /* One of the events defined above */
  long time; /* time to simulate the event */
  Ntk_Node_t *gate; /* node that needs to be simulated */
  struct Event *horizontal; /* Pointer to other events in queue */
  struct Event *vertical;
  struct Event *bottom;
} Event;


/**Struct**********************************************************************

  Synopsis    [Event queue.]

  Description [Event queue. The event queue is a converging list data structure
  used to access simulation events as the time progresses. It is based on the
  concept of circular buffer (time wheel). Each slot in the wheel
  represents a range of time values, i.e., a time interval.
  Since there are events for different times in the same list, the list
  is kept sorted. Each element in the list is itself a vertical list of
  events. Events in the vertical list are all scheduled to be simulated at
  a given time within the time interval mapped by the slot. The following 
  shows the converging list data structure.

  wheel      Vertial lists
    |             |
    v             v
   -------      _____   _____  
   |slot 0| --> |   |-->|   |-->         <---- Horizontal list
   -------      -----   -----
                  |       |
                  v       v
                -----  
                |   |
                ----- 
                  |  
                  v 
	          .
		  . 
		  . 
   -------      _____   _____  
   |slot 1| --> |   |-->|   |-->         <---- Horizontal list
   -------      -----   -----
                  |       |
                  v       v
                -----   ----- 
                |   |-->|   |-->
                -----   -----
                  |       |
                  v       v
	          .       .
		  .       .
		  .       .

  All the horizontal lists point to an event, 'Threshold1', th1.
  Event 'th1' points to a list of events that do not fit in the current time
  interval represented by the wheel. Event 'thi' is the last event,
  i.e., event at time infinity.

  The events th1, th2, th3 and thi are 'control events' and not simulation
  events. They help in managing event queue. For more info. please refer
  to InitQueue in truesimMain.c.

  freeList and nextFre aid in memory management.]

  SeeAlso     [Event]

******************************************************************************/
typedef struct EventQueue {
  Event *th1,*th2,*th3,*thi;
  Event **wheel;
  Event *current;
  long time; /* Current time */
  Event **freeList;
  Event *nextFree;
} EventQueue;


/**Struct**********************************************************************

  Synopsis [Structure to handle simulation needs of the network. This
  data structure needs to be initialized for each node in the network.]

  Description [Structure to handle simulation needs of the network. This
  data structure needs to be initialized for each node in the network.]

  SeeAlso     [Event]

******************************************************************************/
typedef struct TrueSim_t {
  int depth; /* Node's topological depth */
  float delay; /* Node delay */
  char value; /* Node output value */
  float prob; /* Prob. of being 1 */
  float switching; /* Number of switchings */
  float load; /* Load at the output of the node */
  Event *event; /* Pointer back to the event. In the ideal case this should be
		 * a member of the Ntk_Node_t data structure. */
} TrueSim_t;


/**Struct**********************************************************************

  Synopsis [Data structure for the truesim package.]

  Description [Data structure for the truesim package. It holds information
  that remains unchanged during a given simulation run.]

  SeeAlso     [TrueSim_t]

******************************************************************************/
typedef struct TruesimInfoStruct {
  boolean trueDelay; /* TRUE for true delay simulation, FALSE for zero delay */
  int repeatHeader; /* Print PI/PO names along with their values during
		     * simulation */
  int truesimVerbose; /* Verbosity level */
  array_t *depthArray; /* Sorted array (according to top. depth) of array of
			* network nodes */
  st_table *nodeToSimTable; /* node --> TrueSim_t table */
}Truesim_Info_t;

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**Macro***********************************************************************

   Synopsis     [Given a time, the macro returns the slot in the time wheel.]

   SideEffects  [None]

   SeeAlso      [TYPE, TIME, NEXT_HOR, NEXT_VER]

******************************************************************************/

#define WHEEL_POS(t)	(((t) >> LOG_HOR_LEN) & ((1 << LOG_WHEEL_SIZE) - 1))


/**Macro***********************************************************************

  Synopsis     [Allocate a new event or reuse a free event form the freelist.]

  SideEffects  [nextFree is changed.]

  SeeAlso      []

******************************************************************************/

#define EVENT_ALLOC(queue,new) \
  if ((queue)->nextFree == NIL(Event)) { \
       NewPage(queue); \
  } \
  new = (queue)->nextFree; \
  (queue)->nextFree = (queue)->nextFree->horizontal

/**Macro***********************************************************************

  Synopsis     [Return memory associated with an event to the freeList.]

  SideEffects  [nextFree is updated.]

  SeeAlso      []

******************************************************************************/

#define EVENT_FREE(queue,old) \
(old)->horizontal = (queue)->nextFree; \
(queue)->nextFree = (old);

/* Various macros to get details of an event */
#define TYPE(_event) (_event)->type
#define TIME(_event) (_event)->time
#define NEXT_HOR(_event) (_event)->horizontal
#define NEXT_VER(_event) (_event)->vertical
#define NEXT_BOT(_event) (_event)->bottom
#define GATE(_event) (_event)->gate


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN void TruesimEndSimulation(void *data);
EXTERN int TruesimSimulateNetwork(Ntk_Network_t *network, char *simFile, char *probFile, char *delayFile, char *dumpFile, boolean trueDelay, int genVectors, int N);
EXTERN char TruesimEvaluateNode(Ntk_Node_t *node, graph_t *partition, bdd_manager *ddManager, st_table *nodeToSimTable);
EXTERN st_table * TruesimNetworkReadSimTable(Ntk_Network_t *network);
EXTERN array_t * TruesimNetworkReadDepthArray(Ntk_Network_t *network);
EXTERN void TruesimInitializeActivityFields(Ntk_Network_t *network, st_table *nodeToSimTable);
EXTERN bdd_t * TruesimComputeFaninMinterm(bdd_manager *ddManager, Ntk_Node_t *node, st_table *nodeToSimTable);
EXTERN void TruesimPrintNameHeader(Ntk_Network_t *network);
EXTERN void TruesimPrintNetworkNodeLogicState(Ntk_Network_t *network);
EXTERN void TruesimReadDelayFile(Ntk_Network_t *network, char *delayFile, st_table *nodeToSimTable);
EXTERN void TruesimWarmUpPatternSimulate(Ntk_Network_t *network, array_t *inputArray, char *vector);

/**AutomaticEnd***************************************************************/

#endif /* _TRUESIMINT */
