/**CFile***********************************************************************

  FileName    [truesimZero.c]

  PackageName [truesim]

  Synopsis    [Routines to perform zero-delay vector simulation.]

  Author      [Balakrishna KumthekarK] <kumtheka@colorado.edu>]

  Copyright [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#include "truesimInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/* Global variables used in truesim package. */
extern int truesimVerbose;
extern int truesimRptHeader;

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void SetInputValues(array_t *inputArray, char *vector, st_table *nodeToSimTable);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis [This function performs BDD based zero-delay vector simulation.]

  Description [This function performs BDD based zero-delay vector
  simulation. The nodes in the network are simulated according to their
  topological depth. ]

  SideEffects [None]

  SeeAlso []

******************************************************************************/
int
Truesim_ZeroDelayPatternSimulate(
  Ntk_Network_t *network,
  array_t *inputArray,
  array_t *patternArray)
{
  graph_t *partition;
  TrueSim_t *sim;
  st_table *nodeToSimTable;
  Ntk_Node_t *node;
  array_t *depthArray;
  bdd_manager *ddManager = Ntk_NetworkReadMddManager(network);
  lsGen gen;
  char prev,next;
  long numVectors = array_n(patternArray);
  int maxDepth;
  int i,j,k;
  
  nodeToSimTable = TruesimNetworkReadSimTable(network);
  if (nodeToSimTable == NIL(st_table)) {
    (void) fprintf(vis_stderr,
		   "** truesim error: Simulation structures not initialized\n");
    (void) fprintf(vis_stderr,
		   "** truesim error: Call Truesim_InitializeSimulation before ");
    (void) fprintf(vis_stderr,"calling this function.\n");
    return -1;
  }

  depthArray = TruesimNetworkReadDepthArray(network);
  maxDepth = array_n(depthArray);
  
  partition = Part_NetworkReadPartition(network);

  /* Initialize the prob and switching fields of the simulation structure for
     each node in the network. */
  TruesimInitializeActivityFields(network,nodeToSimTable);

  /* Warm up simulation */
  TruesimWarmUpPatternSimulate(network,inputArray,
				  array_fetch(char *,patternArray,0));

  for (i = 1; i < numVectors; i++) {
    char *vector;
    vector = array_fetch(char *,patternArray,i);
    /* Set the input nodes to the input pattern vector. Do not update
       the switching field of the sim structure if it is the first
       vector */
    SetInputValues(inputArray,vector,nodeToSimTable);
    for (j = 1; j < maxDepth; j++) {
      array_t *nodeList;
      
      nodeList = array_fetch(array_t *,depthArray,j);
      arrayForEachItem(Ntk_Node_t *,nodeList,k,node) {
	TrueSim_t *sim;

	st_lookup(nodeToSimTable,(char *)node,&sim);
	next =  TruesimEvaluateNode(node,partition,ddManager,
				    nodeToSimTable);
	if (i != 0) {
	  prev = sim->value;
	  if (prev != next)
	    (sim->switching)++;
	}
	sim->value = next;
	if (next == '1')
	  (sim->prob)++;
      }
    }
    if (truesimVerbose) {
      if (truesimRptHeader != -1) {
	if (i % truesimRptHeader == 0) 
	  TruesimPrintNameHeader(network);
      }
      TruesimPrintNetworkNodeLogicState(network);
    }
  }

  /* Update the statistics for each node */
  Ntk_NetworkForEachNode(network,gen,node) {
    if (!st_lookup(nodeToSimTable,(char *)node,&sim)) {
      (void) fprintf(vis_stderr,
		     "** truesim fatal: In Truesim_ZeroDelayPatternSimulate\n");
      assert(0);
    }
    sim->switching /= ((float) array_n(patternArray));
    sim->prob /= ((float) array_n(patternArray));
  }

  return 1;

} /* End of Truesim_ZeroDelayPatternSimulate */

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis [Simulates a single pattern to initialize initial state of network
  nodes.]

  SideEffects [None]

  SeeAlso []

******************************************************************************/
void
TruesimWarmUpPatternSimulate(
  Ntk_Network_t *network,
  array_t *inputArray,
  char *vector)
{
  graph_t *partition;
  st_table *nodeToSimTable;
  array_t *depthArray;
  Ntk_Node_t *node;
  bdd_manager *ddManager = Ntk_NetworkReadMddManager(network);
  long maxDepth;
  int j,k;
  
  nodeToSimTable = TruesimNetworkReadSimTable(network);

  depthArray = TruesimNetworkReadDepthArray(network);
  maxDepth = array_n(depthArray);
  
  partition = Part_NetworkReadPartition(network);
  /* Set the input nodes to the input pattern vector. Do not update
     the switching field of the sim structure if it is the first
     vector */
  SetInputValues(inputArray,vector,nodeToSimTable);
  for (j = 1; j < maxDepth; j++) {
    array_t *nodeList;
    
    nodeList = array_fetch(array_t *,depthArray,j);
    arrayForEachItem(Ntk_Node_t *,nodeList,k,node) {
      TrueSim_t *sim;
      
      st_lookup(nodeToSimTable,(char *)node,&sim);
      sim->value =  TruesimEvaluateNode(node,partition,ddManager,
					nodeToSimTable);
    }
  }

  /* Print network status */
  if (truesimVerbose) {
    TruesimPrintNameHeader(network);
    TruesimPrintNetworkNodeLogicState(network);
  }

  return ;

} /* End of TruesimWarmUpPatternSimulate */

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis [Initialize the vector pattern for PI.]

  Description [Initialize the vector pattern for PI.]

  SideEffects [None]

  SeeAlso []

******************************************************************************/
static void
SetInputValues(
  array_t *inputArray,
  char *vector,
  st_table *nodeToSimTable)
{
  Ntk_Node_t *node;
  TrueSim_t *sim;
  int i;
  char prev;

  arrayForEachItem(Ntk_Node_t *,inputArray,i,node) {
    st_lookup(nodeToSimTable,(char *)node,&sim);
    prev = sim->value;
    if (prev != vector[i])
      (sim->switching)++;
    sim->value = vector[i];
    if (vector[i] == '1')
      (sim->prob)++;
  }
} /* End of SetInputValues */

