/**CFile***********************************************************************

  FileName    [hrcCmd.c]

  PackageName [hrc]

  Synopsis    [Commands for walking around HSIS/VIS hierarchies.]

  Description []

  SeeAlso     []

  Author      [Yuji Kukimoto]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "hrcInt.h"

static char rcsid[] UNUSED = "$Id: hrcCmd.c,v 1.6 2002/09/08 21:54:25 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int CommandCd(Hrc_Manager_t **hmgr, int argc, char **argv);
static int CommandLs(Hrc_Manager_t **hmgr, int argc, char **argv);
static int CommandPwd(Hrc_Manager_t **hmgr, int argc, char **argv);
static void NodeListChildren(Hrc_Node_t *hnode, boolean recursiveFlag, int depth);
static void PrintSpace(int depth);
static int CommandPrintModels(Hrc_Manager_t **hmgr, int argc, char **argv);
static int CommandPrintHierarchyStats(Hrc_Manager_t **hmgr, int argc, char **argv);
static void PrintNodeStats(Hrc_Node_t *node, boolean isHnode);
static void PrintNodeStatsRecursively(Hrc_Node_t *node);
static int CommandPrintIo(Hrc_Manager_t **hmgr, int argc, char **argv);
static int CommandPrintLatches(Hrc_Manager_t **hmgr, int argc, char **argv);
static int _HrcStringCmp(const void * s1, const void * s2);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Initializes the hrc package.]

  SideEffects []

  SeeAlso     [Hrc_End]

******************************************************************************/
void
Hrc_Init(void)
{
  Cmd_CommandAdd("cd", CommandCd, 0);
  Cmd_CommandAdd("ls", CommandLs, 0);
  Cmd_CommandAdd("pwd", CommandPwd, 0);
  Cmd_CommandAdd("print_models", CommandPrintModels, 0);
  Cmd_CommandAdd("print_hierarchy_stats", CommandPrintHierarchyStats, 0);
  Cmd_CommandAdd("print_io", CommandPrintIo, 0);
  Cmd_CommandAdd("print_latches", CommandPrintLatches, 0);
}

/**Function********************************************************************

  Synopsis    [Ends the hrc package.]

  SideEffects []

  SeeAlso     [Hrc_Init]

******************************************************************************/
void
Hrc_End(void)
{
}
/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [The top-level routine for cd.]

  SideEffects []
  
  CommandName [cd]

  CommandSynopsis [change the current node]

  CommandArguments [ \[-h\] &lt;child_instance_name&gt; or &lt;..&gt;]

  CommandDescription [Changes the current node in the existing
  hierarchy. The argument should be either the name of a child node
  or .. meaning the parent node. Note that cd ../foo is not allowed.<p>
  Command options:<p>

  <dl>

  <dt> -h
  <dd> Print the command usage.

  </dl>
  ]

  SeeAlso     []

******************************************************************************/
static int
CommandCd(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  Hrc_Node_t *currentNode, *parent, *child;
  int c;

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      default:
        goto usage;
    }
  }

  if (argc != 2){
    goto usage;
  }
  
  currentNode = Hrc_ManagerReadCurrentNode(*hmgr);
  if (currentNode == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stdout,"No file has been read in.\n");
    return 1;
  }
  /* go to the parent node */
  if (strcmp(*(++argv),"..") == 0){
    if ((parent = Hrc_NodeReadParentNode(currentNode)) == NIL(Hrc_Node_t)){
      (void)fprintf(vis_stderr,"You are at the root node. Can't cd to a parent.\n");
      return 1;
    }
    Hrc_ManagerSetCurrentNode(*hmgr,parent);
    return 0;
  }

  if ((child = Hrc_NodeFindChildByName(currentNode,*argv)) == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stderr,"No child node whose name is %s exists in the current node.\n",*argv);
    return 1;
  }
  Hrc_ManagerSetCurrentNode(*hmgr,child);
  return 0;

usage:
  (void)fprintf(vis_stderr,"usage: cd [-h] child_name or cd ..\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");  
  return 1;
}

/**Function********************************************************************

  Synopsis    [The top-level routine for ls.]

  SideEffects []

  CommandName [ls]

  CommandSynopsis [list all the child nodes at the current node]

  CommandArguments [\[-h\] \[-R\]]

  CommandDescription [Lists all the child nodes at the current node.<p>
  Command options:<p>
  
  <dl>

  <dt> -h
  <dd> Print the command usage.<p>
  
  <dt> -R
  <dd> Call ls recursively all the way down to leaves.
  </dl>]

  SeeAlso     []

******************************************************************************/
static int
CommandLs(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  int recursiveFlag = 0;
  int c;
  Hrc_Node_t *currentNode;

  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "hR")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      case 'R':
        recursiveFlag = 1;
        break;
      default:
        goto usage;
    }
  }
  /* if there are some arguments for ls */
  if (argc != util_optind){
    goto usage;
  }

  currentNode = Hrc_ManagerReadCurrentNode(*hmgr);
  if (currentNode == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stdout,"No file has been read in.\n");
    return 1;
  }
  NodeListChildren(currentNode, recursiveFlag, 0);
  return 0;

usage:
  (void)fprintf(vis_stderr,"usage: ls [-h] [-R]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");
  (void) fprintf(vis_stderr, "   -R \t\tprint recursive tree\n");  
  
  return 1;
}

/**Function********************************************************************

  Synopsis    [The top-level routine for pwd.]

  SideEffects []

  CommandName [pwd]
  
  CommandSynopsis [print out the full path of the current node 
  from the root node]
  
  CommandArguments [ \[-h\] ]
  
  CommandDescription [Prints out the full path of the current node from
  the root node.<p>
  Command options:<p>

  <dl>

  <dt> -h
  <dd> Print the command usage.

  </dl>
  ]

  SeeAlso     []

******************************************************************************/
static int
CommandPwd(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  Hrc_Node_t *currentNode;
  Hrc_Node_t *rootNode;
  char *separator, *hierarchicalName;
  int c;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      default:
        goto usage;
    }
  }
  if (argc != 1){
    goto usage;
  }
  
  currentNode = Hrc_ManagerReadCurrentNode(*hmgr);
  if (currentNode == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stdout,"No file has been read in.\n");
    return 1;
  }
  rootNode = Hrc_ManagerReadRootNode(*hmgr);
  separator = (char *) ((rootNode == currentNode) ? "" : ".");
  hierarchicalName = Hrc_NodeFindHierarchicalName(currentNode, FALSE);
  (void)fprintf(vis_stdout,"%s%s%s\n", Hrc_NodeReadInstanceName(rootNode),
                separator, hierarchicalName);
  FREE(hierarchicalName);
  return 0;

usage:
  (void)fprintf(vis_stderr,"usage: pwd [-h]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");  
  return 1;
}

/**Function********************************************************************

  Synopsis    [The core routine for the command ls.]

  Description [The core routine for the command ls. If the second argument
  is set to 1, it calls the ls routine recursively all the way down to leaves.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
static void
NodeListChildren(
  Hrc_Node_t *hnode,
  boolean recursiveFlag,
  int depth)
{
  st_generator *gen;
  char *childName;
  Hrc_Node_t *child;
  int newDepth, i;
  array_t *tmpArray;

  newDepth = depth + 1;
  tmpArray = array_alloc(char *,0);
  Hrc_NodeForEachChild(hnode,gen,childName,child){
    array_insert_last(char *,tmpArray,childName);
  }
  array_sort(tmpArray,_HrcStringCmp);
  for (i=0; i < array_n(tmpArray); i++){
    PrintSpace(depth);
    childName = array_fetch(char *,tmpArray,i);
    (void)fprintf(vis_stdout,"%s\n",childName);
    if (recursiveFlag == 1){
      NodeListChildren(Hrc_NodeFindChildByName(hnode,childName),recursiveFlag,newDepth);
    } 
  } 
  array_free(tmpArray);
}


/**Function********************************************************************

  Synopsis    [Prints out spaces.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
static void
PrintSpace(int depth)
{
  int i;
  for (i=0; i < depth; i++){
    (void)fprintf(vis_stdout," ");
  }
}

/**Function********************************************************************

  Synopsis    [This function the statistics of a model.]

  SideEffects []

  CommandName [print_models]
  
  CommandSynopsis [list all the models and their statistics]
  
  CommandArguments [ \[-h\] ]
  
  CommandDescription [Lists all the models and prints information about
  the number of variables, tables, latches etc. in them.<p>
  Command options:<p>

  <dl>

  <dt> -h
  <dd> Print the command usage.

  </dl>
  ]
  
  SeeAlso     []

******************************************************************************/
static int 
CommandPrintModels(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  st_generator *gen;
  char *name;
  Hrc_Model_t *model;
  Hrc_Node_t *node;
  int c;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      default:
        goto usage;
    }
  }
  if(argc != 1) {
    goto usage;
  }

  node = Hrc_ManagerReadCurrentNode(*hmgr);
  if (node == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stderr,"No file has been read in.\n");
    return 1;
  }

  Hrc_ManagerForEachModel(*hmgr, gen, name, model) {
    PrintNodeStats(Hrc_ModelReadMasterNode(model), FALSE);
    (void) fprintf(vis_stdout, "subckts = %d\n",
                   st_count(Hrc_ModelReadSubcktTable(model)));
  }
  return 0;
  
usage:
  (void) fprintf(vis_stderr, "usage: print_models [-h]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");  
  return 1;        
}

/**Function********************************************************************

  Synopsis    [This function prints the statistics of the hierarchy.]
   
  SideEffects []

  CommandName [print_hierarchy_stats]
  
  CommandSynopsis [print the statistics of the current node]
  
  CommandArguments [ \[-h\] \[-R\] ]
  
  CommandDescription [Prints the statistics of the current node. Note that
  it is not necessary to invoke flatten_hierarchy first to create a network.<p>
  Command options:<p>
  <dl>

  <dt> -h
  <dd> Print the command usage <p>
  
  <dt> -R
  <dd> Print statistics for nodes recursively all the way down to leaves.

  </dl>
  ]

  SeeAlso     []

******************************************************************************/
static int
CommandPrintHierarchyStats(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  int c;
  int recursiveFlag = 0;
  Hrc_Node_t *node;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "hR")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      case 'R':
        recursiveFlag = 1;
        break;
      default:
        goto usage;
    }
  }
  /* if there are some arguments left */
  if (argc != util_optind){
    goto usage;
  }

  node = Hrc_ManagerReadCurrentNode(*hmgr);
  if (node == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stderr,"No file has been read in.\n");
    return 1;
  }
  
  if(recursiveFlag) {
    PrintNodeStatsRecursively(node);
  }
  else {
    PrintNodeStats(node, TRUE);
  }
  return 0;
usage:
  (void) fprintf(vis_stderr,"usage: print_hierarchy_stats [-h][-R]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");
  (void) fprintf(vis_stderr, "   -R \t\tprint recursive tree\n");  
  return 1;  
}

/**Function********************************************************************

  Synopsis    [This function prints the statistics of a node.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
static void
PrintNodeStats(
  Hrc_Node_t *node,
  boolean isHnode)
{
  (void) fprintf(vis_stdout, "Model name = %s", Hrc_NodeReadModelName(node));
  if(isHnode) {
    (void) fprintf(vis_stdout, ", Instance name = %s\n",
                   Hrc_NodeReadInstanceName(node));
  }
  else {
    (void) fprintf(vis_stdout, "\n");
  }
  (void) fprintf(vis_stdout, "inputs = %d,", Hrc_NodeReadNumFormalInputs(node));
  (void) fprintf(vis_stdout, " outputs = %d,", Hrc_NodeReadNumFormalOutputs(node));
  (void) fprintf(vis_stdout, " variables = %d,", Hrc_NodeReadNumVariables(node));
  (void) fprintf(vis_stdout, " tables = %d,", Hrc_NodeReadNumTables(node));
  (void) fprintf(vis_stdout, " latches = %d", Hrc_NodeReadNumLatches(node));
  if(isHnode) {
    (void) fprintf(vis_stdout, ", children = %d\n", Hrc_NodeReadNumChildren(node));
  }
  else {
    (void) fprintf(vis_stdout, "\n");
  }
}

/**Function********************************************************************

  Synopsis    [This function prints the statistics of each node in the
               subtree rooted at node. ]

  SideEffects []

  SeeAlso     []

******************************************************************************/
static void
PrintNodeStatsRecursively(
  Hrc_Node_t *node)
{
  Hrc_Node_t *childNode;
  st_generator *gen;
  char *name;
  
  PrintNodeStats(node, TRUE);
  Hrc_NodeForEachChild(node, gen, name, childNode) {
    PrintNodeStatsRecursively(childNode);
  }
}

/**Function********************************************************************

  Synopsis    [This function prints the names of inputs/outputs in
  the current node.]
   
  SideEffects []

  CommandName [print_io]
  
  CommandSynopsis [print the names of inputs/outputs in the current node]
  
  CommandArguments [ \[-h\] ]
  
  CommandDescription [Prints the names of inputs/outputs in the current node.<p>
  Command options:<p>
  <dl>
  <dt> -h
  <dd> Print the command usage.
  </dl>
  ]

  SeeAlso     []

******************************************************************************/
static int
CommandPrintIo(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  int c, i;
  Hrc_Node_t *node;
  Var_Variable_t *var;
  array_t *tmpArray;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      default:
        goto usage;
    }
  }
  /* if there are some arguments left */
  if (argc != util_optind){
    goto usage;
  }
  node = Hrc_ManagerReadCurrentNode(*hmgr);
  if (node == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stderr,"No file has been read in.\n");
    return 1;
  }
  
  if (Hrc_NodeReadNumFormalInputs(node) == 0){
    fprintf(vis_stdout,"No inputs\n");
  }
  else {
    tmpArray = array_alloc(char *,0);
    Hrc_NodeForEachFormalInput(node,i,var){
      array_insert_last(char *,tmpArray,Var_VariableReadName(var));
    }
    array_sort(tmpArray,_HrcStringCmp);
    fprintf(vis_stdout,"inputs:");
    for (i=0; i < array_n(tmpArray); i++){
      fprintf(vis_stdout," %s",array_fetch(char *,tmpArray,i));
    }
    fprintf(vis_stdout,"\n");
    array_free(tmpArray);
  }

  if (Hrc_NodeReadNumFormalOutputs(node) == 0){
    fprintf(vis_stdout,"No outputs\n");
  }
  else {
    tmpArray = array_alloc(char *,0);
    Hrc_NodeForEachFormalOutput(node,i,var){
      array_insert_last(char *,tmpArray,Var_VariableReadName(var));
    }
    array_sort(tmpArray,_HrcStringCmp);
    fprintf(vis_stdout,"outputs:");
    for (i=0; i < array_n(tmpArray); i++){
      fprintf(vis_stdout," %s",array_fetch(char *,tmpArray,i));
    }
    fprintf(vis_stdout,"\n");
    array_free(tmpArray);
  }
  return 0;

usage:
  (void) fprintf(vis_stderr,"usage: print_io [-h]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");
  return 1;  
}

/**Function********************************************************************

  Synopsis    [This function prints the names of latches in
  the current node.]
   
  SideEffects []

  CommandName [print_latches]
  
  CommandSynopsis [print the names of latches in the current node]
  
  CommandArguments [ \[-h\] ]
  
  CommandDescription [Prints the names of latches in the current node.<p>
  Command options:<p>
  <dl>
  <dt> -h
  <dd> Print the command usage.
  </dl>
  ]

  SeeAlso     []

******************************************************************************/
static int
CommandPrintLatches(
  Hrc_Manager_t **hmgr,
  int argc,
  char **argv)
{
  int c, i;
  Hrc_Node_t *node;
  st_generator *gen;
  char *latchName;
  Hrc_Latch_t *latch;
  array_t *tmpArray;
  
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "h")) != EOF) {
    switch(c){
      case 'h':
        goto usage;
      default:
        goto usage;
    }
  }
  /* if there are some arguments left */
  if (argc != util_optind){
    goto usage;
  }
  node = Hrc_ManagerReadCurrentNode(*hmgr);
  if (node == NIL(Hrc_Node_t)){
    (void)fprintf(vis_stderr,"No file has been read in.\n");
    return 1;
  }
  
  if (Hrc_NodeReadNumLatches(node) == 0){
    fprintf(vis_stdout,"No latches\n");
  }
  else {
    tmpArray = array_alloc(char *,0);
    Hrc_NodeForEachLatch(node,gen,latchName,latch){
      array_insert_last(char *,tmpArray,latchName);
    }
    array_sort(tmpArray,_HrcStringCmp);
    for (i=0; i < array_n(tmpArray); i++){
      fprintf(vis_stdout,"%s ",array_fetch(char *,tmpArray,i));
    }
    fprintf(vis_stdout,"\n");
    array_free(tmpArray);
  }
  return 0;

usage:
  (void) fprintf(vis_stderr,"usage: print_latches [-h]\n");
  (void) fprintf(vis_stderr, "   -h \t\tprint the command usage\n");
  return 1;  
}

/**Function********************************************************************

  Synopsis    [This function is used for string comparison in array_sort.]

  Description [This function is used for string comparison in array_sort.]
   
  SideEffects []

  SeeAlso     []

******************************************************************************/
static int
_HrcStringCmp(
  const void * s1,
  const void * s2)
{
  return(strcmp(*(char **)s1, *(char **)s2));
}

