/**CFile***********************************************************************

  FileName    [partBoundary.c]

  PackageName [part]

  Synopsis [Implements the partition of the network with respect to the 
  nodes that comprise the submodules boundaries.]

  Description [The network is composed of an arbitrary set of nodes, each of
  them implementing some function. This partitioning method will produce a
  graph representing the network in which the nodes that constitute submodule 
  boundaries will be preserved in the graph structure.]

  SeeAlso     [partInOut.c partTotal.c]

  Author      [Sunil P Khatri]

  Copyright   [This file was created at the University of California at
  Berkeley.  The University of California at Berkeley makes no warranty
  about the suitability of this software for any purpose.  It is
  presented on an AS IS basis.]

******************************************************************************/

#include "partInt.h"

static char rcsid[] UNUSED = "$Id: partBoundary.c,v 1.5 2002/09/10 05:06:25 fabio Exp $";

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis [Finds names of all partition nodes the boundary partitioning method.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void
partCreateBoundaryNames(
  Hrc_Node_t    *hnode,
  st_table      *tableOfFormalNames)
{
  st_generator *stGen;
  char *childName, *nodeName, *inputVarName, *outputVarName, *formalName;
  Hrc_Node_t *childNode;
  int i;
  Var_Variable_t *inputVar, *outputVar;

  
  Hrc_NodeForEachChild(hnode, stGen, childName, childNode){
    /* add childrens inputs' formal names to tableOfFormalNames */
    Hrc_NodeForEachFormalInput(childNode, i, inputVar){
      inputVarName = Var_VariableReadName(inputVar);
      nodeName = Hrc_NodeFindHierarchicalName(childNode, 1);
      formalName = ALLOC(char, strlen(nodeName) + strlen(inputVarName) +2);
      sprintf(formalName, "%s.%s", nodeName, inputVarName);      
      FREE(nodeName);
      st_insert(tableOfFormalNames, (char *) formalName, (char *) (long) ( -1));
    }
    /* add childrens outputs' formal names to tableOfFormalNames */
    Hrc_NodeForEachFormalOutput(childNode, i, outputVar){
      outputVarName = Var_VariableReadName(outputVar);
      nodeName = Hrc_NodeFindHierarchicalName(childNode, 1);
      formalName = ALLOC(char, strlen(nodeName) + strlen(outputVarName) +2);
      sprintf(formalName, "%s.%s", nodeName, outputVarName);      
      FREE(nodeName);
      st_insert(tableOfFormalNames, (char *) formalName, (char *) (long) ( -1));
    }
    /* recurse */
    partCreateBoundaryNames(childNode, tableOfFormalNames);
  }
}


/**Function********************************************************************

  Synopsis [Implements the partition with respect to the submodule boundary
  nodes.]

  SideEffects []

  SeeAlso     [PartPartitionTotal PartPartitionInputsOutputs]

******************************************************************************/
void
PartPartitionBoundary(
  Ntk_Network_t *network,
  Hrc_Node_t    *hnode,
  graph_t       *partition,
  lsList        rootList,
  lsList        leaveList,
  mdd_t         *careSet,
  int           inTermsOfCombInputs)
{
  int            i;                /* Index for loops */
  st_generator   *stGen;           /* To iterate over the MddIds of the support */
  char           *formalName;
  char           *actualName;
  st_table       *tableOfActualNames;
  st_table       *tableOfFormalNames;
  lsList         nodeList;


  assert(rootList == (lsList)0);
  assert(leaveList == (lsList)0);

  /* create table of all formal names */
  tableOfFormalNames = st_init_table(strcmp, st_strhash);
  partCreateBoundaryNames(hnode, tableOfFormalNames);

  nodeList = lsCreate();
  tableOfActualNames = st_init_table(strcmp, st_strhash);
  st_foreach_item_int(tableOfFormalNames, stGen, &formalName, &i){
    actualName = Ntk_NetworkReadActualNameFromFormalName(network, formalName);
    if(!st_is_member(tableOfActualNames, actualName)){
      lsNewEnd(nodeList, (lsGeneric) actualName, NIL(lsHandle));
      st_insert(tableOfActualNames, actualName, (char *) (long) ( -1));
    }
  }
  
  
  PartPartitionPartial(network, partition, rootList, leaveList, careSet, nodeList,
		       inTermsOfCombInputs);
  lsDestroy(nodeList, (void (*)(lsGeneric))0);  
  st_free_table(tableOfActualNames);
  st_foreach_item_int(tableOfFormalNames, stGen, &formalName, &i){
    FREE(formalName);
  }
  st_free_table(tableOfFormalNames);
} /* End of PartPartitionPartial */

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
